# @package      hubzero-submit-distributor
# @file         AppsAccessInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import isGroupMember

class AppsAccessInfo:
   def __init__(self,
                infoDirectory,
                appsAccessFile):
      self.logger     = logging.getLogger(__name__)
      self.appsAccess = {}
      self.whitelist  = {}
      self.blacklist  = {}

      groupPattern               = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern            = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern             = re.compile('\s*#.*')
      environmentVariablePattern = re.compile('\$\{(.*?)\}')
      groupName                  = ""

      appsAccessPath = os.path.join(infoDirectory,appsAccessFile)
      if os.path.exists(appsAccessPath):
         fpInfo = open(appsAccessPath,'r')
         if fpInfo:
            eof = False
            while not eof:
               record = fpInfo.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   groupPattern.match(record):
                     groupName = groupPattern.match(record).group(2)
                     if not groupName in self.appsAccess:
                        self.appsAccess[groupName] = []
                     groupIndex = len(self.appsAccess[groupName])
                     self.appsAccess[groupName].append({'whitelist':[],
                                                        'blacklist':[],
                                                        'priority':0,
                                                        'classification':'',
                                                        'state':'enabled'
                                                       })
                  elif keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.appsAccess[groupName][groupIndex]:
                        if   isinstance(self.appsAccess[groupName][groupIndex][key],list):
                           self.appsAccess[groupName][groupIndex][key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.appsAccess[groupName][groupIndex][key],bool):
                           self.appsAccess[groupName][groupIndex][key] = bool(value.lower() == 'true')
                        elif isinstance(self.appsAccess[groupName][groupIndex][key],float):
                           self.appsAccess[groupName][groupIndex][key] = float(value)
                        elif isinstance(self.appsAccess[groupName][groupIndex][key],int):
                           self.appsAccess[groupName][groupIndex][key] = int(value)
                        else:
                           self.appsAccess[groupName][groupIndex][key] = value
                     else:
                        self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s for group %s" % \
                                                                  (key,value,groupName)))
               else:
                  eof = True
            fpInfo.close()

         markedForDeletion = []
         for groupName in self.appsAccess:
            if not isGroupMember(groupName):
               markedForDeletion.append(groupName)
         for groupName in markedForDeletion:
            del self.appsAccess[groupName]
         del markedForDeletion

         for groupName in self.appsAccess:
            markedForDeletion = []
            for groupIndex in xrange(len(self.appsAccess[groupName])):
               state          = self.appsAccess[groupName][groupIndex]['state']
               classification = self.appsAccess[groupName][groupIndex]['classification']
               if state == 'disabled' or classification == '':
                  markedForDeletion.append(groupIndex)
            markedForDeletion.reverse()
            for groupIndex in markedForDeletion:
               del self.appsAccess[groupName][groupIndex]
            del markedForDeletion

         for groupName in self.appsAccess:
            for groupIndex in xrange(len(self.appsAccess[groupName])):
               for element in self.appsAccess[groupName][groupIndex]['whitelist']:
                  element = os.path.expandvars(os.path.expanduser(element))
                  if not re.search(environmentVariablePattern,element):
                     priority       = self.appsAccess[groupName][groupIndex]['priority']
                     classification = self.appsAccess[groupName][groupIndex]['classification']
                     if not priority in self.whitelist:
                        self.whitelist[priority] = {}
                     if not element in self.whitelist[priority]:
                        self.whitelist[priority][element] = classification

               for element in self.appsAccess[groupName][groupIndex]['blacklist']:
                  element = os.path.expandvars(os.path.expanduser(element))
                  if not re.search(environmentVariablePattern,element):
                     priority       = self.appsAccess[groupName][groupIndex]['priority']
                     classification = self.appsAccess[groupName][groupIndex]['classification']
                     if not priority in self.blacklist:
                        self.blacklist[priority] = {}
                     if not element in self.blacklist[priority]:
                        self.blacklist[priority][element] = classification
      else:
         self.logger.log(logging.ERROR,getLogMessage("AppsAccess configuration file %s is missing" % (appsAccessPath)))


   def isSubmissionAllowed(self,
                           executablePath):
      submissionAllowed = False
      classification = None
      expandedPath = os.path.expandvars(os.path.expanduser(executablePath))
      canonicalPath = os.path.realpath(expandedPath)
      if os.path.isfile(canonicalPath):
         if os.access(canonicalPath,os.X_OK) and os.access(canonicalPath,os.R_OK):
            submissionAllowed = True

      if submissionAllowed:
         blacklistPriority = -1
         priorities = self.blacklist.keys()
         priorities.sort(reverse=True)
         for priority in priorities:
            for element in self.blacklist[priority]:
               if re.match(element,expandedPath) or re.match(element,canonicalPath):
                  blacklistPriority       = priority
                  blacklistClassification = self.blacklist[priority][element]
                  break
            if blacklistPriority > 0:
               break

         whitelistPriority = -1
         priorities = self.whitelist.keys()
         priorities.sort(reverse=True)
         for priority in priorities:
            for element in self.whitelist[priority]:
               if re.match(element,expandedPath) or re.match(element,canonicalPath):
                  whitelistPriority = priority
                  whitelistClassification = self.whitelist[priority][element]
                  break
            if whitelistPriority > 0:
               break

         if blacklistPriority >= whitelistPriority:
            submissionAllowed = False
         else:
            classification = whitelistClassification

      return(submissionAllowed,classification)


