# @package      hubzero-submit-distributor
# @file         RemoteRunjobAppScript.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os.path
import re
import logging

class RemoteRunjobAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                localJobId,
                instanceId,
                isParametric,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                environment,
                isMultiCoreRequest,
                managerInfo,
                timeHistoryLogs):
      self.logger             = logging.getLogger(__name__)
      self.hubUserName        = hubUserName
      self.hubUserId          = hubUserId
      self.runName            = runName
      self.localJobId         = localJobId
      self.instanceId         = instanceId
      self.isParametric       = isParametric
      self.transferExecutable = transferExecutable
      if transferExecutable:
         self.executable      = os.path.join(".",os.path.basename(executable))
      else:
         self.executable      = executable
      self.stdinput           = stdinput
      self.arguments          = arguments
      self.environment        = environment
      self.isMultiCoreRequest = isMultiCoreRequest
      self.computationMode    = managerInfo['computationMode']
      self.moduleInitialize   = managerInfo['moduleInitialize']
      self.modulesLoad        = managerInfo['modulesLoad']
      self.modulesUnload      = managerInfo['modulesUnload']
      self.timestampStart     = timeHistoryLogs['timestampStart']
      self.timestampFinish    = timeHistoryLogs['timestampFinish']
      self.timeResults        = timeHistoryLogs['timeResults']


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# RemoteRunjobAppScript:makeSerialTemplate
#
MODULEINITIALIZE
MODULESUNLOAD
MODULESLOAD

CURDIR=`pwd`

SCRIPT=`basename $0`
SITE_SCRATCH_DIR=$3
TARBALL=${SITE_SCRATCH_DIR}/$1
WORKINGDIR=${SITE_SCRATCH_DIR}/$2

if [ ! -d ${SITE_SCRATCH_DIR} ] ; then
   echo "Jobs directory ${SITE_SCRATCH_DIR} is missing"
   exit 13
fi

if [ "${USER}" = "" ] ; then
   if [ "${LOGNAME}" != "" ] ; then
      export USER=${LOGNAME}
   else
      export USER=HUBzero
   fi
fi

APP_PROGRAM=EXECUTABLE
if [ ! -x ${APP_PROGRAM} ] ; then
   if [ "${SITE_APPLICATION_ROOT}" != "" ] ; then
      if [ -x ${SITE_APPLICATION_ROOT}/${APP_PROGRAM} ] ; then
         APP_PROGRAM=${SITE_APPLICATION_ROOT}/${APP_PROGRAM}
      fi
   fi
fi
STD_INPUT=STDIN

mkdir -p ${WORKINGDIR}
chmod 700 ${WORKINGDIR}
tar xzm -C ${WORKINGDIR} -f ${SITE_SCRATCH_DIR}/JOBID_INSTANCEID_input.tar.gz
#
cd ${WORKINGDIR}
sleep 2
touch .__fileTimeMarker
sleep 1

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START
#
# create output file now - if job times out the
# file is available for transfer by condor
#
tar zcf ${TARBALL} ./TS_START
#
export PATH=${PATH}:${WORKINGDIR}

${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS ${APP_PROGRAM} ARGUMENTS < ${STD_INPUT}

date +"%s" > TS_FINISH
#
# tar: Cannot update compressed archives
# so recreate output file for transfer by condor
#
# Allow for file/directory names with embedded spaces.
find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf ${TARBALL} --no-recursion

cd ${CURDIR}
rm -rf ${WORKINGDIR}
"""


   def __buildAppScript(self):
      # setup regex's for the template
      re_runName          = re.compile("RUNNAME")
      re_jobId            = re.compile("JOBID")
      re_instanceId       = re.compile("INSTANCEID")
      re_exe              = re.compile("EXECUTABLE")
      re_stdinput         = re.compile("STDIN")
      re_arguments        = re.compile("ARGUMENTS")
      re_tsStart          = re.compile("TS_START")
      re_tsFinish         = re.compile("TS_FINISH")
      re_timeResults      = re.compile("TIME_RESULTS")
      re_moduleInitialize = re.compile("MODULEINITIALIZE")
      re_modulesUnload    = re.compile("MODULESUNLOAD")
      re_modulesLoad      = re.compile("MODULESLOAD")
      re_hubUserName      = re.compile("HUBUSERNAME")
      re_hubUserId        = re.compile("HUBUSERID")

      template = self.__makeSerialTemplate()

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      template = re_stdinput.sub(self.stdinput,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      moduleInitialize = ""
      if self.moduleInitialize != "":
         moduleInitialize = "type module > /dev/null 2>&1\n" + \
                            "if [ $? != 0 ]; then\n" + \
                            "   . " + self.moduleInitialize + "\n" + \
                            "fi\n"
      template = re_moduleInitialize.sub(moduleInitialize,template)

      modulesUnload = ""
      if self.modulesUnload != "":
         for module in self.modulesUnload:
            modulesUnload += 'module unload ' + module + '\n'
      template = re_modulesUnload.sub(modulesUnload,template)

      modulesLoad = ""
      if self.modulesLoad != "":
         for module in self.modulesLoad:
            modulesLoad += 'module load ' + module + '\n'
      template = re_modulesLoad.sub(modulesLoad,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __makeMPITemplate(self):
      return """#!/bin/sh
# RemoteRunjobAppScript:makeMPITemplate

#echo $0 "$@"
#printenv | sort

MODULEINITIALIZE
MODULESUNLOAD
MODULESLOAD

SCRIPT=`basename $0`
SITE_SCRATCH_DIR=$3
TARBALL=${SITE_SCRATCH_DIR}/$1
WORKINGDIR=${SITE_SCRATCH_DIR}/$2
shift 3

MPIRANKVARIABLE=""
if   [ "_${MPIRUN_RANK}_" != "__" ] ; then
   MPIRANKVARIABLE=${MPIRUN_RANK}
elif [ "_${GMPI_ID}_" != "__" ] ; then
   MPIRANKVARIABLE=${GMPI_ID}
elif [ "_${RMS_RANK}_" != "__" ] ; then
   MPIRANKVARIABLE=${RMS_RANK}
elif [ "_${MXMPI_ID}_" != "__" ] ; then
   MPIRANKVARIABLE=${MXMPI_ID}
elif [ "_${MSTI_RANK}_" != "__" ] ; then
   MPIRANKVARIABLE=${MSTI_RANK}
elif [ "_${PMI_RANK}_" != "__" ] ; then
   MPIRANKVARIABLE=${PMI_RANK}
elif [ "_${PMI_ID}_" != "__" ] ; then
   MPIRANKVARIABLE=${PMI_ID}
elif [ "_${OMPI_MCA_ns_nds_vpid}_" != "__" ] ; then
   MPIRANKVARIABLE=${OMPI_MCA_ns_nds_vpid}
elif [ "_${OMPI_MCA_orte_ess_vpid}_" != "__" ] ; then
   MPIRANKVARIABLE=${OMPI_MCA_orte_ess_vpid}
else
   args=( "$@" )
   for (( i1=0 ; i1 < ${#args[@]} ; i1++ )) ; do
      if   [ "${args[i1]}" = "-p4pg" ] ; then
         p4rank=0
      elif [ "${args[i1]}" = "-p4wd" ] ; then
         (( i1++ ))
         args[${i1}]=${WORKINGDIR}
      elif [ "${args[i1]}" = "-p4rmrank" ] ; then
         (( i1++ ))
         p4rank=${args[i1]}
      fi
   done
   if [ "${p4rank}" = "0" ] ; then
      set -- ${args[@]}
   fi
   if [ "_${p4rank}_" != "__" ] ; then
      MPIRANKVARIABLE=${p4rank}
   fi
fi

if [ ! -d ${SITE_SCRATCH_DIR} ] ; then
   echo "Jobs directory ${SITE_SCRATCH_DIR} is missing"
   exit 13
fi

if [ "${USER}" = "" ] ; then
   if [ "${LOGNAME}" != "" ] ; then
      export USER=${LOGNAME}
   else
      export USER=HUBzero
   fi
fi

CURDIR=`pwd`

APP_PROGRAM=EXECUTABLE
if [ ! -x ${APP_PROGRAM} ] ; then
   if [ "${SITE_APPLICATION_ROOT}" != "" ] ; then
      if [ -x ${SITE_APPLICATION_ROOT}/${APP_PROGRAM} ] ; then
         APP_PROGRAM=${SITE_APPLICATION_ROOT}/${APP_PROGRAM}
      fi
   fi
fi
STD_INPUT=STDIN

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

if [ "_${MPIRANKVARIABLE}_" = "__" ] ; then
   mkdir -p ${WORKINGDIR}
   chmod 700 ${WORKINGDIR}
   cd ${WORKINGDIR}
   touch .__fileTimeMarker
   sleep 1
   date +"%s" > TS_START
   #
   echo "Unable to determine MPI rank"
   #

   date +"%s" > TS_FINISH

   # Allow for file/directory names with embedded spaces.
   find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf ${TARBALL} --no-recursion

   cd ${CURDIR}
   rm -rf ${WORKINGDIR}
else
   if [ "${MPIRANKVARIABLE}" = "0" ] ; then

      mkdir -p ${WORKINGDIR}
      chmod 700 ${WORKINGDIR}
      tar xzm -C ${WORKINGDIR} -f ${SITE_SCRATCH_DIR}/JOBID_INSTANCEID_input.tar.gz
   #
      cd ${WORKINGDIR}
      sleep 2
      touch .__fileTimeMarker
      sleep 1
      date +"%s" > TS_START
   #
   # create output file now - if job times out the
   # file is available for transfer by condor
   #
      tar zcf ${TARBALL} ./TS_START
   #
   fi

   until [ -f ${TARBALL} ] ; do
      sleep 2
   done

   cd ${WORKINGDIR}
   export PATH=${PATH}:${WORKINGDIR}
   ENVIRONMENT

   rankExtension=`printf "%06d" ${MPIRANKVARIABLE}`
   if   [ "${p4rank}" = "0" ] ; then
      ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} ${APP_PROGRAM} ARGUMENTS "$@" < ${STD_INPUT}
   elif [ "${p4rank}" != "" ] ; then
      ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} ${APP_PROGRAM}           "$@" < ${STD_INPUT}
   else
      ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} ${APP_PROGRAM} ARGUMENTS "$@" < ${STD_INPUT}
   fi

   if [ "${MPIRANKVARIABLE}" = "0" ] ; then
      date +"%s" > TS_FINISH

      touch TIME_RESULTS-0
      cat TIME_RESULTS-[0-9]* >> TIME_RESULTS
      rm -f TIME_RESULTS-[0-9]*
   #
   # tar: Cannot update compressed archives
   # so recreate output file for transfer by condor
   #
   # Allow for file/directory names with embedded spaces.
      find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf ${TARBALL} --no-recursion

      cd ${CURDIR}
      rm -rf ${WORKINGDIR}
   fi
fi
"""


   def __buildAppMPIScript(self):
      # setup regex's for the template
      re_runName          = re.compile("RUNNAME")
      re_jobId            = re.compile("JOBID")
      re_instanceId       = re.compile("INSTANCEID")
      re_exe              = re.compile("EXECUTABLE")
      re_stdinput         = re.compile("STDIN")
      re_arguments        = re.compile("ARGUMENTS")
      re_environment      = re.compile("ENVIRONMENT")
      re_tsStart          = re.compile("TS_START")
      re_tsFinish         = re.compile("TS_FINISH")
      re_timeResults      = re.compile("TIME_RESULTS")
      re_moduleInitialize = re.compile("MODULEINITIALIZE")
      re_modulesUnload    = re.compile("MODULESUNLOAD")
      re_modulesLoad      = re.compile("MODULESLOAD")
      re_hubUserName      = re.compile("HUBUSERNAME")
      re_hubUserId        = re.compile("HUBUSERID")

      template = self.__makeMPITemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      template = re_stdinput.sub(self.stdinput,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_environment.sub(environmentExport,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      moduleInitialize = ""
      if self.moduleInitialize != "":
         moduleInitialize = "type module > /dev/null 2>&1\n" + \
                            "if [ $? != 0 ]; then\n" + \
                            "   . " + self.moduleInitialize + "\n" + \
                            "fi"
      template = re_moduleInitialize.sub(moduleInitialize,template)

      modulesUnload = ""
      if self.modulesUnload != "":
         for module in self.modulesUnload:
            modulesUnload += 'module unload ' + module + '\n'
      template = re_modulesUnload.sub(modulesUnload,template)

      modulesLoad = ""
      if self.modulesLoad != "":
         for module in self.modulesLoad:
            modulesLoad += 'module load ' + module + '\n'
      template = re_modulesLoad.sub(modulesLoad,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript()
         elif self.computationMode == 'parallel':
            appScript = self.__buildAppScript()
         elif self.computationMode == 'matlabmpi':
            appScript = ""
      else:
         appScript = self.__buildAppScript()

      return(appScriptName,appScript)


