# @package      hubzero-submit-distributor
# @file         SitesInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
import random
import copy
import socket

from hubzero.submit.LogMessage      import logJobId as log
from hubzero.submit.GroupMembership import isGroupMember

class SitesInfo:
   def __init__(self,
                infoDirectory,
                sitesFile,
                allowedVenueMechanisms):
      self.sites = {}

      sitePattern     = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      siteName        = ""

      sitesPath = os.path.join(infoDirectory,sitesFile)
      if os.path.exists(sitesPath):
         fpInfo = open(sitesPath,'r')
         if fpInfo:
            eof = False
            while not eof:
               record = fpInfo.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sitePattern.match(record):
                     siteName = sitePattern.match(record).group(2)
                     self.sites[siteName] = {'venues':[],
                                             'gsiHost':socket.gethostname(),
                                             'cloudDesignator':'',
                                             'tunnelDesignator':'',
                                             'siteMonitorDesignator':'',
                                             'venueMechanism':'',
                                             'venueGatekeeper':',,'.split(','),
                                             'remoteUser':'',
                                             'remoteBatchAccount':'',
                                             'remoteBatchSystem':'',
                                             'remoteBatchQueue':'',
                                             'remoteBatchPartition':'',
                                             'remoteBatchPartitionSize':'',
                                             'remoteBatchConstraints':'',
                                             'parallelEnvironment':'',
                                             'remoteBinDirectory':os.path.join('${HOME}','bin'),
                                             'remoteApplicationRootDirectory':'',
                                             'remoteScratchDirectory':'',
                                             'remotePpn':'',
                                             'remoteManager':'',
                                             'remoteHostAttribute':'',
                                             'stageFiles':True,
                                             'passUseEnvironment':False,
                                             'arbitraryExecutableAllowed':True,
                                             'executableClassificationsAllowed':['*'],
                                             'members':[],
                                             'state':'enabled',
                                             'failoverSite':'None',
                                             'checkProbeResult':True,
                                             'restrictedToUsers':[],
                                             'restrictedToGroups':[],
                                             'logUserRemotely':False,
                                             'undeclaredSiteSelectionWeight':0.,
                                             'minimumWallTime':0.,
                                             'maximumWallTime':0.,
                                             'minimumCores':1,
                                             'maximumCores':0,
                                             'pegasusTemplates':[],
                                             'pegasusSites':[]
                                            }
                  elif keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.sites[siteName]:
                        if   isinstance(self.sites[siteName][key],list):
                           self.sites[siteName][key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.sites[siteName][key],bool):
                           self.sites[siteName][key] = bool(value.lower() == 'true')
                        elif isinstance(self.sites[siteName][key],float):
                           self.sites[siteName][key] = float(value)
                        elif isinstance(self.sites[siteName][key],int):
                           self.sites[siteName][key] = int(value)
                        else:
                           self.sites[siteName][key] = value
                     else:
                        log("Undefined key = value pair %s = %s for site %s" % (key,value,siteName))
               else:
                  eof = True
            fpInfo.close()

         for site in self.sites:
            arbitraryExecutableAllowed = self.sites[site]['arbitraryExecutableAllowed']
            executableClassificationsAllowed = self.sites[site]['executableClassificationsAllowed']
            if len(executableClassificationsAllowed) == 1:
               if executableClassificationsAllowed[0] == '*':
                  if not arbitraryExecutableAllowed:
                     self.sites[site]['executableClassificationsAllowed'] = ['staged', 'apps']
            del self.sites[site]['arbitraryExecutableAllowed']

         for site in self.sites:
            venueMechanism = self.sites[site]['venueMechanism']
            if venueMechanism != '':
               if not venueMechanism in allowedVenueMechanisms:
                  self.sites[site]['state'] = 'restrictedByVenueMechanism'

         userName = os.getenv("USER")
         for site in self.sites:
            restrictedToUsers = self.sites[site]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not userName in restrictedToUsers:
                  self.sites[site]['state'] = 'restrictedByUser'

         for site in self.sites:
            if self.sites[site]['state'] == 'enabled':
               restrictedToGroups = self.sites[site]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.sites[site]['state'] = 'restrictedByGroup'

         for site in self.sites:
            members = self.sites[site]['members']
            if len(members) > 0:
               markedForDeletion = []
               for member in members:
                  if member in self.sites:
                     if self.sites[member]['state'] != 'enabled':
                        markedForDeletion.append(member)
               for member in markedForDeletion:
                  self.sites[site]['members'].remove(member)
               del markedForDeletion

         for site in self.sites:
            failoverSite = self.sites[site]['failoverSite']
            if failoverSite != 'None' and failoverSite != 'grid':
               if self.sites[failoverSite]['state'] != 'enabled':
                  self.sites[site]['failoverSite'] = 'None'

         for site in self.sites:
            pegasusTemplates = copy.copy(self.sites[site]['pegasusTemplates'])
            self.sites[site]['pegasusTemplates'] = {}
            self.sites[site]['pegasusTemplates']['rc']    = ""
            self.sites[site]['pegasusTemplates']['sites'] = ""
            self.sites[site]['pegasusTemplates']['tc']    = ""
            for pegasusTemplate in pegasusTemplates:
               pegasusTemplatePath = os.path.join(infoDirectory,pegasusTemplate)
               if os.path.exists(pegasusTemplatePath):
                  if   pegasusTemplate.endswith('.rc'):
                     self.sites[site]['pegasusTemplates']['rc']    = pegasusTemplatePath
                  elif pegasusTemplate.endswith('.xml'):
                     self.sites[site]['pegasusTemplates']['sites'] = pegasusTemplatePath
                  elif pegasusTemplate.endswith('.txt'):
                     self.sites[site]['pegasusTemplates']['tc']    = pegasusTemplatePath
               else:
                  log("Pegasus template file %s does not exist." % (pegasusTemplatePath))
            del pegasusTemplates
      else:
         log("Sites configuration file %s is missing" % (sitesPath))


   def getEnabledSites(self):
      enabledSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            enabledSites.append(site)

      return(enabledSites)


   def purgeDisabledSites(self,
                          siteNames):
      reasonsDenied = {}
      markedForDeletion = []
      for siteName in siteNames:
         try:
            site  = self.sites[siteName]
            state = site['state']
            if state != 'enabled':
               markedForDeletion.append(siteName)
               reasonsDenied[siteName] = state
         except:
            pass
      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      del markedForDeletion

      return(reasonsDenied)


   def __checkProbeResult(self,
                          siteName):
      try:
         site = self.sites[siteName]
         checkProbeResult = site['checkProbeResult']
      except:
         checkProbeResult = True

      return(checkProbeResult)


   def getIgnoreProbeSites(self):
      ignoreProbeSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            if not self.__checkProbeResult(site):
               ignoreProbeSites.append(site)

      return(ignoreProbeSites)


   def purgeOfflineSites(self,
                         siteNames,
                         remoteProbeMonitor):
      goToGrid = False
      if remoteProbeMonitor:
         markedForDeletion = []
         substituteSites   = []
         for siteName in siteNames:
            if self.__checkProbeResult(siteName) and not remoteProbeMonitor.isSiteAvailable(siteName):
               markedForDeletion.append(siteName)
               try:
                  site         = self.sites[siteName]
                  failoverSite = site['failoverSite']
                  if failoverSite != 'None':
                     if failoverSite == 'grid':
                        goToGrid = True
                     else:
                        if remoteProbeMonitor.isSiteAvailable(failoverSite):
                           substituteSites.append(failoverSite)
               except:
                  pass
         for siteName in markedForDeletion:
            siteNames.remove(siteName)
         if len(siteNames) > 0:
            goToGrid = False
         else:
            if len(substituteSites) > 0:
               for siteName in substituteSites:
                  siteNames.append(siteName)
               goToGrid = False

         del substituteSites
         del markedForDeletion

      return(goToGrid)


   def purgeResourceLimitedSites(self,
                                 siteNames,
                                 nCores,
                                 jobWallTime):
      markedForDeletion = []
      for siteName in self.sites:
         site = self.sites[siteName]
         if site['state'] == 'enabled':
            if int(nCores) < site['minimumCores']:
               markedForDeletion.append(siteName)
            if site['maximumCores'] > 0:
               if int(nCores) > site['maximumCores']:
                  markedForDeletion.append(siteName)
            if float(jobWallTime) < site['minimumWallTime']:
               markedForDeletion.append(siteName)
            if site['maximumWallTime'] > 0.:
               if float(jobWallTime) > site['maximumWallTime']:
                  markedForDeletion.append(siteName)

            members = site['members']
            if len(members) > 0:
               membersMarkedForDeletion = []
               for member in members:
                  if member in self.sites:
                     memberSite = self.sites[member]
                     if memberSite['state'] == 'enabled':
                        if int(nCores) < memberSite['minimumCores']:
                           membersMarkedForDeletion.append(member)
                        if memberSite['maximumCores'] > 0:
                           if int(nCores) > memberSite['maximumCores']:
                              membersMarkedForDeletion.append(member)
                        if float(jobWallTime) < memberSite['minimumWallTime']:
                           membersMarkedForDeletion.append(member)
                        if memberSite['maximumWallTime'] > 0.:
                           if float(jobWallTime) > memberSite['maximumWallTime']:
                              membersMarkedForDeletion.append(member)
               for member in membersMarkedForDeletion:
                  site['members'].remove(member)
               del membersMarkedForDeletion
               if len(site['members']) == 0:
                  markedForDeletion.append(siteName)

      for siteName in markedForDeletion:
         site = self.sites[siteName]
         site['state'] = 'resourceLimit'
         if siteName in siteNames:
            siteNames.remove(siteName)

      del markedForDeletion


   def selectUndeclaredSites(self,
                             siteNames,
                             executableClassification,
                             maximumSelectedSites):
      selectedUndeclaredSites = []

      if len(siteNames) > 0:
         siteAdded = True
         while siteAdded and (len(selectedUndeclaredSites) < maximumSelectedSites):
            siteAdded = False
            undeclaredSiteSelectionWeights = {}
            possibleSites = []
            for siteName in siteNames:
               if not siteName in selectedUndeclaredSites:
                  try:
                     site    = self.sites[siteName]
                     members = site['members']
                     if len(members) > 0:
                        addMembers = True
                        for member in members:
                           if member in selectedUndeclaredSites:
                              addMembers = False
                              break
                        if addMembers:
                           undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                           for member in members:
                              if not member in possibleSites:
                                 possibleSites.append(member)
                                 undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                              else:
                                 undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
                     else:
                        if not siteName in possibleSites:
                           undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                           possibleSites.append(siteName)
                           undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                        else:
                           undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
                  except:
                     pass

            if len(possibleSites) > 0:
               markedForDeletion = []
               for possibleSite in possibleSites:
                  try:
                     site = self.sites[possibleSite]
                     if (not '*' in site['executableClassificationsAllowed']) and \
                        (not executableClassification in site['executableClassificationsAllowed'] ):
                        markedForDeletion.append(possibleSite)
                  except:
                     markedForDeletion.append(possibleSite)
               for possibleSite in markedForDeletion:
                  possibleSites.remove(possibleSite)
               del markedForDeletion

               if len(possibleSites) > 0:
                  nWeights = 0
                  undeclaredSelectionWeightSum = 0.
                  markedForDeletion = []
                  for possibleSite in possibleSites:
                     if   undeclaredSiteSelectionWeights[possibleSite] > 0.:
                        nWeights += 1
                        undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]
                     elif undeclaredSiteSelectionWeights[possibleSite] < 0.:
                        markedForDeletion.append(possibleSite)
                  for possibleSite in markedForDeletion:
                     possibleSites.remove(possibleSite)
                  del markedForDeletion

                  if nWeights == 0:
                     possibleIndex = random.randint(0,len(possibleSites)-1)
                     selectedSite = possibleSites[possibleIndex]
                     selectedUndeclaredSites.append(selectedSite)
                     siteAdded = True
                  else:
                     randomWeight = random.random()
                     weightSum = 0.
                     for possibleSite in possibleSites:
                        undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                        if undeclaredSiteSelectionWeight > 0.:
                           weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                           if weightSum >= randomWeight:
                              selectedSite = possibleSite
                              selectedUndeclaredSites.append(selectedSite)
                              siteAdded = True
                              break

            del possibleSites
            del undeclaredSiteSelectionWeights

      return(selectedUndeclaredSites)


   def selectSite(self,
                  siteNames,
                  executableClassification):
      selectedSite = ""
      undeclaredSiteSelectionWeights = {}

      if len(siteNames) > 0:
         possibleSites = []
         for siteName in siteNames:
            try:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                  for member in members:
                     if not member in possibleSites:
                        possibleSites.append(member)
                        undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                     else:
                        undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
               else:
                  if not siteName in possibleSites:
                     undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                     possibleSites.append(siteName)
                     undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                  else:
                     undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
            except:
               pass

         if len(possibleSites) > 0:
            markedForDeletion = []
            for possibleSite in possibleSites:
               try:
                  site = self.sites[possibleSite]
                  if (not '*' in site['executableClassificationsAllowed']) and \
                     (not executableClassification in site['executableClassificationsAllowed'] ):
                     markedForDeletion.append(possibleSite)
               except:
                  markedForDeletion.append(possibleSite)
            for possibleSite in markedForDeletion:
               possibleSites.remove(possibleSite)
            del markedForDeletion

            if len(possibleSites) > 0:
               undeclaredSelectionWeightSum = 0.
               for possibleSite in possibleSites:
                  undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]

               if undeclaredSelectionWeightSum == 0.:
                  possibleIndex = random.randint(0,len(possibleSites)-1)
                  selectedSite = possibleSites[possibleIndex]
               else:
                  randomWeight = random.random()
                  weightSum = 0.
                  for possibleSite in possibleSites:
                     undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                     if undeclaredSiteSelectionWeight > 0.:
                        weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                        if weightSum >= randomWeight:
                           selectedSite = possibleSite
                           break

      del undeclaredSiteSelectionWeights

      return(selectedSite)


   def selectSites(self,
                   siteNames):
      selectedSites = []

      if len(siteNames) > 0:
         siteNameIndex = random.randint(0,len(siteNames)-1)
         siteName = siteNames[siteNameIndex]
         try:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               memberIndex = random.randint(0,len(members)-1)
               selectedSites.append(members[memberIndex])
            else:
               selectedSites.append(siteName)
         except:
            markedForDeletion = []
            for siteName in siteNames:
               try:
                  site = self.sites[siteName]
                  markedForDeletion.append(siteName)
               except:
                  selectedSites.append(siteName)
            for siteName in markedForDeletion:
               siteNames.remove(siteName)
            del markedForDeletion

      return(selectedSites)


   def getSiteKeyValue(self,
                       siteName,
                       key):
      value = ""

      if siteName in self.sites:
         site    = self.sites[siteName]
         members = site['members']
         if len(members) > 0:
            for member in members:
               if member in self.sites:
                  site = self.sites[member]
                  if key in site:
                     value = site[key]
                     break
         else:
            if key in site:
               value = site[key]

      return(value)


   def getSiteVenues(self,
                     siteName):
      venues = []
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'venues' in site:
            venues = site['venues']

      return(venues)


   def getSiteMembers(self,
                      siteName):
      members = []
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'members' in site:
            members = site['members']

      return(members)


   def getSitePegasusSite(self,
                          siteName):
      selectedPegasusSite = ""
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'pegasusSites' in site:
            pegasusSites = site['pegasusSites']
            totalWeightedSum = 0.
            for pegasusSite in pegasusSites:
               try:
                  site,weight = pegasusSite.split(':')
                  totalWeightedSum += float(weight)
               except:
                  pass

            if totalWeightedSum == 0:
               pegasusIndex = random.randint(0,len(pegasusSites)-1)
               selectedPegasusSite = pegasusSites[pegasusIndex]
            else:
               randomWeight = random.random()
               weightedSum = 0.
               for pegasusSite in pegasusSites:
                  try:
                     site,weight = pegasusSite.split(':')
                     if float(weight) > 0.:
                        weightedSum += float(weight)/totalWeightedSum
                        if weightedSum >= randomWeight:
                           selectedPegasusSite = site
                           break
                  except:
                     pass

      return(selectedPegasusSite)


   def getSitesWithKeyValue(self,
                            key,
                            value,
                            siteNames):
      sitesWithKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if   isinstance(site[key],list):
                              if value in site[key]:
                                 sitesWithKeyValue.append(member)
                           elif site[key] == value:
                              sitesWithKeyValue.append(member)
               else:
                  if key in site:
                     if   isinstance(site[key],list):
                        if value in site[key]:
                           sitesWithKeyValue.append(siteName)
                     elif site[key] == value:
                        sitesWithKeyValue.append(siteName)

      return(sitesWithKeyValue)


   def getSitesWithoutKeyValue(self,
                               key,
                               value,
                               siteNames):
      sitesWithoutKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if   isinstance(site[key],list):
                              if not value in site[key]:
                                 sitesWithoutKeyValue.append(member)
                           elif site[key] != value:
                              sitesWithoutKeyValue.append(member)
                        else:
                           sitesWithoutKeyValue.append(siteName)
               else:
                  if key in site:
                     if   isinstance(site[key],list):
                        if not value in site[key]:
                           sitesWithoutKeyValue.append(siteName)
                     elif site[key] != value:
                        sitesWithoutKeyValue.append(siteName)
                  else:
                     sitesWithoutKeyValue.append(siteName)

      return(sitesWithoutKeyValue)


   def siteExists(self,
                  siteName):
      return(siteName in self.sites)


   def getExpandedSiteNames(self,
                            siteNames,
                            remoteProbeMonitor):
      expandedSiteNames = []

      for siteName in siteNames:
         if siteName in self.sites:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               for member in members:
                  if member in self.sites:
                     if remoteProbeMonitor and self.__checkProbeResult(member):
                        if remoteProbeMonitor.isSiteAvailable(member):
                           expandedSiteNames.append(member)
                     else:
                        expandedSiteNames.append(member)
                  else:
                     expandedSiteNames.append(siteName)
            else:
               if remoteProbeMonitor and self.__checkProbeResult(siteName):
                  if remoteProbeMonitor.isSiteAvailable(siteName):
                     expandedSiteNames.append(siteName)
               else:
                  expandedSiteNames.append(siteName)
         else:
            expandedSiteNames.append(siteName)

      random.shuffle(expandedSiteNames)

      return(expandedSiteNames)


   def getDefaultSiteInfo(self):
      siteInfo = {}
      siteInfo['venues']                           = []
      siteInfo['venueIndex']                       = -1
      siteInfo['venue']                            = "Unknown"
      siteInfo['venueMechanism']                   = ''
      siteInfo['cloudDesignator']                  = ""
      siteInfo['tunnelDesignator']                 = ""
      siteInfo['siteMonitorDesignator']            = ""
      siteInfo['venueGatekeeper']                  = ""
      siteInfo['remoteUser']                       = ""
      siteInfo['sharedUserSpace']                  = False
      siteInfo['stageFiles']                       = True
      siteInfo['passUseEnvironment']               = False
      siteInfo['logUserRemotely']                  = False
      siteInfo['remoteBatchAccount']               = ""
      siteInfo['remoteBatchSystem']                = ""
      siteInfo['remoteBatchQueue']                 = ""
      siteInfo['remoteBatchPartition']             = ""
      siteInfo['remoteBatchPartitionSize']         = ""
      siteInfo['remoteBatchConstraints']           = ""
      siteInfo['parallelEnvironment']              = ""
      siteInfo['remoteBinDirectory']               = ""
      siteInfo['remoteApplicationRootDirectory']   = ""
      siteInfo['remoteScratchDirectory']           = ""
      siteInfo['remoteManager']                    = ""
      siteInfo['hostAttributes']                   = ""
      siteInfo['executableClassificationsAllowed'] = ['*']
      siteInfo['remotePpn']                        = ""
      siteInfo['pegasusTemplates']                 = []

      return(siteInfo)


   def getSiteInfo(self,
                   siteName):
      siteInfo = {}

      site = self.sites[siteName]
      siteInfo['venues']                           = site['venues']
      if len(siteInfo['venues']) > 0:
         siteInfo['venueIndex']                    = random.randint(0,len(siteInfo['venues'])-1)
         siteInfo['venue']                         = siteInfo['venues'][siteInfo['venueIndex']]
      else:
         siteInfo['venueIndex']                    = -1
         siteInfo['venue']                         = ""
      siteInfo['gsiHost']                          = site['gsiHost']
 
      siteInfo['venueMechanism']                   = site['venueMechanism']
      siteInfo['cloudDesignator']                  = site['cloudDesignator']
      siteInfo['tunnelDesignator']                 = site['tunnelDesignator']
      siteInfo['siteMonitorDesignator']            = site['siteMonitorDesignator']
      siteInfo['venueGatekeeper']                  = site['venueGatekeeper']
      siteInfo['remoteUser']                       = site['remoteUser']
      if   siteInfo['remoteUser'].startswith('USER:'):
         siteInfo['sharedUserSpace']               = True
      elif siteInfo['remoteUser'].startswith('USER'):
         siteInfo['sharedUserSpace']               = True
      else:
         siteInfo['sharedUserSpace']               = False
      siteInfo['stageFiles']                       = site['stageFiles']
      siteInfo['passUseEnvironment']               = site['passUseEnvironment']
      siteInfo['logUserRemotely']                  = site['logUserRemotely']
      siteInfo['remoteBatchAccount']               = site['remoteBatchAccount']
      siteInfo['remoteBatchSystem']                = site['remoteBatchSystem']
      siteInfo['remoteBatchQueue']                 = site['remoteBatchQueue']
      siteInfo['remoteBatchPartition']             = site['remoteBatchPartition']
      siteInfo['remoteBatchPartitionSize']         = site['remoteBatchPartitionSize']
      siteInfo['remoteBatchConstraints']           = site['remoteBatchConstraints']
      siteInfo['parallelEnvironment']              = site['parallelEnvironment']
      siteInfo['remoteBinDirectory']               = site['remoteBinDirectory']
      siteInfo['remoteApplicationRootDirectory']   = site['remoteApplicationRootDirectory']
      siteInfo['remoteScratchDirectory']           = site['remoteScratchDirectory']
      if 'remoteManager' in site:
         siteInfo['remoteManager']                 = site['remoteManager']
      else:
         siteInfo['remoteManager']                 = ""
      siteInfo['hostAttributes']                   = site['remoteHostAttribute']
      siteInfo['executableClassificationsAllowed'] = site['executableClassificationsAllowed']
      siteInfo['remotePpn']                        = site['remotePpn']
      if 'pegasusTemplates' in site:
         siteInfo['pegasusTemplates']              = site['pegasusTemplates']
      else:
         siteInfo['pegasusTemplates']              = []

      return(siteInfo)


