# @package      hubzero-submit-distributor
# @file         JobKill.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

__version__ = '2.5.1'

import sys
import os
import re
import signal
import subprocess
from errno import EPIPE

from hubzero.submit.CommandParser    import CommandParser
from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

class JobKill:
   def __init__(self,
                jobMonitorHost,
                jobMonitorPort):

      self.version          = __version__
      self.operationMode    = 0
      self.jobMonitorHost   = jobMonitorHost
      self.jobMonitorPort   = jobMonitorPort
      self.remoteJobMonitor = RemoteJobMonitor(jobMonitorHost,jobMonitorPort)

      self.commandParser = None
      self.killIds       = []

      self.hubUserId = os.getuid()

      self.abortAttempted = False
      self.abortSignal    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         pass


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         pass


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      exitCode = 0
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODERUNKILL:
         self.killIds = [int(killId) for killId in self.commandParser.getOption('killIds')]
      else:
         exitCode = 1

      return(exitCode)

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortSignal    = signalNumber


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   @staticmethod
   def killProcess(pid):
      try:
         os.kill(pid,signal.SIGINT)
         for retry in range(0,12):
            time.sleep(10)
            os.kill(pid,0)

         os.kill(pid,signal.SIGTERM)
         for retry in range(0,6):
            time.sleep(10)
            os.kill(pid,0)

         os.kill(pid,signal.SIGKILL)
      except:
         pass


   def killActiveJob(self):
      """Kill jobs"""

      for killId in self.killIds:
         activeJobPid = self.remoteJobMonitor.queryUserActiveJobPid(self.hubUserId,killId)
         if activeJobPid:
            self.killProcess(activeJobPid)


