#
# @package      hubzero-submit-distributor
# @file         VenueMechanismGsiSsh.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import sys
import os
import re
import stat
import time
import datetime
import shutil

from LogMessage             import logID as log
from JobStatistic           import JobStatistic
from VenueMechanismCore     import VenueMechanismCore
from RemoteBatchCONDOR      import RemoteBatchCONDOR
from RemoteBatchLL          import RemoteBatchLL
from RemoteBatchLSF         import RemoteBatchLSF
from RemoteBatchPBS         import RemoteBatchPBS
from RemoteBatchPBS8        import RemoteBatchPBS8
from RemoteBatchSLURM       import RemoteBatchSLURM
from RemoteBatchAppScript   import RemoteBatchAppScript
from RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from RemoteInstantAppScript import RemoteInstantAppScript

class VenueMechanismGsiSsh(VenueMechanismCore):
   def __init__(self,
                binDirectory,
                receiveInputCommand,
                submitBatchJobCommand,
                transmitResultsCommand,
                cleanupJobCommand,
                killBatchJobCommand,
                remoteBatchSystem,
                localJobId,
                trial,
                destination,
                venue,
                venues,
                venueIndex,
                siteMonitorDesignator,
                x509ProxyPath,
                logUserRemotely,
                stageFiles,
                stageInTarFile,
                scratchDirectory,
                transferExecutable,
                executable,
                remoteApplicationRootDirectory,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                computationMode,
                mpiRankVariable,
                preManagerCommands,
                managerCommand,
                postManagerCommands,
                nCpus,
                nNodes,
                ppn,
                hostAttributes,
                remoteBatchQueue,
                wallTime,
                remoteBatchAccount,
                remoteBatchConstraints,
                remoteBatchPartition,
                remoteBatchPartitionSize,
                quotaLimit,
                timestampTransferred,
                timestampStart,
                timestampFinish,
                timeResults):
      VenueMechanismCore.__init__(self)

      self.venueMechanism                 = 'gsissh'
      remoteBinDirectory = binDirectory.replace("$","\$")
      self.receiveInput                   = os.path.join(remoteBinDirectory,receiveInputCommand)
      self.submitBatchJob                 = os.path.join(remoteBinDirectory,submitBatchJobCommand)
      self.transmitResults                = os.path.join(remoteBinDirectory,transmitResultsCommand)
      self.cleanupJob                     = os.path.join(remoteBinDirectory,cleanupJobCommand)
      self.killBatchJob                   = os.path.join(remoteBinDirectory,killBatchJobCommand)
      if stageFiles:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.workingDirectory            = os.path.join(scratchDirectory,str(epoch) + '_' + localJobId)
      else:
         self.workingDirectory            = os.getcwd()
      self.remoteBatchSystem              = remoteBatchSystem
      self.localJobId                     = localJobId
      self.trial                          = trial
      self.destination                    = destination
      self.venue                          = venue
      self.venues                         = venues
      self.venueIndex                     = venueIndex
      self.siteMonitorDesignator          = siteMonitorDesignator
      self.x509ProxyPath                  = x509ProxyPath
      self.logUserRemotely                = logUserRemotely
      self.stageFiles                     = stageFiles
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile              = stageInTarFile
      else:
         self.stageInTarFile              = stageInTarFile + '.gz'
      self.scratchDirectory               = scratchDirectory
      self.transferExecutable             = transferExecutable
      self.executable                     = executable
      self.remoteApplicationRootDirectory = remoteApplicationRootDirectory
      self.stdinput                       = stdinput
      self.arguments                      = arguments
      self.useEnvironment                 = useEnvironment
      self.environment                    = environment
      self.isMultiCoreRequest             = isMultiCoreRequest
      self.computationMode                = computationMode
      self.mpiRankVariable                = mpiRankVariable
      self.preManagerCommands             = preManagerCommands
      self.managerCommand                 = managerCommand
      self.postManagerCommands            = postManagerCommands
      self.nCpus                          = nCpus
      self.nNodes                         = nNodes
      self.ppn                            = ppn
      self.hostAttributes                 = hostAttributes
      self.remoteBatchQueue               = remoteBatchQueue
      self.wallTime                       = wallTime
      self.remoteBatchAccount             = remoteBatchAccount
      self.remoteBatchConstraints         = remoteBatchConstraints
      self.remoteBatchPartition           = remoteBatchPartition
      self.remoteBatchPartitionSize       = remoteBatchPartitionSize
      self.quotaLimit                     = quotaLimit
      self.timestampTransferred           = timestampTransferred
      self.timestampStart                 = timestampStart
      self.timestampFinish                = timestampFinish
      self.timeResults                    = timeResults
      self.batchLogName                   = ""
      self.appScriptName                  = ""
      self.batchScriptName                = ""
      self.nodeFileName                   = ""
      self.remoteJobIdNumber              = ""
      self.sshBaseCommand                 = ""

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob = False
         self.trialDirectory = os.getcwd()
      else:
         self.isBatchJob = True
         self.trialDirectory = "%s_%02d" % (self.localJobId,trial)

      self.jobSubmitted  = False
      self.jobStatistics = {}
      self.jobIndex = 0
      if self.isMultiCoreRequest:
         self.jobStatistics[self.jobIndex] = JobStatistic(nCpus)
      else:
         self.jobStatistics[self.jobIndex] = JobStatistic(1)


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      log("workingDirectory " + self.workingDirectory)
      if   self.remoteBatchSystem == 'PBS':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS(self.localJobId,self.trial,appScriptName,self.environment,
                                      self.stageFiles,self.transferExecutable,
                                      self.executable,self.remoteApplicationRootDirectory,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.computationMode,
                                      self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                      self.nNodes,self.ppn,self.hostAttributes,self.remoteBatchQueue,
                                      self.wallTime,self.remoteBatchAccount,self.quotaLimit,
                                      self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'PBS8':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchPBS8(self.localJobId,self.trial,appScriptName,self.environment,
                                       self.stageFiles,self.transferExecutable,
                                       self.executable,self.remoteApplicationRootDirectory,self.arguments,self.stdinput,
                                       self.isMultiCoreRequest,self.computationMode,
                                       self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                       self.nNodes,self.ppn,self.hostAttributes,self.remoteBatchQueue,
                                       self.wallTime,self.remoteBatchAccount,self.quotaLimit,
                                       self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'LSF':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLSF(self.localJobId,self.trial,appScriptName,self.environment,
                                      self.stageFiles,self.transferExecutable,
                                      self.executable,self.remoteApplicationRootDirectory,self.arguments,self.stdinput,
                                      self.isMultiCoreRequest,self.computationMode,
                                       self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                      self.nNodes,self.ppn,self.wallTime,self.remoteBatchAccount,
                                      self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'LL':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchLL(self.localJobId,self.trial,appScriptName,self.environment,
                                     self.stageFiles,self.transferExecutable,
                                     self.executable,self.remoteApplicationRootDirectory,self.arguments,self.stdinput,
                                     self.isMultiCoreRequest,self.computationMode,
                                     self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                     self.nNodes,self.ppn,
                                     self.remoteBatchPartition,self.remoteBatchPartitionSize,self.remoteBatchConstraints,
                                     self.wallTime,self.quotaLimit,
                                     self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'SLURM':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchSLURM(self.localJobId,self.trial,appScriptName,self.environment,
                                        self.stageFiles,self.transferExecutable,
                                        self.executable,self.remoteApplicationRootDirectory,self.arguments,self.stdinput,
                                        self.isMultiCoreRequest,self.computationMode,
                                        self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                        self.nNodes,self.ppn,
                                        self.remoteBatchPartition,self.remoteBatchPartitionSize,self.remoteBatchConstraints,
                                        self.wallTime,self.quotaLimit,
                                        self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'CONDOR':
         remoteAppScript = RemoteBatchAppScript(self.localJobId,self.trial,
                                                self.stageFiles,self.transferExecutable,
                                                self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                self.useEnvironment,self.environment,
                                                self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                self.timestampStart,self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteBatchCONDOR(self.localJobId,self.trial,appScriptName,self.environment,
                                         self.isMultiCoreRequest,self.computationMode,
                                         self.wallTime)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
      elif self.remoteBatchSystem == 'SCRIPT':
         remoteAppScript = RemoteInstantAppScript(self.localJobId,self.trial,self.workingDirectory,
                                                  self.stageFiles,self.transferExecutable,
                                                  self.executable,self.remoteApplicationRootDirectory,self.stdinput,self.arguments,
                                                  self.useEnvironment,self.environment,
                                                  self.isMultiCoreRequest,self.computationMode,self.mpiRankVariable,
                                                  self.timestampTransferred,self.timestampStart,
                                                  self.timestampFinish,self.timeResults)
         appScriptName,appScript = remoteAppScript.buildAppScript()
         remoteBatch = RemoteInstantSCRIPT(self.localJobId,self.trial,self.workingDirectory,appScriptName,
                                           self.isMultiCoreRequest,self.computationMode,
                                           self.preManagerCommands,self.managerCommand,self.postManagerCommands,
                                           self.nNodes,self.ppn,self.venue,self.venues,self.venueIndex,
                                           self.timestampTransferred,self.timestampStart,self.timestampFinish,self.timeResults)
         batchLogName,batchScriptName,batchScript = remoteBatch.buildBatchScript()
         nodeFileName,nodeList = remoteBatch.getBatchNodeList()
         if nodeFileName:
            fpNodes = open(nodeFileName,'w')
            if fpNodes:
               fpNodes.write('\n'.join(nodeList)+'\n')
               fpNodes.close()
               self.nodeFileName = nodeFileName

      self.batchLogName    = batchLogName
      self.appScriptName   = appScriptName
      self.batchScriptName = batchScriptName
      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = 1
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

      if batchScript != "":
         fpBatchScript = open(batchScriptName,'w')
         if fpBatchScript:
            fpBatchScript.write(batchScript)
            fpBatchScript.close()
            scriptFiles.append(batchScriptName)
            if self.remoteBatchSystem == 'SCRIPT':
               os.chmod(batchScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         else:
            log("could not open %s for writing" % (batchScriptName))
            sys.stderr.write("could not open %s for writing\n" % (batchScriptName))
            sys.stderr.flush()
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if appScript != "":
         fpAppScript = open(appScriptName,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(appScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            scriptFiles.append(appScriptName)
         else:
            log("could not open %s for writing" % (appScriptName))
            sys.stderr.write("could not open %s for writing\n" % (appScriptName))
            sys.stderr.flush()
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self,
                 remoteTunnelMonitor,
                 tunnelsInfo):
      self.filesSent = True
      if self.isBatchJob:
         os.mkdir(self.trialDirectory)

      command = 'update-known-hosts' + " " + self.venue
      log("command = " + command)
      exitStatus,stdOutput,stdError = self.executeCommand(command)

      if self.remoteBatchQueue != "":
         self.jobStatistics[self.jobIndex]['venue'] = self.remoteBatchQueue + '@' + self.venue
      else:
         self.jobStatistics[self.jobIndex]['venue'] = self.venue
      if not exitStatus and self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         self.sshBaseCommand = "gsissh -T -x -a " + self.venue

         command = "cat " + self.stageInTarFile + " | " + self.sshBaseCommand + \
                                " \"" + self.receiveInput + " " + remoteWorkingDirectory + " " + self.timestampTransferred + "\""
         log("command = " + command)
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         exitStatus,stdOutput,stdError = self.executeSSHCommand(command,remoteTunnelMonitor,"")
         log(stdOutput)

      if exitStatus:
         self.filesSent = False
         self.jobStatistics[self.jobIndex]['exitCode'] = 11
#        if stdOutput != "":
#           stdFile = os.path.join(self.trialDirectory,"%s_%02d.stdout" % (self.localJobId,self.trial))
#           fpStd = open(stdFile,'a')
#           if fpStd:
#              fpStd.write(stdOutput)
#              fpStd.close()
         if stdError != "":
            stdFile = os.path.join(self.trialDirectory,"%s_%02d.stderr" % (self.localJobId,self.trial))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.filesSent)


   def executeJob(self,
                  hubUserName,
                  hubUserId,
                  remoteJobMonitor,
                  remoteTunnelMonitor):
      exitStatus = 0

      if not self.stageFiles:
         if self.remoteBatchSystem == 'SCRIPT':
            self.sshBaseCommand = "gsissh -T -x -a " + self.venue
            if self.batchScriptName != "":
               command = self.sshBaseCommand + " " + os.path.join(os.getcwd(),self.batchScriptName)
            else:
               command = self.sshBaseCommand + " " + os.path.join(os.getcwd(),self.appScriptName)
            log("command = " + command)
            exitStatus,stdOutput,stdError = self.executeCommand(command,True)
            self.jobSubmitted = True
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         if self.remoteBatchSystem == 'SCRIPT':
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            if self.batchScriptName != "":
               commandArgs.append(os.path.join('.',self.batchScriptName))
            else:
               commandArgs.append(os.path.join('.',self.appScriptName))
            if self.logUserRemotely:
               commandArgs.append(hubUserName)
               commandArgs.append(str(hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + str(self.trial))
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""

            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,remoteTunnelMonitor,"",True)
            self.jobSubmitted = True
         else:
            commandArgs = []
            commandArgs.append(self.submitBatchJob)
            commandArgs.append(remoteWorkingDirectory)
            commandArgs.append(self.batchScriptName)
            if self.logUserRemotely:
               commandArgs.append(hubUserName)
               commandArgs.append(str(hubUserId))
               commandArgs.append(self.localJobId.lstrip('0') + '_' + str(self.trial))
               commandArgs.append(self.executable)
            command = self.sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,remoteJobId,stdError = self.executeSSHCommand(command,remoteTunnelMonitor,"")
            if not exitStatus:
               remoteJobId = remoteJobId.strip()
               log("remoteJobId = " + remoteJobId)
               self.jobSubmitted = True
               if   self.remoteBatchSystem == 'PBS':
                  self.remoteJobIdNumber = remoteJobId.split('.')[0]
               elif self.remoteBatchSystem == 'PBS8':
                  self.remoteJobIdNumber = remoteJobId.split('.')[0]
               elif self.remoteBatchSystem == 'LSF':
#              Job <851126> is submitted to default queue <normal>.
                  self.remoteJobIdNumber = remoteJobId.split('<')[1].split('>')[0]
               elif self.remoteBatchSystem == 'LL':
                  self.remoteJobIdNumber = remoteJobId
               elif self.remoteBatchSystem == 'SLURM':
                  self.remoteJobIdNumber = remoteJobId.split()[-1]
               elif self.remoteBatchSystem == 'CONDOR':
#              Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
                  self.remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"

               self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
               remoteJobMonitor.postNewJobSubmission(self.siteMonitorDesignator,self.remoteJobIdNumber)

         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
#        if stdOutput != "":
#           stdFile = os.path.join(self.trialDirectory,"%s_%02d.stdout" % (self.localJobId,self.trial))
#           fpStd = open(stdFile,'a')
#           if fpStd:
#              fpStd.write(stdOutput)
#              fpStd.close()
         if stdError != "":
            stdFile = os.path.join(self.trialDirectory,"%s_%02d.stderr" % (self.localJobId,self.trial))
            fpStd = open(stdFile,'a')
            if fpStd:
               fpStd.write(stdError)
               fpStd.close()

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      return((self.isBatchJob,self.siteMonitorDesignator,self.remoteJobIdNumber,self.destination))


   def waitForJob(self,
                  remoteJobMonitor):
      if (self.siteMonitorDesignator != "") and \
         (self.remoteJobIdNumber != "") and \
         (self.destination != ""):
         remoteJobMonitor.waitForBatchJob(self.siteMonitorDesignator,self.remoteJobIdNumber,self.destination)


   def retrieveFiles(self,
                     remoteTunnelMonitor):
      exitStatus = 0

      if self.stageFiles:
         remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
         if self.jobSubmitted:
            command = self.sshBaseCommand + " \"" + self.transmitResults + " " + remoteWorkingDirectory + "\"" + \
                                            " | tar xzmf - -C " + self.trialDirectory
            log("command = " + command)
            os.environ['X509_USER_PROXY'] = self.x509ProxyPath
            exitStatus,stdOutput,stdError = self.executeSSHCommand(command,remoteTunnelMonitor,"")
            log(stdOutput)
            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 stdFile = os.path.join(self.trialDirectory,"%s_%02d.stdout" % (self.localJobId,self.trial))
#                 fpStd = open(stdFile,'a')
#                 if fpStd:
#                    fpStd.write(stdOutput)
#                    fpStd.close()
               if stdError != "":
                  stdFile = os.path.join(self.trialDirectory,"%s_%02d.stderr" % (self.localJobId,self.trial))
                  fpStd = open(stdFile,'a')
                  if fpStd:
                     fpStd.write(stdError)
                     fpStd.close()

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self,
                    remoteTunnelMonitor):
      if not self.filesCleanedup:
         if self.stageFiles:
            remoteWorkingDirectory = self.workingDirectory.replace("$","\$")
            if self.jobSubmitted:
               command = self.sshBaseCommand + " \"" + self.cleanupJob + " " + remoteWorkingDirectory + "\""
               log("command = " + command)
               os.environ['X509_USER_PROXY'] = self.x509ProxyPath
               log(self.executeSSHCommand(command,remoteTunnelMonitor,"")[1])

         if self.batchScriptName:
            if os.path.isfile(self.batchScriptName):
               os.remove(self.batchScriptName)
         if self.appScriptName:
            if os.path.isfile(self.appScriptName):
               os.remove(self.appScriptName)
         if self.nodeFileName:
            if os.path.isfile(self.nodeFileName):
               os.remove(self.nodeFileName)
         if self.batchLogName:
            logFile = os.path.join(self.trialDirectory,self.batchLogName)
            if os.path.isfile(logFile):
               os.remove(logFile)

         self.filesCleanedup = True


   def removeTrialDirectory(self):
      if self.trialDirectory != os.getcwd():
         shutil.rmtree(self.trialDirectory,True)


   def killScripts(self,
                   signalNumber,
                   remoteTunnelMonitor):
      if   self.remoteJobIdNumber:
         command = self.sshBaseCommand + \
                   " \"" + self.killBatchJob + " " + self.remoteJobIdNumber + " " + self.remoteBatchSystem + "\""
         log("command = " + command)
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         log(self.executeSSHCommand(command,remoteTunnelMonitor,"")[1])
      elif self.remoteBatchSystem == 'SCRIPT' and self.childPid != 0:
#        log("Send TERM to child script process %d" % (self.childPid))
#        os.kill(childPid,signalNumber.SIGTERM)
         if self.batchScriptName != "":
            command = self.sshBaseCommand + " pkill -TERM -f " + self.batchScriptName
         else:
            command = self.sshBaseCommand + " pkill -TERM -f " + self.appScriptName
         log("command = " + command)
         os.environ['X509_USER_PROXY'] = self.x509ProxyPath
         log(self.executeSSHCommand(command,remoteTunnelMonitor,"")[1])

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 8 | signalNumber

      self.scriptsKilled = True

