#
# @package      hubzero-submit-distributor
# @file         ToolsInfo.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import os.path
import re
import random

from LogMessage      import log as log
from GroupMembership import isGroupMember

class ToolsInfo:
   def __init__(self,
                infoDirectory,
                toolsFile):
      self.tools      = {}

      toolPattern     = re.compile('(.*\[)(.*)(])')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      toolName        = ""

      fpInfo = open(os.path.join(infoDirectory,toolsFile), "r")
      if fpInfo:
         eof = False
         while not eof:
            record = fpInfo.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   toolPattern.match(record):
                  toolName = toolPattern.match(record).group(2)
                  if not toolName in self.tools:
                     self.tools[toolName] = []
                  toolGroupIndex = len(self.tools[toolName])
                  self.tools[toolName].append({'destinations':[], \
                                               'executablePath':'', \
                                               'isMPI':'', \
                                               'restrictedToUsers':[], \
                                               'restrictedToGroups':[], \
                                               'environment':[], \
                                               'softenvExtensions':{}, \
                                               'remoteManager':'', \
                                               'state':'enabled' \
                                              })
               elif keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.tools[toolName][toolGroupIndex]:
                     if   isinstance(self.tools[toolName][toolGroupIndex][key],list):
                        self.tools[toolName][toolGroupIndex][key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.tools[toolName][toolGroupIndex][key],dict):
                        dictKey,dictValue = value.split(':')
                        self.tools[toolName][toolGroupIndex][key][dictKey.strip()] = dictValue.strip()
                     elif isinstance(self.tools[toolName][toolGroupIndex][key],bool):
                        self.tools[toolName][toolGroupIndex][key] = bool(value.lower() == 'true')
                     else:
                        self.tools[toolName][toolGroupIndex][key] = value
                  else:
                     log("Undefined tool key %s %s in tool %s" % (key,value,toolName))
            else:
               eof = True
         fpInfo.close()

         for toolName in self.tools:
            markedForDeletion = []
            for toolGroupIndex in xrange(len(self.tools[toolName])):
               state = self.tools[toolName][toolGroupIndex]['state']
               if state == 'disabled':
                  markedForDeletion.append(toolGroupIndex)
            markedForDeletion.reverse()
            for toolGroupIndex in markedForDeletion:
               del self.tools[toolName][toolGroupIndex]
            del markedForDeletion


   def applyUserRestriction(self,
                            userName):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToUsers = self.tools[toolName][toolGroupIndex]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if userName in restrictedToUsers:
                  userOK = True
               else:
                  userOK = False
               if not userOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def applyGroupRestriction(self):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToGroups = self.tools[toolName][toolGroupIndex]['restrictedToGroups']
            if len(restrictedToGroups) > 0:
               groupOK = False
               for restrictedToGroup in restrictedToGroups:
                  if isGroupMember(restrictedToGroup):
                     groupOK = True
                     break
               if not groupOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def isExecutableTool(self,
                        executable):
      return(executable in self.tools)


   def isPermissionGranted(self,
                           executable):
      permissionGranted = False
      if len(self.tools[executable]) > 0:
         permissionGranted = True

      return(permissionGranted)


   def selectTool(self,
                  executable,
                  userDestinations,
                  remoteProbeMonitor):
      toolDestinations  = []
      executablePath    = ""
      environment       = []
      softenvExtensions = {}
      remoteManager     = ""

      toolInfo = self.tools[executable]
      if len(userDestinations) == 0:
         toolGroupIndexes = range(len(toolInfo))
         while len(toolGroupIndexes) > 0 and len(toolDestinations) == 0:
            toolGroupIndex = random.choice(toolGroupIndexes)
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               toolDestinations.append(toolDestination)
            if remoteProbeMonitor:
               remoteProbeMonitor.purgeOfflineSites(toolDestinations)
            if len(toolDestinations) == 0:
               toolGroupIndexes.remove(toolGroupIndex)
      else:
         toolGroupIndex = -1
         for userDestination in userDestinations:
            for groupIndex in xrange(len(toolInfo)):
               if userDestination in toolInfo[groupIndex]['destinations']:
                  toolGroupIndex = groupIndex
                  break
            if toolGroupIndex != -1:
               break

         if toolGroupIndex != -1:
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if toolDestination in userDestinations:
                  toolDestinations.append(toolDestination)
         else:
            log("Invalid destination(s) specified: %s" % (userDestinations))

      if len(toolDestinations) > 0:
         executablePath       = toolInfo[toolGroupIndex]['executablePath']
         if 'environment' in toolInfo[toolGroupIndex]:
            environment       = toolInfo[toolGroupIndex]['environment']
         if 'softenvExtensions' in toolInfo[toolGroupIndex]:
            softenvExtensions = toolInfo[toolGroupIndex]['softenvExtensions']
         if 'remoteManager' in toolInfo[toolGroupIndex]:
            remoteManager     = toolInfo[toolGroupIndex]['remoteManager']

      return(toolDestinations,executablePath,remoteManager,environment,softenvExtensions)

