#
# @package      hubzero-submit-distributor
# @file         SitesInfo.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import os
import re
import random

from LogMessage      import log as log
from GroupMembership import isGroupMember

class SitesInfo:
   def __init__(self,
                infoDirectory,
                sitesFile,
                allowedVenueMechanisms):
      self.sites = {}

      sitePattern     = re.compile('(.*\[)(.*)(])')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      siteName        = ""

      fpInfo = open(os.path.join(infoDirectory,sitesFile), "r")
      if fpInfo:
         eof = False
         while not eof:
            record = fpInfo.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   sitePattern.match(record):
                  siteName = sitePattern.match(record).group(2)
                  self.sites[siteName] = {'venues':[], \
                                          'tunnelDesignator':'', \
                                          'siteMonitorDesignator':'', \
                                          'venueMechanism':'', \
                                          'venueGatekeeper':',,'.split(','), \
                                          'remoteUser':'', \
                                          'remoteBatchAccount':'', \
                                          'remoteBatchSystem':'', \
                                          'remoteBatchQueue':'', \
                                          'remoteBatchPartition':'', \
                                          'remoteBatchPartitionSize':'', \
                                          'remoteBatchConstraints':'', \
                                          'remoteBinDirectory':os.path.join('${HOME}','bin'), \
                                          'remoteApplicationRootDirectory':'', \
                                          'remoteScratchDirectory':'', \
                                          'remotePpn':'', \
                                          'remoteManager':'', \
                                          'remoteHostAttribute':'', \
                                          'stageFiles':True, \
                                          'passUseEnvironment':False, \
                                          'arbitraryExecutableAllowed':True, \
                                          'members':[], \
                                          'state':'enabled', \
                                          'failoverSite':'None', \
                                          'checkProbeResult':True, \
                                          'restrictedToUsers':[], \
                                          'restrictedToGroups':[], \
                                          'logUserRemotely':False, \
                                          'undeclaredSiteSelectionWeight':0. \
                                         }
               elif keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.sites[siteName]:
                     if   isinstance(self.sites[siteName][key],list):
                        self.sites[siteName][key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.sites[siteName][key],bool):
                        self.sites[siteName][key] = bool(value.lower() == 'true')
                     elif isinstance(self.sites[siteName][key],float):
                        self.sites[siteName][key] = float(value)
                     else:
                        self.sites[siteName][key] = value
                  else:
                     log("Undefined key = value pair %s = %s for site %s" % (key,value,siteName))
            else:
               eof = True
         fpInfo.close()

         for site in self.sites:
            venueMechanism = self.sites[site]['venueMechanism']
            if venueMechanism != '':
               if not venueMechanism in allowedVenueMechanisms:
                  self.sites[site]['state'] = 'restrictedByVenueMechanism'

         userName = os.getenv("USER")
         for site in self.sites:
            restrictedToUsers = self.sites[site]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not userName in restrictedToUsers:
                  self.sites[site]['state'] = 'restrictedByUser'

         for site in self.sites:
            if self.sites[site]['state'] == 'enabled':
               restrictedToGroups = self.sites[site]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.sites[site]['state'] = 'restrictedByGroup'

         for site in self.sites:
            members = self.sites[site]['members']
            if len(members) > 0:
               markedForDeletion = []
               for member in members:
                  if member in self.sites:
                     if self.sites[member]['state'] != 'enabled':
                        markedForDeletion.append(member)
               for member in markedForDeletion:
                  self.sites[site]['members'].remove(member)
               del markedForDeletion

         for site in self.sites:
            failoverSite = self.sites[site]['failoverSite']
            if failoverSite != 'None' and failoverSite != 'grid':
               if self.sites[failoverSite]['state'] != 'enabled':
                  self.sites[site]['failoverSite'] = 'None'


   def getEnabledSites(self):
      enabledSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            enabledSites.append(site)

      return(enabledSites)


   def purgeDisabledSites(self,
                          siteNames):
      reasonsDenied = {}
      markedForDeletion = []
      for siteName in siteNames:
         try:
            site  = self.sites[siteName]
            state = site['state']
            if state != 'enabled':
               markedForDeletion.append(siteName)
               reasonsDenied[siteName] = state
         except:
            pass
      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      del markedForDeletion

      return(reasonsDenied)


   def __checkProbeResult(self,
                          siteName):
      try:
         site = self.sites[siteName]
         checkProbeResult = site['checkProbeResult']
      except:
         checkProbeResult = True

      return(checkProbeResult)


   def getIgnoreProbeSites(self):
      ignoreProbeSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            if not self.__checkProbeResult(site):
               ignoreProbeSites.append(site)

      return(ignoreProbeSites)


   def purgeOfflineSites(self,
                         siteNames,
                         remoteProbeMonitor):
      goToGrid = False
      if remoteProbeMonitor:
         markedForDeletion = []
         substituteSites   = []
         for siteName in siteNames:
            if self.__checkProbeResult(siteName) and not remoteProbeMonitor.isSiteAvailable(siteName):
               markedForDeletion.append(siteName)
               try:
                  site         = self.sites[siteName]
                  failoverSite = site['failoverSite']
                  if not failoverSite == 'None':
                     if failoverSite == 'grid':
                        goToGrid = True
                     else:
                        if remoteProbeMonitor.isSiteAvailable(failoverSite):
                           substituteSites.append(failoverSite)
               except:
                  pass
         for siteName in markedForDeletion:
            siteNames.remove(siteName)
         if len(siteNames) > 0:
            goToGrid = False
         else:
            if len(substituteSites) > 0:
               for siteName in substituteSites:
                  siteNames.append(siteName)
               goToGrid = False

         del substituteSites
         del markedForDeletion

      return(goToGrid)


   def selectUndeclaredSites(self,
                             siteNames,
                             arbitraryExecutableRequired,
                             maximumSelectedSites):
      selectedUndeclaredSites = []

      if len(siteNames) > 0:
         siteAdded = True
         while siteAdded and (len(selectedUndeclaredSites) < maximumSelectedSites):
            siteAdded = False
            undeclaredSiteSelectionWeights = {}
            possibleSites = []
            for siteName in siteNames:
               if not siteName in selectedUndeclaredSites:
                  try:
                     site    = self.sites[siteName]
                     members = site['members']
                     if len(members) > 0:
                        addMembers = True
                        for member in members:
                           if member in selectedUndeclaredSites:
                              addMembers = False
                              break
                        if addMembers:
                           undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                           for member in members:
                              if not member in possibleSites:
                                 possibleSites.append(member)
                                 undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                              else:
                                 undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
                     else:
                        if not siteName in possibleSites:
                           undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                           possibleSites.append(siteName)
                           undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                        else:
                           undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
                  except:
                     pass

            if len(possibleSites) > 0:
               if arbitraryExecutableRequired:
                  markedForDeletion = []
                  for possibleSite in possibleSites:
                     try:
                        site = self.sites[possibleSite]
                        if not site['arbitraryExecutableAllowed']:
                           markedForDeletion.append(possibleSite)
                     except:
                        markedForDeletion.append(possibleSite)
                  for possibleSite in markedForDeletion:
                     possibleSites.remove(possibleSite)
                  del markedForDeletion

               if len(possibleSites) > 0:
                  nWeights = 0
                  undeclaredSelectionWeightSum = 0.
                  markedForDeletion = []
                  for possibleSite in possibleSites:
                     if   undeclaredSiteSelectionWeights[possibleSite] > 0.:
                        nWeights += 1
                        undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]
                     elif undeclaredSiteSelectionWeights[possibleSite] < 0.:
                        markedForDeletion.append(possibleSite)
                  for possibleSite in markedForDeletion:
                     possibleSites.remove(possibleSite)
                  del markedForDeletion

                  if nWeights == 0:
                     possibleIndex = random.randint(0,len(possibleSites)-1)
                     selectedSite = possibleSites[possibleIndex]
                     selectedUndeclaredSites.append(selectedSite)
                     siteAdded = True
                  else:
                     randomWeight = random.random()
                     weightSum = 0.
                     for possibleSite in possibleSites:
                        undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                        if undeclaredSiteSelectionWeight > 0.:
                           weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                           if weightSum >= randomWeight:
                              selectedSite = possibleSite
                              selectedUndeclaredSites.append(selectedSite)
                              siteAdded = True
                              break

            del possibleSites
            del undeclaredSiteSelectionWeights

      return(selectedUndeclaredSites)


   def selectSite(self,
                  siteNames,
                  arbitraryExecutableRequired):
      selectedSite = ""
      undeclaredSiteSelectionWeights = {}

      if len(siteNames) > 0:
         possibleSites = []
         for siteName in siteNames:
            try:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight'] / float(len(members))
                  for member in members:
                     if not member in possibleSites:
                        possibleSites.append(member)
                        undeclaredSiteSelectionWeights[member] = undeclaredSiteSelectionWeight
                     else:
                        undeclaredSiteSelectionWeights[member] += undeclaredSiteSelectionWeight
               else:
                  if not siteName in possibleSites:
                     undeclaredSiteSelectionWeight = site['undeclaredSiteSelectionWeight']
                     possibleSites.append(siteName)
                     undeclaredSiteSelectionWeights[siteName] = undeclaredSiteSelectionWeight
                  else:
                     undeclaredSiteSelectionWeights[siteName] += undeclaredSiteSelectionWeight
            except:
               pass

         if len(possibleSites) > 0:
            if arbitraryExecutableRequired:
               markedForDeletion = []
               for possibleSite in possibleSites:
                  try:
                     site = self.sites[possibleSite]
                     if not site['arbitraryExecutableAllowed']:
                        markedForDeletion.append(possibleSite)
                  except:
                     markedForDeletion.append(possibleSite)
               for possibleSite in markedForDeletion:
                  possibleSites.remove(possibleSite)
               del markedForDeletion

            if len(possibleSites) > 0:
               undeclaredSelectionWeightSum = 0.
               for possibleSite in possibleSites:
                  undeclaredSelectionWeightSum += undeclaredSiteSelectionWeights[possibleSite]

               if undeclaredSelectionWeightSum == 0.:
                  possibleIndex = random.randint(0,len(possibleSites)-1)
                  selectedSite = possibleSites[possibleIndex]
               else:
                  randomWeight = random.random()
                  weightSum = 0.
                  for possibleSite in possibleSites:
                     undeclaredSiteSelectionWeight = undeclaredSiteSelectionWeights[possibleSite]
                     if undeclaredSiteSelectionWeight > 0.:
                        weightSum += undeclaredSiteSelectionWeight/undeclaredSelectionWeightSum
                        if weightSum >= randomWeight:
                           selectedSite = possibleSite
                           break

      del undeclaredSiteSelectionWeights

      return(selectedSite)


   def selectSites(self,
                   siteNames):
      selectedSites = []

      if len(siteNames) > 0:
         siteNameIndex = random.randint(0,len(siteNames)-1)
         siteName = siteNames[siteNameIndex]
         try:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               memberIndex = random.randint(0,len(members)-1)
               selectedSites.append(members[memberIndex])
            else:
               selectedSites.append(siteName)
         except:
            markedForDeletion = []
            for siteName in siteNames:
               try:
                  site = self.sites[siteName]
                  markedForDeletion.append(siteName)
               except:
                  selectedSites.append(siteName)
            for siteName in markedForDeletion:
               siteNames.remove(siteName)
            del markedForDeletion

      return(selectedSites)


   def getSiteKeyValue(self,
                       siteName,
                       key):
      value = ""

      if siteName in self.sites:
         site    = self.sites[siteName]
         members = site['members']
         if len(members) > 0:
            for member in members:
               if member in self.sites:
                  site = self.sites[member]
                  if key in site:
                     value = site[key]
                     break
         else:
            if key in site:
               value = site[key]

      return(value)


   def getSiteVenues(self,
                     siteName):
      venues = []
      if siteName in self.sites:
         site = self.sites[siteName]
         if 'venues' in site:
            venues = site['venues']

      return(venues)


   def getSitesWithKeyValue(self,
                            key,
                            value,
                            siteNames):
      sitesWithKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if site[key] == value:
                              sitesWithKeyValue.append(member)
               else:
                  if key in site:
                     if site[key] == value:
                        sitesWithKeyValue.append(siteName)

      return(sitesWithKeyValue)


   def getSitesWithoutKeyValue(self,
                               key,
                               value,
                               siteNames):
      sitesWithoutKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if site[key] != value:
                              sitesWithoutKeyValue.append(member)
                        else:
                           sitesWithoutKeyValue.append(siteName)
               else:
                  if key in site:
                     if site[key] != value:
                        sitesWithoutKeyValue.append(siteName)
                  else:
                     sitesWithoutKeyValue.append(siteName)

      return(sitesWithoutKeyValue)


   def siteExists(self,
                  siteName):
      return(siteName in self.sites)


   def getExpandedSiteNames(self,
                            siteNames,
                            remoteProbeMonitor):
      expandedSiteNames = []

      for siteName in siteNames:
         if siteName in self.sites:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               for member in members:
                  if member in self.sites:
                     if self.__checkProbeResult(member):
                        if remoteProbeMonitor.isSiteAvailable(member):
                           expandedSiteNames.append(member)
                     else:
                        expandedSiteNames.append(member)
                  else:
                     expandedSiteNames.append(siteName)
            else:
               if self.__checkProbeResult(siteName):
                  if remoteProbeMonitor.isSiteAvailable(siteName):
                     expandedSiteNames.append(siteName)
               else:
                  expandedSiteNames.append(siteName)
         else:
            expandedSiteNames.append(siteName)

      random.shuffle(expandedSiteNames)

      return(expandedSiteNames)


   def getSiteInfo(self,
                   siteName):
      site = self.sites[siteName]

      venues                         = site['venues']
      if len(venues) > 0:
         venueIndex                  = random.randint(0,len(venues)-1)
         venue                       = venues[venueIndex]
      else:
         venueIndex                  = -1
         venue                       = ""

      tunnelDesignator               = site['tunnelDesignator']
      venueMechanism                 = site['venueMechanism']
      siteMonitorDesignator          = site['siteMonitorDesignator']
      venueGatekeeper                = site['venueGatekeeper']
      remoteUser                     = site['remoteUser']
      stageFiles                     = site['stageFiles']
      passUseEnvironment             = site['passUseEnvironment']
      remoteBatchAccount             = site['remoteBatchAccount']
      remoteBatchSystem              = site['remoteBatchSystem']
      remoteBatchQueue               = site['remoteBatchQueue']
      remoteBatchPartition           = site['remoteBatchPartition']
      remoteBatchPartitionSize       = site['remoteBatchPartitionSize']
      remoteBatchConstraints         = site['remoteBatchConstraints']
      binDirectory                   = site['remoteBinDirectory']
      remoteApplicationRootDirectory = site['remoteApplicationRootDirectory']
      scratchDirectory               = site['remoteScratchDirectory']
      if 'remoteManager' in site:
         remoteManager               = site['remoteManager']
      else:
         remoteManager               = ""

      hostAttributes                 = site['remoteHostAttribute']
      arbitraryExecutableAllowed     = site['arbitraryExecutableAllowed']
      logUserRemotely                = site['logUserRemotely']
      remotePpn                      = site['remotePpn']

      return(venues,venueIndex,venue,venueMechanism,tunnelDesignator,siteMonitorDesignator,venueGatekeeper,remoteUser,
             stageFiles,passUseEnvironment,logUserRemotely,
             remoteBatchAccount,remoteBatchSystem,remoteBatchQueue,
             remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints,
             binDirectory,remoteApplicationRootDirectory,scratchDirectory,
             remoteManager,hostAttributes,arbitraryExecutableAllowed,remotePpn)

