#
# @package      hubzero-submit-distributor
# @file         LocalBatchCONDOR.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import re

class LocalBatchCONDOR:
   def __init__(self,
                localJobId,
                trial,
                trialDirectory,
                stageFiles,
                appScriptName,
                environment,
                isMultiCoreRequest,
                computationMode,
                wallTime):
      self.localJobId         = localJobId
      self.trial              = trial
      self.trialDirectory     = trialDirectory
      self.stageFiles         = stageFiles
      self.appScriptName      = appScriptName
      self.environment        = environment
      self.isMultiCoreRequest = isMultiCoreRequest
      self.computationMode    = computationMode
      self.wallTime           = str(int(wallTime))


   def __makeSerialStagedTemplate(self):
      return """universe = vanilla
executable = APPSCRIPTNAME
transfer_executable = true
transfer_input_files = JOBID_input.tar.gz
transfer_output_files = JOBID_output.tar.gz
environment = "ENVIRONMENT"
output = TRIALDIRECTORY/condor_JOBID.stdout
error = TRIALDIRECTORY/condor_JOBID.stderr
log = TRIALDIRECTORY/condor_JOBID.log
should_transfer_files = YES
when_to_transfer_output = ON_EXIT
notification = never
periodic_release = (CurrentTime > (EnteredCurrentStatus + 300)) && ((HoldReasonCode > 5) && (HoldReasonCode < 15))
# (Job is Held AND Error && # Holds >= 3)
periodic_remove = (JobStatus == 5) && ((HoldReasonCode > 5) && (HoldReasonCode < 15)) && (NumShadowStarts >= 3)
queue
"""


   def __makeSerialNotStagedTemplate(self):
      return """universe = vanilla
executable = APPSCRIPTNAME
transfer_executable = false
environment = "ENVIRONMENT"
output = condor_JOBID.stdout
error = condor_JOBID.stderr
log = condor_JOBID.log
when_to_transfer_output = ON_EXIT
notification = never
periodic_release = (CurrentTime > (EnteredCurrentStatus + 300)) && ((HoldReasonCode > 5) && (HoldReasonCode < 15))
# (Job is Held AND Error && # Holds >= 3)
periodic_remove = (JobStatus == 5) && ((HoldReasonCode > 5) && (HoldReasonCode < 15)) && (NumShadowStarts >= 3)
queue
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_jobid          = re.compile("JOBID")
      re_appScriptName  = re.compile("APPSCRIPTNAME")
      re_walltime       = re.compile("WALLTIME")
      re_environment    = re.compile("ENVIRONMENT")
      re_trialDirectory = re.compile("TRIALDIRECTORY")

      if self.stageFiles:
         template = self.__makeSerialStagedTemplate()
      else:
         template = self.__makeSerialNotStagedTemplate()

      template = re_jobid.sub("%s_%02d" % (self.localJobId,self.trial),template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_walltime.sub(self.wallTime,template)
      template = re_environment.sub(self.environment,template)
      template = re_trialDirectory.sub(self.trialDirectory,template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = ""
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
      else:
         batchLogName = "condor_%s_%02d.log" % (self.localJobId,self.trial)
         batchScript = self.__buildSerialFile()
         batchScriptName = "%s_%02d.condor" % (self.localJobId,self.trial)

      return(batchLogName,batchScriptName,batchScript)

