#
# @package      hubzero-submit-distributor
# @file         JobDistributor.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
from optparse import OptionParser,SUPPRESS_HELP
import sys
import os
import re
import signal
import select
import time
import popen2
import random
import traceback
import math
import copy

from LogMessage               import openLog, logJobId as log, logSetJobId
from SitesInfo                import SitesInfo
from ToolsInfo                import ToolsInfo
from ManagersInfo             import ManagersInfo
from TunnelsInfo              import TunnelsInfo
from EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from VenueMechanismLocal      import VenueMechanismLocal
from VenueMechanismSsh        import VenueMechanismSsh
from VenueMechanismGsiSsh     import VenueMechanismGsiSsh
from RemoteJobMonitor         import RemoteJobMonitor
from RemoteTunnelMonitor      import RemoteTunnelMonitor
from TimeConversion           import hhmmssTomin
from GroupMembership          import isSubmitMember
from TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand
from JobStatistic             import JobStatistic

LOGFILE        = "distributor.log"
GRIDLOGFILE    = "gridjobid.log"
HISTORYLOGFILE = "gridjobhistory.log"

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

LOCALJOBID = ".__local_jobid"

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"

GRIDRESOURCE = ".__grid_resource"
GRIDHISTORY  = ".__grid_history"
GRIDJOBID    = ".__grid_jobid"

class JobDistributor:
   def __init__(self,
                distributorVersion,
                installedDirectory,
                logDirectory,
                checkQuota,
                jobMonitorHost,
                jobMonitorPort,
                probeMonitorHost,
                probeMonitorPort,
                tunnelMonitorHost,
                tunnelMonitorPort,
                allowedVenueMechanisms):

      self.jobId                  = 0
      self.jobs                   = {}
      self.jobStatistics          = {}
      self.fpUserLogPath          = None
      self.abortAttempted         = False
      self.disableStateCheck      = False
      if distributorVersion == 'dev':
         self.disableStateCheck   = True
      self.disableProbeCheck      = True
      self.quotaLimit             = -1
      self.childPid               = 0
      self.bufferSize             = 4096
      self.logPath                = os.path.join(logDirectory,LOGFILE)
      self.gridJobIdLogPath       = os.path.join(logDirectory,GRIDLOGFILE)
      self.gridHistoryLogPath     = os.path.join(logDirectory,HISTORYLOGFILE)
      self.dataDirectory          = installedDirectory
      self.binDirectory           = os.path.join(installedDirectory,'bin')
      self.distributorVersion     = distributorVersion
      self.checkQuota             = checkQuota
      self.jobMonitorHost         = jobMonitorHost
      self.jobMonitorPort         = jobMonitorPort
      self.probeMonitorHost       = probeMonitorHost
      self.probeMonitorPort       = probeMonitorPort
      self.tunnelMonitorHost      = tunnelMonitorHost
      self.tunnelMonitorPort      = tunnelMonitorPort
      self.allowedVenueMechanisms = allowedVenueMechanisms
      self.runjobExists           = False

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.waitForJobsInfo = {}

      self.remoteJobMonitor    = None
      self.remoteProbeMonitor  = None
      self.remoteTunnelMonitor = None

      self.sitesInfo                = None
      self.tunnelsInfo              = None
      self.toolsInfo                = None
      self.managersInfo             = None
      self.environmentWhitelistInfo = None

      self.successfulTrial      = None
      self.maximumSelectedSites = 1
      self.userDestinations     = []
      self.inputFiles           = []
      self.outputFiles          = []
      self.isMultiCoreRequest   = False
      self.nCpus                = 1
      self.ppn                  = ""
      self.wallTime             = 60
      self.environment          = ""
      self.manager              = ""
      self.managerSpecified     = False
      self.computationMode      = 'serial'
      self.preManagerCommands   = []
      self.managerCommand       = ""
      self.postManagerCommands  = []
      self.mpiRankVariable      = ""
      self.managedEnvironment   = []
      self.moduleInitialize     = ""
      self.modulesLoad          = []
      self.modulesUnload        = []
      self.x509SubmitProxy      = ""
      self.transferExecutable   = True
      self.stdinput             = ""
      self.executable           = ""

      self.commandOptions   = None
      self.commandArguments = None
      self.submitVenues     = None
      self.localJobId       = None
      self.hubUserId        = None
      self.hubUserName      = None

      openLog(self.logPath)

      if self.checkQuota:
         self.getUserQuota()

      self.abortGlobal = {}
      self.abortGlobal['abortAttempted'] = self.abortAttempted
      self.abortGlobal['abortSignal']    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def setEnvironment(self,
                      pbsRoot,
                      condorRoot,
                      condorConfig,
                      runjobRoot):
      if pbsRoot != "":
         os.environ['PATH'] = os.path.join(pbsRoot,'bin') + ':' + os.environ['PATH']
      if condorRoot != "":
         os.environ['PATH'] = os.path.join(condorRoot,'bin') + ':' + \
                              os.path.join(condorRoot,'sbin') + ':' + os.environ['PATH']
         if condorConfig != "":
            os.environ['CONDOR_CONFIG'] = condorConfig
         if runjobRoot != "":
            os.environ['PATH'] = os.path.join(runjobRoot,'bin') + ':' + os.environ['PATH']
            if not self.executeCommand("which runjob.sh")[0]:
               self.runjobExists = True
               self.allowedVenueMechanisms.append('runjob')
      os.environ['PATH'] = self.binDirectory + ':' + os.environ['PATH']


   def getEnvironment(self):
      self.hubUserName  = os.getenv("USER")
      self.hubUserId    = os.getuid()
      self.submitVenues = os.getenv("SUBMITVENUES")


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      def grabMultipleArguments(option,opt_str,value,parser):
         assert value is None
         value = getattr(parser.values,option.dest)
         if not value:
            value = []
         rargs = parser.rargs
         while rargs:
            arg = rargs[0]

            # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
            # etc.  Note that this also stops on "-3" or "-3.0", so if
            # your option takes numeric values, you will need to handle
            # this.
            if  (arg == "--"):
               del rargs[0]
               break
            elif((arg[:2] == "--" and len(arg) > 2) or
                 (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
               break
            else:
               if not arg in value:
                  value.append(arg)
               del rargs[0]

         if len(value) > 0:
            setattr(parser.values,option.dest,value)


      def grabStringArgument(option,opt_str,value,parser):
         assert value is None
         value = getattr(parser.values,option.dest)
         if not value:
            value = []
         rargs = parser.rargs
         if rargs:
            arg = rargs[0]

            # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
            # etc.  Note that this also stops on "-3" or "-3.0", so if
            # your option takes numeric values, you will need to handle
            # this.
            if((arg[:2] == "--" and len(arg) > 2) or
               (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
               pass
            else:
               if not arg in value:
                  value.append(arg)
               del rargs[0]

         if len(value) > 0:
            setattr(parser.values,option.dest,value)


      exitCode = 0
      parser = OptionParser(prog="submit",add_help_option=False)
      parser.disable_interspersed_args()
#     parser.add_option("-v","--venue",action="store",type="string",dest="destination", \
#                                      help="Remote job destination")
#     parser.add_option("-j","--jobid",action="store",type="long",dest="jobId", \
#                                      help="Global job identifier")
      parser.add_option("-j","--jobid",action="store",type="long",dest="jobId", \
                                       help=SUPPRESS_HELP)

      if doubleDashTerminator:
         parser.add_option("-v","--venue",action="callback",callback=grabMultipleArguments,dest="destinations", \
                                          help="Remote job destination list terminated by --")
         parser.add_option("-i","--inputfile",action="callback",callback=grabMultipleArguments,dest="inputFiles", \
                                              help="Input file list terminated by --")
#        parser.add_option("-o","--outputfile",action="callback",callback=grabMultipleArguments,dest="outputFiles", \
#                                              help="Output file list terminated by --")
         parser.add_option("-o","--outputfile",action="callback",callback=grabMultipleArguments,dest="outputFiles", \
                                               help=SUPPRESS_HELP)
      else:
#        parser.add_option("-v","--venue",action="append",dest="destinations", \
#                                         help="Remote job destination")
#        parser.add_option("-i","--inputfile",action="append",dest="inputFiles", \
#                                             help="Input file")
##       parser.add_option("-o","--outputfile",action="append",dest="outputFiles", \
##                                             help="Output file")
#        parser.add_option("-o","--outputfile",action="append",dest="outputFiles", \
#                                              help=SUPPRESS_HELP)
         parser.add_option("-v","--venue",action="callback",callback=grabStringArgument,dest="destinations", \
                                          help="Remote job destination")
         parser.add_option("-i","--inputfile",action="callback",callback=grabStringArgument,dest="inputFiles", \
                                              help="Input file")
#        parser.add_option("-o","--outputfile",action="callback",callback=grabStringArgument,dest="outputFiles", \
#                                              help="Output file")
         parser.add_option("-o","--outputfile",action="callback",callback=grabStringArgument,dest="outputFiles", \
                                               help=SUPPRESS_HELP)

      parser.add_option("-n","--nCpus",action="store",type="int",dest="nCpus", \
                                       help="Number of processors for MPI execution")
      parser.add_option("-N","--ppn",action="store",type="int",dest="ppn", \
                                     help="Number of processors/node for MPI execution")
      parser.add_option("-w","--wallTime",action="store",type="string",dest="wallTime", \
                                          help="Estimated walltime hh:mm:ss or minutes")
#     parser.add_option("-e","--env",action="append",type="string",dest="environment", \
#                                    help="Variable=value")
#     parser.add_option("-m","--manager",action="store",type="string",dest="manager", \
#                                        help="Multiprocessor job manager")
      parser.add_option("-e","--env",action="callback",callback=grabStringArgument,dest="environment", \
                                     help="Variable=value")
      parser.add_option("-m","--manager",action="callback",callback=grabStringArgument,dest="manager", \
                                         help="Multiprocessor job manager")
      parser.add_option("-r","--redundancy",action="store",type="int",dest="nRedundant", \
                                       help="Number of indentical simulations to execute in parallel")
      parser.set_defaults(metrics=False)
      parser.add_option("-M","--metrics",action="store_true",dest="metrics", \
                                         help="Report resource usage on exit")
      parser.set_defaults(waitSwitch=False)
      parser.add_option("-W","--wait",action="store_true",dest="waitSwitch", \
                                      help="Wait for reduced job load before submission")
      parser.set_defaults(help=False)
      parser.add_option("-h","--help",action="store_true",dest="help", \
                                      help="Report command usage")

      options,arguments = parser.parse_args(sys.argv[1:])

      if options.help:
         parser.print_help()
      else:
         if len(arguments) == 0:
            log("Command must be supplied")
            sys.stderr.write("Command must be supplied\n")
            sys.stderr.flush()
            exitCode = 1

      self.commandOptions   = options
      self.commandArguments = []
      for argument in arguments:
         self.commandArguments.append(argument.replace(' ','\ '))
      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      def getJobId(dataDirectory):
         nextLocalJobId = 0

         exitCode = 0
         localJobIdFileName = os.path.join(dataDirectory,"localJobId.dat")
         if not os.path.isfile(localJobIdFileName):
            fpLocalJobId = open(localJobIdFileName,'w')
            if fpLocalJobId:
               fpLocalJobId.write('0')
               fpLocalJobId.close()
            else:
               log("could not open %s for creating" % (localJobIdFileName))
               sys.stderr.write("could not open %s for creating\n" % (localJobIdFileName))
               sys.stderr.flush()
               exitCode = 1

         if not exitCode:
            fpLocalJobId = open(localJobIdFileName,'r')
            if fpLocalJobId:
               previousLocalJobId = int(fpLocalJobId.readline().strip())
               fpLocalJobId.close()
               nextLocalJobId = previousLocalJobId+1
               fpLocalJobId = open(localJobIdFileName,'w')
               if fpLocalJobId:
                  fpLocalJobId.write('%d' % (nextLocalJobId))
                  fpLocalJobId.close()
               else:
                  log("could not open %s for writing" % (localJobIdFileName))
                  sys.stderr.write("could not open %s for writing\n" % (localJobIdFileName))
                  sys.stderr.flush()
            else:
               log("could not open %s for reading" % (localJobIdFileName))
               sys.stderr.write("could not open %s for reading\n" % (localJobIdFileName))
               sys.stderr.flush()

         return(nextLocalJobId)

      exitCode = 0

      if not self.commandOptions.help:
         if self.commandOptions.jobId:
            self.jobId = self.commandOptions.jobId
         if self.jobId == 0:
            self.jobId = getJobId(self.dataDirectory)

      self.localJobId = "%08d" % (self.jobId)
      logSetJobId(self.jobId)

      if not self.commandOptions.help:
         if self.jobId == 0:
            exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode
      return(exitCode)


   def setInfo(self):
      self.sitesInfo = SitesInfo(self.dataDirectory,"sites.dat",self.allowedVenueMechanisms)
      self.tunnelsInfo = TunnelsInfo(self.dataDirectory,"tunnels.dat")

      self.toolsInfo = ToolsInfo(self.dataDirectory,"tools.dat")
      self.toolsInfo.applyUserRestriction(self.hubUserName)
      self.toolsInfo.applyGroupRestriction()

      self.managersInfo = ManagersInfo(self.dataDirectory,"managers.dat")

      self.environmentWhitelistInfo = EnvironmentWhitelistInfo(self.dataDirectory,"environmentwhitelist.dat")


   def showUsage(self):
      exitCode = 0
      if self.commandOptions.help:
         venues = []
         if self.runjobExists:
            if self.disableStateCheck:
               condorStatusCommand = "condor_status -format '%s\t' Name"
            else:
               condorStatusCommand = "condor_status -avail -format '%s\t' Name"
            log("command = " + condorStatusCommand)
            exitStatus,stdOutput,stdError = self.executeLaunchCommand(condorStatusCommand)
            if exitStatus == 0:
               venues = stdOutput.split()
         for site in self.sitesInfo.getEnabledSites():
            venues.append(site)

         if len(venues) > 0:
            venues.sort()
            sys.stdout.write("\nCurrently available DESTINATIONs are:\n")
            for venue in venues:
               sys.stdout.write("   %s\n" % (venue))
            sys.stdout.flush()
         del venues

         validManagers = self.managersInfo.getManagerNames()
         if len(validManagers) > 0:
            validManagers.sort()
            sys.stdout.write("\nCurrently available MANAGERs are:\n")
            for validManager in validManagers:
               sys.stdout.write("   %s\n" % (validManager))
            sys.stdout.flush()
         del validManagers

         exitCode = 1
         self.jobStatistics[0]['jobSubmissionMechanism'] = "help"
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def reportExitCondition(self):
      if len(self.jobs) == 0:
         jobIndices = self.jobStatistics.keys()
         maximumJobIndex = max(jobIndices)

         try:
            mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
            remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
            remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
            exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
            cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
            elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
            waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
            nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']

            message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                      (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
            log(message)
            message += "\n"
            if self.fpUserLogPath:
               self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
               self.fpUserLogPath.flush()
            os.write(3,message)
         except:
            pass
      else:
         trialIndex = 0
         for trial in self.jobs:
            if (not self.successfulTrial) or \
               (self.successfulTrial and (trial != self.successfulTrial)):
               jobIndices = self.jobs[trial].jobStatistics.keys()
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[trial].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[trial].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[trial].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[trial].jobStatistics[jobIndex]['exitCode']
                        if jobIndex != maximumJobIndex:
                           if exitCode == 0:
                              exitCode = 65533
                        cpuTime        = self.jobs[trial].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[trial].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[trial].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[trial].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[trial].jobStatistics[jobIndex]['nCores']

                        trialIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                  (trialIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        log(message)
                        message += "\n"
                        if self.fpUserLogPath:
                           self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                           self.fpUserLogPath.flush()
                        os.write(3,message)
               except:
                  pass

         if self.successfulTrial:
            trial = self.successfulTrial
            jobIndices = self.jobs[trial].jobStatistics.keys()
            jobIndices.sort()
            maximumJobIndex = max(jobIndices)

            try:
               for jobIndex in jobIndices:
                  if jobIndex != 0 or maximumJobIndex == 0:
                     mechanism      = self.jobs[trial].jobStatistics[jobIndex]['jobSubmissionMechanism']
                     remoteId       = self.jobs[trial].jobStatistics[jobIndex]['remoteJobIdNumber']
                     remoteLocation = self.jobs[trial].jobStatistics[jobIndex]['venue']
                     exitCode       = self.jobs[trial].jobStatistics[jobIndex]['exitCode']
                     if jobIndex != maximumJobIndex:
                        if exitCode == 0:
                           exitCode = 65533
                     cpuTime        = self.jobs[trial].jobStatistics[jobIndex]['userTime'] + \
                                      self.jobs[trial].jobStatistics[jobIndex]['sysTime']
                     elapsedRunTime = self.jobs[trial].jobStatistics[jobIndex]['elapsedRunTime']
                     waitTime       = self.jobs[trial].jobStatistics[jobIndex]['waitingTime']
                     nCpuUsed       = self.jobs[trial].jobStatistics[jobIndex]['nCores']

                     trialIndex += 1
                     message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                               (trialIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                     log(message)
                     message += "\n"
                     if self.fpUserLogPath:
                        self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                        self.fpUserLogPath.flush()
                     os.write(3,message)
            except:
               pass

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortGlobal['abortAttempted'] = self.abortAttempted
         self.abortGlobal['abortSignal']    = signalNumber


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGINT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGHUP!")
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGQUIT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGABRT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGTERM!")
      self.sigGEN_handler(signalNumber,frame)


   def getUserQuota(self):
      quotaCommand = "quota -w | tail -n 1"
      log("quotaCommand = " + quotaCommand)
      exitStatus,stdOutput,stdError = self.executeCommand(quotaCommand)
      if exitStatus == 0:
         try:
            quotaResults = stdOutput.split()
            if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
               currentUsage = int(quotaResults[1].strip("*"))
               self.quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
         except:
            log(stdOutput)
            log(stdError)
            log(traceback.format_exc())


   def getStdInputFile(self,
                       inputFiles):
      stdInputFile = '/dev/null'
      toCheck = []
      stdinFd = sys.stdin.fileno()
      toCheck.append(stdinFd)
      ready = select.select(toCheck,[],[],0.) # wait for input
      if stdinFd in ready[0]:
         content = sys.stdin.read()
         if content != "":
            stdInputFile = ".__%08d.stdin" % (self.jobId)
            stdinFile = open(stdInputFile,'w')
            stdinFile.write(content)
            stdinFile.close()
            inputFiles.append(stdInputFile)
         del content

      return(stdInputFile)


   def executeCommand(self,
                      command,
                      streamOutput=False):
      child         = popen2.Popen3(command,1)
      self.childPid = child.pid
      child.tochild.close() # don't need to talk to child
      childout      = child.fromchild
      childoutFd    = childout.fileno()
      childerr      = child.childerr
      childerrFd    = childerr.fileno()

      outEOF = errEOF = 0

      outData = []
      errData = []

      while 1:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         ready = select.select(toCheck,[],[]) # wait for input
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = 1
            outData.append(outChunk)
            if streamOutput:
               sys.stdout.write(outChunk)
               sys.stdout.flush()

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = 1
            errData.append(errChunk)
            if streamOutput:
               sys.stderr.write(errChunk)
               sys.stderr.flush()

         if outEOF and errEOF:
            break

      err = child.wait()
      self.childPid = 0
      if err != 0:
         if os.WIFSIGNALED(err):
            log("%s failed w/ exit code %d signal %d" % (command,os.WEXITSTATUS(err),os.WTERMSIG(err)))
         else:
            err = os.WEXITSTATUS(err)
            log("%s failed w/ exit code %d" % (command,err))
         if not streamOutput:
            log("%s" % ("".join(errData)))

      return(err,"".join(outData),"".join(errData))


   def executeLaunchCommand(self,
                            launchCommand,
                            streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

      while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      return(exitStatus,stdOutput,stdError)


   def buildJobDescription(self):
      exitCode            = 0

      userDestinations    = []
      inputFiles          = []
      outputFiles         = []
      isMultiCoreRequest  = False
      nCpus               = 1
      ppn                 = ""
      wallTime            = 60
      environment         = ""
      manager             = ""
      managerSpecified    = False
      computationMode     = 'serial'
      preManagerCommands  = []
      managerCommand      = ""
      postManagerCommands = []
      mpiRankVariable     = ""
      managedEnvironment  = []
      moduleInitialize    = ""
      modulesLoad         = []
      modulesUnload       = []
      x509SubmitProxy     = ""

      if self.submitVenues == None:
         if self.commandOptions.destinations:
            userDestinations = self.commandOptions.destinations
      else:
         userDestinations = self.submitVenues.split(',')
      if len(userDestinations) > 0:
         reasonsDenied = self.sitesInfo.purgeDisabledSites(userDestinations)
         if len(userDestinations) == 0:
            errorMessage = "Access to all sites listed as possible execution hosts has been denied.\n"
            for userDestination in reasonsDenied:
               errorMessage += "   %20s %s\n" % (userDestination,reasonsDenied[userDestination])
            errorMessage += "Please select another site or attempt execution at a later time."
            log(errorMessage)
            sys.stderr.write(errorMessage + "\n")
            sys.stderr.flush()
            exitCode = 7
         del reasonsDenied

      if self.commandOptions.inputFiles:
         for inputFile in self.commandOptions.inputFiles:
            inputFiles.append(inputFile.replace(' ','\ '))

      if self.commandOptions.outputFiles:
         for outputFile in self.commandOptions.outputFiles:
            outputFiles.append(outputFile.replace(' ','\ '))

      if self.commandOptions.manager:
         manager = self.commandOptions.manager[0]
         managerExists,computationMode, \
                       preManagerCommands,managerCommand,postManagerCommands, \
                       mpiRankVariable,managedEnvironment, \
                       moduleInitialize,modulesLoad,modulesUnload = self.managersInfo.getManagerInfo(manager)
         if managerExists:
            managerSpecified = True
         else:
            validManagers = self.managersInfo.getManagerNames()
            log("Invalid manager %s specified. Valid managers are %s" % (manager,validManagers))
            sys.stderr.write("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
            sys.stderr.flush()
            del validManagers
            if not exitCode:
               exitCode = 1

      if self.commandOptions.nCpus:
         isMultiCoreRequest = True
         nCpus = int(self.commandOptions.nCpus)

      if self.commandOptions.ppn:
         if isMultiCoreRequest:
            ppn = str(self.commandOptions.ppn)

      if self.commandOptions.nRedundant:
         self.maximumSelectedSites = self.commandOptions.nRedundant

      if self.commandOptions.wallTime:
         if ":" in self.commandOptions.wallTime:
            wallTime = hhmmssTomin(self.commandOptions.wallTime)
            if wallTime < 0.:
               log("wallTime must be > 0")
               sys.stderr.write("wallTime must be > 0\n")
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
         else:
            try:
               wallTime = float(self.commandOptions.wallTime)
            except ValueError:
               log("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes")
               sys.stderr.write("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes\n")
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1

      if self.commandOptions.environment:
         for environmentVariableValue in self.commandOptions.environment:
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               log("Invalid environment variable %s specified." % (environmentVariableValue))
               sys.stderr.write("Invalid environment variable %s specified.\n" % (environmentVariableValue))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1

            if   environmentVariable == "IGNORESTATECHECK":
               self.disableStateCheck = True
            elif environmentVariable == "IGNOREPROBECHECK":
               self.disableProbeCheck = True
            elif environmentVariable == "X509_SUBMIT_PROXY":
               x509SubmitProxy = value
            elif environmentVariable == "USER_LOG_PATH":
               try:
                  self.fpUserLogPath = open(value,'a')
               except:
                  log("User log %s could not be opened." % (value))
                  sys.stderr.write("User log %s could not be opened.\n" % (value))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 1
            elif environmentVariable == "USER_JOBID_PATH":
               try:
                  fpUserJobId = open(value,'w')
                  if fpUserJobId:
                     fpUserJobId.write("%d\n" % (self.jobId))
                     fpUserJobId.close()
               except:
                  log("User jobId file %s could not be created." % (value))
                  sys.stderr.write("User jobId file %s could not be created.\n" % (value))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 1
            elif environmentVariable == "MODULES_INIT_SH":
               moduleInitialize = value
            elif environmentVariable == "MODULES_LOAD":
               for module in value.split(','):
                  modulesLoad.append(module)
            elif environmentVariable == "MODULES_UNLOAD":
               for module in value.split(','):
                  modulesUnload.append(module)
            else:
               if self.environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
                  environment += environmentVariable + "=" + value + " "
               else:
                  log("Environment variable %s could not be set." % (environmentVariable))
                  sys.stderr.write("Environment variable %s could not be set.\n" % (environmentVariable))
                  sys.stderr.flush()
         environment = environment.strip()

      if isMultiCoreRequest:
         if computationMode == 'serial':
            computationMode = 'mpi'

      if len(self.commandArguments) > 1:
         for arg in self.commandArguments[1:]:
            if not arg.startswith('-'):
               if os.path.isfile(arg.replace('\ ',' ')):
                  if not arg in inputFiles:
                     inputFiles.append(arg)

      executable         = ""
      transferExecutable = True
      stdinput           = ""

      executable = self.commandArguments[0]
      if "/" in executable:
         if   not os.path.isfile(executable):
            log("Missing executable: %s" % (executable))
            sys.stderr.write("Missing executable: %s\n" % (executable))
            sys.stderr.flush()
            if not exitCode:
               exitCode = 1
         elif not os.access(executable,os.X_OK):
            log("Permission denied on executable: %s" % (executable))
            sys.stderr.write("Permission denied on executable: %s\n" % (executable))
            sys.stderr.flush()
            if not exitCode:
               exitCode = 1
      else:
         transferExecutable = False

      stdinput = self.getStdInputFile(inputFiles)

      if transferExecutable:
         if not executable.startswith("/apps/"):
            if not isSubmitMember():
               log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
               sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
      else:
         applicationIsStaged = False
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               applicationIsStaged = True
         if not applicationIsStaged and not isSubmitMember():
            log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
            sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
            sys.stderr.flush()
            if not exitCode:
               exitCode = 1

      if len(userDestinations) > 0:
         goToGrid = self.sitesInfo.purgeOfflineSites(userDestinations,self.remoteProbeMonitor)
         if len(userDestinations) == 0 and not goToGrid:
            errorMessage = "All sites listed as possible execution hosts are out of service.\n" + \
                           "Please select another site or attempt execution at a later time."
            log(errorMessage)
            sys.stderr.write(errorMessage + "\n")
            sys.stderr.flush()
            exitCode = 8

      self.userDestinations    = copy.copy(userDestinations)
      self.inputFiles          = copy.copy(inputFiles)
      self.outputFiles         = copy.copy(outputFiles)
      self.isMultiCoreRequest  = copy.copy(isMultiCoreRequest)
      self.nCpus               = copy.copy(nCpus)
      self.ppn                 = copy.copy(ppn)
      self.wallTime            = copy.copy(wallTime)
      self.environment         = copy.copy(environment)
      self.manager             = copy.copy(manager)
      self.managerSpecified    = copy.copy(managerSpecified)
      self.computationMode     = copy.copy(computationMode)
      self.preManagerCommands  = copy.copy(preManagerCommands)
      self.managerCommand      = copy.copy(managerCommand)
      self.postManagerCommands = copy.copy(postManagerCommands)
      self.mpiRankVariable     = copy.copy(mpiRankVariable)
      self.managedEnvironment  = copy.copy(managedEnvironment)
      self.moduleInitialize    = copy.copy(moduleInitialize)
      self.modulesLoad         = copy.copy(modulesLoad)
      self.modulesUnload       = copy.copy(modulesUnload)
      self.x509SubmitProxy     = copy.copy(x509SubmitProxy)
      self.transferExecutable  = copy.copy(transferExecutable)
      self.stdinput            = copy.copy(stdinput)
      self.executable          = copy.copy(executable)

      self.jobStatistics[0]['exitCode'] = exitCode
      return(exitCode)


   def buildJobInstances(self):
      class InstancesError(Exception):
         pass

      class TrialError(Exception):
         pass

      exitCode = 0
      enteredCommand = " ".join(self.commandArguments)

      userDestinations   = copy.copy(self.userDestinations)
      managerSpecified   = copy.copy(self.managerSpecified)
      x509SubmitProxy    = copy.copy(self.x509SubmitProxy)
      transferExecutable = copy.copy(self.transferExecutable)
      stdinput           = copy.copy(self.stdinput)
      executable         = copy.copy(self.executable)

      toolDestinations  = []
      toolEnvironment   = []
      softenvExtensions = {}
      destination       = ""
      destinations      = []

      try:
         if not transferExecutable:
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,self.remoteProbeMonitor)
                  toolDestinations,executablePath,remoteManager, \
                                   toolEnvironment, \
                                   softenvExtensions = self.toolsInfo.selectTool(executable,expandedUserDestinations,
                                                                                 self.remoteProbeMonitor)
                  del expandedUserDestinations
                  if len(toolDestinations) == 0:
                     log("The combination of destination and executable is not properly specified.")
                     sys.stderr.write("The combination of destination and executable is not properly specified.\n")
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstancesError

                  if not managerSpecified and (remoteManager != ""):
                     managerExists,computationMode, \
                                   preManagerCommands,managerCommand,postManagerCommands, \
                                   mpiRankVariable,environmentManager, \
                                   moduleInitializeManager,modulesLoadManager,modulesUnloadManager = \
                                                                                     self.managersInfo.getManagerInfo(remoteManager)
                     if managerExists:
                        self.manager               = copy.copy(remoteManager)
                        self.managerSpecified      = True
                        self.computationMode       = copy.copy(computationMode)
                        self.preManagerCommands    = copy.copy(preManagerCommands)
                        self.managerCommand        = copy.copy(managerCommand)
                        self.postManagerCommands   = copy.copy(postManagerCommands)
                        self.mpiRankVariable       = copy.copy(mpiRankVariable)
                        if len(self.managedEnvironment) == 0:
                           self.managedEnvironment = copy.copy(environmentManager)
                        if self.moduleInitialize == "":
                           self.moduleInitialize   = copy.copy(moduleInitializeManager)
                        if len(self.modulesLoad) == 0:
                           self.modulesLoad        = copy.copy(modulesLoadManager)
                        if len(self.modulesUnload) == 0:
                           self.modulesUnload      = copy.copy(modulesUnloadManager)
                     else:
                        log("Invalid manager %s specified for tool %s." % (remoteManager,executable))
                        sys.stderr.write("Invalid manager %s specified for tool %s.\n" % (remoteManager,executable))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstancesError
                  executable = executablePath
                  self.executable = copy.copy(executable)
               else:
                  log("Access to %s is restricted by user or group" % (executable))
                  sys.stderr.write("Access to %s is restricted by user or group\n" % (executable))
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstancesError

         trialDestinations = []

         if   len(toolDestinations) == 1:
# if tool is staged and only one site is available
            destination = toolDestinations[0]
            trialDestinations.append(destination)
         elif len(toolDestinations) == 0:
# if tool is not staged and only one user requested site is available, userDestinations reduced to grid sites
            destinations = self.sitesInfo.selectSites(userDestinations)
            if len(destinations) == 1:
               destination = destinations[0]
               trialDestinations.append(destination)

         if destination == "":
            if len(toolDestinations) > 1:
# if tool is staged pick non-grid site
               nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('venueMechanism','runjob',toolDestinations)
               destinations = self.sitesInfo.selectSites(nonRunjobSites)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]
                  trialDestinations.append(destination)

         if destination == "":
            if len(userDestinations) > 1:
# if user requested non-grid site is available
               nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('venueMechanism','runjob',userDestinations)
               destinations = self.sitesInfo.selectSites(nonRunjobSites)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]
                  trialDestinations.append(destination)

         if   (self.sitesInfo.siteExists(destination)) and \
               self.sitesInfo.getSiteKeyValue(destination,'venueMechanism') != 'runjob':
# selected a non-grid site
            pass
         elif (self.sitesInfo.siteExists(destination)) and \
               self.sitesInfo.getSiteKeyValue(destination,'venueMechanism') == 'runjob':
# selected a grid site
            pass
         else:
            if   self.runjobExists and (len(toolDestinations) > 0 or len(userDestinations) > 0 or destination != ""):
# defer to tool or user specified runjob grid site selection
               if len(trialDestinations) == 0:
                  trialDestinations.append('2BeDetermined')
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteProbeMonitor:
                  self.remoteProbeMonitor.purgeOfflineSites(enabledSites)
               arbitraryExecutableRequired = not executable.startswith("/apps/")
               destinations = self.sitesInfo.selectUndeclaredSites(enabledSites, \
                                                                   arbitraryExecutableRequired, \
                                                                   self.maximumSelectedSites)
               if len(destinations) > 0:
                  for trialDestination in destinations:
                     trialDestinations.append(trialDestination)
               else:
                  log("Currently no DESTINATIONs are available")
                  sys.stderr.write("Currently no DESTINATIONs are available\n")
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstancesError

         self.remoteJobMonitor    = RemoteJobMonitor(self.jobMonitorHost,self.jobMonitorPort)
         self.remoteTunnelMonitor = RemoteTunnelMonitor(self.tunnelMonitorHost,self.tunnelMonitorPort)

         filesToRemove      = []
         emptyFilesToRemove = []
         trial = 0
         for destination in trialDestinations:
            trial += 1
            exitCode            = 0
            inputFiles          = copy.copy(self.inputFiles)
            outputFiles         = copy.copy(self.outputFiles)
            isMultiCoreRequest  = copy.copy(self.isMultiCoreRequest)
            nCpus               = copy.copy(self.nCpus)
            ppn                 = copy.copy(self.ppn)
            wallTime            = copy.copy(self.wallTime)
            environment         = copy.copy(self.environment)
            manager             = copy.copy(self.manager)
            managerSpecified    = copy.copy(self.managerSpecified)
            computationMode     = copy.copy(self.computationMode)
            preManagerCommands  = copy.copy(self.preManagerCommands)
            managerCommand      = copy.copy(self.managerCommand)
            postManagerCommands = copy.copy(self.postManagerCommands)
            mpiRankVariable     = copy.copy(self.mpiRankVariable)
            managedEnvironment  = copy.copy(self.managedEnvironment)
            moduleInitialize    = copy.copy(self.moduleInitialize)
            modulesLoad         = copy.copy(self.modulesLoad)
            modulesUnload       = copy.copy(self.modulesUnload)
            executable          = copy.copy(self.executable)

            venue                      = "Unknown"
            venueMechanism             = ''
            nNodes                     = "1"
            stageFiles                 = True
            passUseEnvironment         = False
            hostAttributes             = ""
            arbitraryExecutableAllowed = True
            arguments                  = ""

            try:
               if   (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'venueMechanism') != 'runjob':
                  venues,venueIndex,venue,venueMechanism,tunnelDesignator,siteMonitorDesignator,venueGatekeeper,remoteUser, \
                         stageFiles,passUseEnvironment,logUserRemotely, \
                         remoteBatchAccount,remoteBatchSystem,remoteBatchQueue, \
                         remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints, \
                         binDirectory,remoteApplicationRootDirectory,scratchDirectory, \
                         remoteManager,hostAttributes,arbitraryExecutableAllowed,remotePpn = self.sitesInfo.getSiteInfo(destination)
                  if not managerSpecified and remoteManager != "":
                     managerExists,computationMode, \
                                   preManagerCommands,managerCommand,postManagerCommands, \
                                   mpiRankVariable,environmentManager, \
                                   moduleInitializeManager,modulesLoadManager,modulesUnloadManager = \
                                                                                     self.managersInfo.getManagerInfo(remoteManager)
                     if managerExists:
                        managerSpecified = True
                        if len(managedEnvironment) == 0:
                           managedEnvironment = environmentManager
                        if moduleInitialize == "":
                           moduleInitialize = moduleInitializeManager
                        if len(modulesLoad) == 0:
                           modulesLoad = modulesLoadManager
                        if len(modulesUnload) == 0:
                           modulesUnload = modulesUnloadManager
                     else:
                        log("Invalid manager %s specified for site %s." % (remoteManager,destination))
                        sys.stderr.write("Invalid manager %s specified for site %s.\n" % (remoteManager,destination))
                        sys.stderr.flush()
                        exitCode = 1
                        raise TrialError
                  if ppn == "":
                     ppn = remotePpn
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               elif (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'venueMechanism') == 'runjob':
                  venueMechanism        = 'runjob'
                  runjobVenues          = self.sitesInfo.getSiteVenues(destination)
                  gridsite              = " ".join(runjobVenues)
                  siteMonitorDesignator = 'condorG'
                  remoteBatchSystem     = '?'
               else:
                  if   self.runjobExists and (len(toolDestinations) > 0 or len(userDestinations) > 0 or destination != ""):
                     venueMechanism        = 'runjob'
                     if   len(toolDestinations) > 0:
                        gridsite           = " ".join(toolDestinations)
                     elif len(userDestinations) > 0:
                        gridsite           = " ".join(userDestinations)
                     else:
                        gridsite           = destination
                     siteMonitorDesignator = 'condorG'
                     remoteBatchSystem     = '?'
                  elif self.runjobExists or transferExecutable:
                     log("Invalid destination selection")
                     sys.stderr.write("Invalid destination selection\n")
                     sys.stderr.flush()
                     exitCode = 1
                     raise TrialError

               if transferExecutable:
                  if not executable.startswith("/apps/"):
                     if not arbitraryExecutableAllowed:
                        errorMessage = "submission of arbitrary executables is not permitted to selected site." + \
                                       "  Attempting to execute:"
                        log("%s %s" % (errorMessage,executable))
                        sys.stderr.write("%s %s\n" % (errorMessage,executable))
                        sys.stderr.flush()
                        exitCode = 1
                        raise TrialError

               localJobIdFile = "%s.%s_%02d" % (LOCALJOBID,self.localJobId,trial)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               if stageFiles:
                  stageInTarFile = "%s_%02d_input.tar" % (self.localJobId,trial)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  log("command = " + tarCommand)
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     sys.stderr.write(tarStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise TrialError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  remoteArgs.append(self.commandArguments[0])
                  if len(self.commandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.commandArguments[1:]:
                        rarg = arg
                        for tarFile in tarFiles:
                           if re.match(".*" + re.escape(tarFile) + "$",arg):
                              rarg = "./" + tarFile
                              break
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(remoteArgs[1:])
                  arguments = arguments.strip()
               else:
                  stageInTarFile = ""
                  remoteCommand = " ".join(self.commandArguments)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(self.commandArguments[1:])
                  arguments = arguments.strip()

               useEnvironment = ""
               if passUseEnvironment:
                  reChoice = re.compile(".*_CHOICE$")
                  environmentVars = filter(reChoice.search,os.environ.keys())
                  if "PROMPT_CHOICE" in environmentVars:
                     environmentVars.remove("PROMPT_CHOICE")
                  if "EDITOR_CHOICE" in environmentVars:
                     environmentVars.remove("EDITOR_CHOICE")
                  if len(environmentVars) > 0:
                     useEnvironment = ". /apps/environ/.setup\n"
                     for environmentVar in environmentVars:
                        useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolEnvironment) > 0:
                  for environmentVariableValue in toolEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        log("Tool environment variable must be of the form Variable = Value.")
                        sys.stderr.write("Tool environment variable must be of the form Variable = Value.\n")
                        sys.stderr.flush()

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               timestampTransferred = "%s.%s_%02d" % (TIMESTAMPTRANSFERRED,self.localJobId,trial)
               timestampStart       = "%s.%s_%02d" % (TIMESTAMPSTART,self.localJobId,trial)
               timestampFinish      = "%s.%s_%02d" % (TIMESTAMPFINISH,self.localJobId,trial)
               timeResults          = "%s.%s_%02d" % (TIMERESULTS,self.localJobId,trial)
               gridResource         = "%s.%s_%02d" % (GRIDRESOURCE,self.localJobId,trial)
               gridHistory          = "%s.%s_%02d" % (GRIDHISTORY,self.localJobId,trial)
               gridJobId            = "%s.%s_%02d" % (GRIDJOBID,self.localJobId,trial)

               if   venueMechanism == 'local':
                  self.jobs[trial] = VenueMechanismLocal(binDirectory,
                                                         RECEIVEINPUTCOMMAND,SUBMITBATCHJOBCOMMAND,TRANSMITRESULTSCOMMAND,
                                                         CLEANUPJOBCOMMAND,KILLBATCHJOBCOMMAND,remoteBatchSystem,
                                                         self.localJobId,trial,venue,venues,venueIndex,siteMonitorDesignator,
                                                         logUserRemotely,
                                                         stageFiles,transferExecutable,executable,remoteApplicationRootDirectory,
                                                         stdinput,arguments,useEnvironment,environment,
                                                         isMultiCoreRequest,computationMode,mpiRankVariable,
                                                         preManagerCommands,managerCommand,postManagerCommands,
                                                         nCpus,nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,
                                                         remoteBatchAccount,self.quotaLimit,
                                                         timestampTransferred,timestampStart,timestampFinish,timeResults)
               elif venueMechanism == 'ssh':
                  self.jobs[trial] = VenueMechanismSsh(binDirectory,
                                                       RECEIVEINPUTCOMMAND,SUBMITBATCHJOBCOMMAND,TRANSMITRESULTSCOMMAND,
                                                       CLEANUPJOBCOMMAND,KILLBATCHJOBCOMMAND,remoteBatchSystem,
                                                       self.localJobId,trial,destination,venue,venues,venueIndex,
                                                       siteMonitorDesignator,tunnelDesignator,
                                                       remoteUser,logUserRemotely,
                                                       stageFiles,stageInTarFile,scratchDirectory,
                                                       transferExecutable,executable,remoteApplicationRootDirectory,
                                                       stdinput,arguments,useEnvironment,environment,
                                                       isMultiCoreRequest,computationMode,mpiRankVariable,
                                                       preManagerCommands,managerCommand,postManagerCommands,
                                                       nCpus,nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,
                                                       remoteBatchAccount,
                                                       remoteBatchConstraints,remoteBatchPartition,remoteBatchPartitionSize,
                                                       self.quotaLimit,
                                                       timestampTransferred,timestampStart,timestampFinish,timeResults)
               elif venueMechanism == 'gsissh':
                  self.jobs[trial] = VenueMechanismGsiSsh(binDirectory,
                                                          RECEIVEINPUTCOMMAND,SUBMITBATCHJOBCOMMAND,TRANSMITRESULTSCOMMAND,
                                                          CLEANUPJOBCOMMAND,KILLBATCHJOBCOMMAND,remoteBatchSystem,
                                                          self.localJobId,trial,destination,venue,venues,venueIndex,
                                                          siteMonitorDesignator,x509SubmitProxy,
                                                          logUserRemotely,
                                                          stageFiles,stageInTarFile,scratchDirectory,
                                                          transferExecutable,executable,remoteApplicationRootDirectory,
                                                          stdinput,arguments,useEnvironment,environment,
                                                          isMultiCoreRequest,computationMode,mpiRankVariable,
                                                          preManagerCommands,managerCommand,postManagerCommands,
                                                          nCpus,nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,
                                                          remoteBatchAccount,
                                                          remoteBatchConstraints,remoteBatchPartition,remoteBatchPartitionSize,
                                                          self.quotaLimit,
                                                          timestampTransferred,timestampStart,timestampFinish,timeResults)

               exitCode,scriptFiles = self.jobs[trial].createScripts()
               if exitCode:
                  raise TrialError

               if stageFiles:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     log("command = " + tarCommand)
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        log(tarStdError)
                        log(tarStdOutput)
                        sys.stderr.write(tarStdError)
                        sys.stderr.flush()
                        exitCode = 1
                        raise TrialError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  log("command = " + niceCommand)
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     log(gzipStdError)
                     log(gzipStdOutput)
                     sys.stderr.write(gzipStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise TrialError
                  stageInTarFile += '.gz'
                  filesToRemove.append(stageInTarFile)

               self.waitForJobsInfo[trial] = {}
               waitForJobInfo = self.jobs[trial].getWaitForJobInfo()
               self.waitForJobsInfo[trial]['isBatchJob']            = waitForJobInfo[0]
               self.waitForJobsInfo[trial]['siteMonitorDesignator'] = waitForJobInfo[1]

            except TrialError:
               if trial in self.jobs:
                  self.jobs[trial].jobStatistics[0]['jobSubmissionMechanism'] = venueMechanism
                  self.jobs[trial].jobStatistics[0]['venue']                  = venue
                  self.jobs[trial].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = venueMechanism
                  self.jobStatistics[0]['venue']                  = venue
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s.stderr" % (self.localJobId)
            emptyFilesToRemove.append(stdFile)
            for batchQueue in 'pbs','lsf','ll','slurm','condor':
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%02d.%s" % (batchQueue,self.localJobId,trial,outFile)
                  emptyFilesToRemove.append(batchQueueOutputFile)

         nonBatchJobRan = False
         markedForDeletion = []
         for trial in self.waitForJobsInfo:
            if not self.waitForJobsInfo[trial]['isBatchJob']:
               if not nonBatchJobRan:
                  if self.jobs[trial].sendFiles(self.remoteTunnelMonitor,self.tunnelsInfo):
                     if self.jobs[trial].executeJob(self.hubUserName,self.hubUserId,
                                                    self.remoteJobMonitor,self.remoteTunnelMonitor):
                        self.jobs[trial].retrieveFiles(self.remoteTunnelMonitor)
                        self.jobs[trial].cleanupFiles(self.remoteTunnelMonitor)
                  self.jobs[trial].recordJobStatistics()
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'D'
                  self.successfulTrial = trial
                  nonBatchJobRan = True
               else:
                  markedForDeletion.append(trial)

         if nonBatchJobRan:
            for trial in markedForDeletion:
               del self.jobs[trial]
         del markedForDeletion

         jobsRunning = False
         for trial in self.waitForJobsInfo:
            if self.waitForJobsInfo[trial]['isBatchJob']:
               if self.jobs[trial].sendFiles(self.remoteTunnelMonitor,self.tunnelsInfo):
                  if self.jobs[trial].executeJob(self.hubUserName,self.hubUserId,
                                                 self.remoteJobMonitor,self.remoteTunnelMonitor):
                     waitForJobInfo = self.jobs[trial].getWaitForJobInfo()
                     self.waitForJobsInfo[trial]['remoteJobId']     = waitForJobInfo[2]
                     if len(waitForJobInfo) == 4:
                        self.waitForJobsInfo[trial]['knownSite']    = waitForJobInfo[3]
                     else:
                        self.waitForJobsInfo[trial]['knownSite']    = ""
                     self.waitForJobsInfo[trial]['recentJobStatus'] = '?'
                     jobsRunning = True
                  else:
                     self.waitForJobsInfo[trial]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'D'

         while jobsRunning and (not self.successfulTrial) and (not self.abortGlobal['abortAttempted']):
            completeRemoteJobIndexes = self.remoteJobMonitor.waitForBatchJobs(self.waitForJobsInfo,self.abortGlobal)
            for trial in completeRemoteJobIndexes:
               self.jobs[trial].retrieveFiles(self.remoteTunnelMonitor)
               self.jobs[trial].cleanupFiles(self.remoteTunnelMonitor)
               self.jobs[trial].recordJobStatistics()
               if self.jobs[trial].wasJobSuccessful():
                  self.successfulTrial = trial
            if len(completeRemoteJobIndexes) == 0:
               jobsRunning = False

         if self.abortGlobal['abortAttempted']:
            for trial in self.waitForJobsInfo:
               if self.waitForJobsInfo[trial]['recentJobStatus'] != 'D':
                  self.jobs[trial].killScripts(self.abortGlobal['abortSignal'],self.remoteTunnelMonitor)
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'K'
            self.remoteJobMonitor.waitForKilledBatchJobs(self.waitForJobsInfo)
            for trial in self.waitForJobsInfo:
               if self.waitForJobsInfo[trial]['recentJobStatus'] == 'KD':
                  self.jobs[trial].cleanupFiles(self.remoteTunnelMonitor)
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'D'

         if self.successfulTrial:
            for trial in self.waitForJobsInfo:
               if self.waitForJobsInfo[trial]['recentJobStatus'] != 'D':
                  self.jobs[trial].killScripts(signal.SIGUSR1,self.remoteTunnelMonitor)
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'K'
            self.remoteJobMonitor.waitForKilledBatchJobs(self.waitForJobsInfo)
            for trial in self.waitForJobsInfo:
               if   self.waitForJobsInfo[trial]['recentJobStatus'] == 'KD':
                  self.jobs[trial].cleanupFiles(self.remoteTunnelMonitor)
                  self.waitForJobsInfo[trial]['recentJobStatus'] = 'D'
               elif trial != self.successfulTrial:
                  self.jobs[trial].cleanupFiles(self.remoteTunnelMonitor)

         if self.successfulTrial:
            self.jobs[self.successfulTrial].recoverFiles(False)
            fileToMove = "%s_%02d.stdout" % (self.localJobId,self.successfulTrial)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.stdout" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.stderr" % (self.localJobId,self.successfulTrial)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.stderr" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.SUCCESS" % (self.localJobId,self.successfulTrial)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.SUCCESS" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
         else:
            for trial in self.waitForJobsInfo:
               self.jobs[trial].recoverStdFiles()
            if len(self.jobs) > 0:
               recoverTrial = random.choice(xrange(len(self.jobs)))+1
               self.jobs[recoverTrial].recoverFiles(True)

         for trial in self.waitForJobsInfo:
            self.jobs[trial].removeTrialDirectory()

         for fileToRemove in filesToRemove:
            if(os.path.exists(fileToRemove)):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if(os.path.exists(fileToRemove)):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

      except InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(stdinput):
         os.remove(stdinput)

      return(exitCode)

