# @package      hubzero-submit-common
# @file         SubmitClient.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import sys
import re
import shutil
import select
import time
import signal
import resource
import subprocess
import math
import stat
import csv

from hubzero.submit.LogMessage             import logID as log
from hubzero.submit.CommandParser          import CommandParser
from hubzero.submit.ParameterTemplate      import ParameterTemplate
from hubzero.submit.LocalWorkflowPEGASUS   import LocalWorkflowPEGASUS
from hubzero.submit.LocalBatchAppScript    import LocalBatchAppScript
from hubzero.submit.JobScanner             import JobScanner
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.TarCommand             import buildCreate as buildCreateTarCommand
from hubzero.submit.ServerConnection       import ServerConnection
from hubzero.submit.ClientIdAuthAttributes import ClientIdAuthAttributes

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"
EXITCODE             = ".__exit_code"

class SubmitClient:
   def __init__(self,
                pegasusVersion,
                pegasusHome):
      self.configData = {'listenURIs':[]
                        }

      self.clientIdAuthAttributes = ClientIdAuthAttributes()
      self.serverConnection       = None
      self.commandParser          = None
      self.localWorkflowPEGASUS   = None
      self.jobScanner             = None

      self.serverData                         = {}
      self.serverData['authz']                = 0
      self.serverData['session']              = 0
      self.serverData['jobId']                = 0
      self.serverData['event']                = ""
      self.serverData['pegasusRC']            = ""
      self.serverData['pegasusSites']         = ""

      self.clientData                        = {}
      self.clientData['pegasusVersion']      = pegasusVersion
      self.clientData['pegasusHome']         = pegasusHome
      self.clientData['showHelp']            = False
      self.clientData['reportMetrics']       = False
      self.clientData['isClientTTY']         = True
      self.clientData['readyToExit']         = False
      self.clientData['selectTimeout']       = 5*60.
      self.clientData['transferFiles']       = []
      self.clientData['inputObjects']        = {}
      self.clientData['outputObjects']       = {}
      self.clientData['isParametric']        = False
      self.clientData['localExecution']      = False
      self.clientData['childPid']            = None
      self.clientData['childReadStdout']     = None
      self.clientData['childReadStderr']     = None
      self.clientData['localStartTime']      = time.time()
      self.clientData['localFinishTime']     = None
      self.clientData['childHasExited']      = False
      self.clientData['exitCode']            = 2
      self.clientData['jobStates']           = {}
      self.clientData['jobScanPath']         = ""
      self.clientData['enteredCommand']      = ""
      self.clientData['startDate']           = None
      self.clientData['jobPath']             = os.getcwd()
      self.clientData['workDirectory']       = os.getcwd()
      self.clientData['workDirectoryInode']  = os.lstat(self.clientData['workDirectory']).st_ino
      self.clientData['importObjects']       = {}
      self.clientData['exportObjects']       = {}
      self.clientData['closePendingObjects'] = []
      self.clientData['filesToRemove']       = []
      self.clientData['emptyFilesToRemove']  = []

      signal.signal(signal.SIGINT,self.handleTerminationSignal)
      signal.signal(signal.SIGHUP,self.handleTerminationSignal)
      signal.signal(signal.SIGQUIT,self.handleTerminationSignal)
      signal.signal(signal.SIGABRT,self.handleTerminationSignal)
      signal.signal(signal.SIGTERM,self.handleTerminationSignal)
      signal.signal(signal.SIGCHLD,self.handleLocalCompleteSignal)


   def handleTerminationSignal(self,
                               signalNumber,
                               frame):
      log("submit client was terminated by a signal %d." % (signalNumber))
      if not self.clientData['readyToExit']:
         if self.serverConnection:
            self.serverConnection.postServerMessage('clientSignal %d\n' % (signalNumber))
         else:
            self.clientData['readyToExit'] = True
            exitCode = (1 << 7) + signalNumber
            self.exit(exitCode)


   def handleLocalCompleteSignal(self,
                                 signalNumber,
                                 frame):
      self.clientData['childHasExited'] = True


   def configure(self,
                 configFilePath):
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')

      fpConfig = open(configFilePath,'r')
      if fpConfig:
         eof = False
         while not eof:
            record = fpConfig.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.configData:
                     if   isinstance(self.configData[key],list):
                        self.configData[key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.configData[key],bool):
                        self.configData[key] = bool(value.lower() == 'true')
                     elif isinstance(self.configData[key],float):
                        self.configData[key] = float(value)
                     elif isinstance(self.configData[key],int):
                        self.configData[key] = int(value)
                     else:
                        self.configData[key] = value
                  else:
                     log("Undefined key = value pair %s = %s" % (key,value))
            else:
               eof = True
         fpConfig.close()

      configured = True
      if len(self.configData['listenURIs']) == 0:
         log("listenURIs missing from %s" % (configFilePath))
         configured = False

      return(configured)


   def haveIdAuthAttributes(self):
      return(self.clientIdAuthAttributes.haveIdAuthAttributes())


   def exit(self,
            exitCode=0):
      if self.clientData['readyToExit']:
         if not self.jobScanner:
            sys.exit(exitCode)
      else:
         if self.serverConnection:
            self.serverConnection.postServerMessage('clientExit %d\n' % (exitCode))
            self.clientData['readyToExit'] = True
         else:
            sys.exit(exitCode)


   def connect(self):
      isConnected = False
      if self.clientIdAuthAttributes.haveIdAuthAttributes():
         self.serverConnection = ServerConnection(self.configData['listenURIs'])
         isConnected = self.serverConnection.isConnected()

      return(isConnected)


   def sendContext(self):
      for arg in sys.argv[0:]:
         self.serverConnection.postServerMessageBySize('arg',arg)

      for environmentVar in os.environ:
         self.serverConnection.postServerMessagesBySize('var',[environmentVar,os.environ[environmentVar]])

      umask = os.umask(0)
      os.umask(umask)
      self.serverConnection.postServerMessage('umask 0%o\n' % (umask))
      self.serverConnection.postServerMessage('pwd %s %ld\n' % (self.clientData['workDirectory'],
                                                                self.clientData['workDirectoryInode']))
      self.clientData['isClientTTY'] = sys.stdin.isatty() and sys.stdout.isatty()
      self.serverConnection.postServerMessage('isClientTTY %d\n' % (self.clientData['isClientTTY']))
      self.serverConnection.postServerMessage('pegasusVersion %s\n' % (self.clientData['pegasusVersion']))


   def signon(self):
      signonAttributes = self.clientIdAuthAttributes.getSignonAttributes()
      self.serverConnection.postServerMessage('username %s\n' % (signonAttributes['username']))
      if 'password' in signonAttributes:
         self.serverConnection.postServerMessage('password %s\n' % (signonAttributes['password']))
      if 'token' in signonAttributes:
         self.serverConnection.postServerMessage('token %s\n' % (signonAttributes['token']))
      self.serverConnection.postServerMessage('signon\n')


   def getInputObjects(self):
      serverReader = self.serverConnection.getServerInputObject()
      childProcessInputObjects = []
      if self.clientData['childReadStdout']:
         childProcessInputObjects.append(self.clientData['childReadStdout'])
      if self.clientData['childReadStderr']:
         childProcessInputObjects.append(self.clientData['childReadStderr'])

      return(serverReader,
             self.clientData['inputObjects'].keys(),
             self.clientData['exportObjects'].keys(),
             childProcessInputObjects)


   def getOutputObjects(self):
      serverWriter = self.serverConnection.getServerOutputObject()

      return(serverWriter,
             self.clientData['importObjects'].keys())


   def readFile(self,
                fileObject):
      if   fileObject in self.clientData['inputObjects']:
         inputFile = self.clientData['inputObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.serverConnection.postServerMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.serverConnection.postServerMessagesBySize('read %s' % (inputFile),[text])
         except:
            self.exit(1)
      elif fileObject in self.clientData['exportObjects']:
         inputFile = self.clientData['exportObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.serverConnection.postServerMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.serverConnection.postServerMessagesBySize('read %s' % (inputFile),[text])
         except:
            self.exit(1)


   def writeFile(self,
                 fileObject):
      if fileObject in self.clientData['importObjects']:
         writeBuffer = self.clientData['importObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.clientData['importObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)


   def closeFile(self,
                 fileObject):
      if fileObject in self.clientData['inputObjects']:
         del self.clientData['inputObjects'][fileObject]

      if fileObject in self.clientData['exportObjects']:
         exportFile = self.clientData['exportObjects'][fileObject]
         exportPath = os.path.join(os.sep,"tmp",exportFile)
         if(os.path.exists(exportPath)):
            os.remove(exportPath)
         del self.clientData['exportObjects'][fileObject]

      if fileObject in self.clientData['outputObjects']:
         del self.clientData['outputObjects'][fileObject]

      transferTarFile = ""
      if fileObject in self.clientData['importObjects']:
         transferTarFile = self.clientData['importObjects'][fileObject]['path']
         del self.clientData['importObjects'][fileObject]
         if len(self.clientData['importObjects']) == 0:
            self.serverConnection.postServerMessage('importFilesComplete\n')

      if fileObject == self.clientData['childReadStdout']:
         self.clientData['childReadStdout'] = None
      if fileObject == self.clientData['childReadStderr']:
         self.clientData['childReadStderr'] = None

      try:
         fileObject.close()
         if transferTarFile:
            tarCommand = ['tar','xvf',os.path.join(self.clientData['workDirectory'],transferTarFile)]
            log("command: %s" % (tarCommand))
            child = subprocess.Popen(tarCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            tarStdOutput,tarStdError = child.communicate()
            tarExitStatus = child.returncode
            if tarExitStatus == 0:
               os.remove(os.path.join(self.clientData['workDirectory'],transferTarFile))
            else:
               log("Failed to extract transfer tarfile %s" % (transferTarFile))
               if tarStdOutput:
                  log(tarStdOutput)
               if tarStdError:
                  log(tarStdError)
      except:
         pass


   def checkClosePendingObjects(self):
      markedForDeletion = []
      for fileObject in self.clientData['closePendingObjects']:
         if fileObject in self.clientData['importObjects']:
            if self.clientData['importObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
      for fileObject in markedForDeletion:
         self.clientData['closePendingObjects'].remove(fileObject)
      del markedForDeletion


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      contactServer = False
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      showHelp = self.commandParser.getOption('help') and \
                 not self.commandParser.getOption('helpManagers') and \
                 not self.commandParser.getOption('helpVenues') and \
                 not self.commandParser.getOption('helpTools')
      if   showHelp:
         self.commandParser.showUsage()
      elif self.commandParser.validateArguments():
         self.clientData['showHelp'] = self.commandParser.getOption('help') or \
                                       self.commandParser.getOption('helpManagers') or \
                                       self.commandParser.getOption('helpVenues') or \
                                       self.commandParser.getOption('helpTools')
         if self.clientData['showHelp']:
            contactServer = True
         else:
            exitCode = self.commandParser.setSweepParameters()
            if exitCode == 0:
               exitCode = self.commandParser.setCSVDataParameters()
               if exitCode == 0:
                  parameterCombinationCount = self.commandParser.getParameterCombinationCount()
                  if parameterCombinationCount > 0:
                     self.clientData['isParametric'] = True

                  enteredCommand = self.commandParser.getEnteredCommand()
                  if enteredCommand == "":
                     sys.stderr.write("Command must be supplied\n")
                     sys.stderr.flush()
                  else:
                     enteredExecutable = self.commandParser.getEnteredExecutable()
                     if   self.clientData['isParametric'] and '@:' in enteredExecutable:
                        enteredExecutable = enteredExecutable.replace('@:','')
                        message = "Parameter substitution is not allowed within the command file: %s\n" % (enteredExecutable)
                        sys.stderr.write(message)
                     elif self.clientData['isParametric'] and '@@' in enteredExecutable:
                        message = "Parameter substitution is not allowed in the command name: %s\n" % (enteredExecutable)
                        sys.stderr.write(message)
                     elif self.clientData['isParametric'] and \
                          self.commandParser.getParametricArgumentCount() == 0 and \
                          self.commandParser.getParametricInputCount() == 0:
                        message = "Parameters supplied for %d jobs, " % (parameterCombinationCount) + \
                                  "but there are no template files or\ncommand substitutions that would make the jobs unique.  " + \
                                  "You should prefix any template files\nwith '@:' or " + \
                                  "embed parameter names (in the form '@@name') into command line arguments.\n"
                        sys.stderr.write(message)
                     else:
                        self.clientData['localExecution'] = self.commandParser.getOption('localExecution')
                        self.clientData['reportMetrics']  = self.commandParser.getOption('metrics')
                        contactServer = True

      return(contactServer)


   def reportDebugOutput(self):
      return(self.commandParser.getOption('debug'))


   def __updateWorkflowStatusSheet(self):
      jobStatusReportOrders = {}
      jobStatusReportOrders['waiting']    = 5
      jobStatusReportOrders['aborted']    = 1
      jobStatusReportOrders['setup']      = 6
      jobStatusReportOrders['setting up'] = 6
      jobStatusReportOrders['failed']     = 3
      jobStatusReportOrders['executing']  = 4
      jobStatusReportOrders['finished']   = 2

      parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
      parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
      if '.' in parameterCombinationsBase:
         parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
      tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
      tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
      copyTmpFile = False

      fpCSVIn = open(self.clientData['jobScanPath'],'rb')
      if fpCSVIn:
         csvReader = csv.reader(fpCSVIn)
         fpCSVOut = open(tmpParameterCombinationsPath,'wb')
         if fpCSVOut:
            csvWriter = csv.writer(fpCSVOut)
            csvWriter.writerow(('# command: ' + self.clientData['enteredCommand'],))
            csvWriter.writerow(('# started: ' + self.clientData['startDate'],))
            nCompleted = 0
            for instance in self.clientData['jobStates']:
               jobStatusState       = self.clientData['jobStates'][instance]
               jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
               if jobStatusState in ['finished','failed','aborted']:
                  nCompleted += 1
            csvWriter.writerow(('# completed: %d/%d jobs' % (nCompleted,len(self.clientData['jobStates'])),))
            parameterNames = []
            while len(parameterNames) <= 1:
               parameterNames = csvReader.next()
            csvWriter.writerow(parameterNames)
            parameterCombinations = {}
            for parameterCombination in csvReader:
               instance = int(parameterCombination[0])
               if instance in self.clientData['jobStates']:
                  jobStatusState       = self.clientData['jobStates'][instance]
                  jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
                  parameterCombination[1] = jobStatusState
                  if not jobStatusReportOrder in parameterCombinations:
                     parameterCombinations[jobStatusReportOrder] = []
                  parameterCombinations[jobStatusReportOrder].append(parameterCombination)
            jobStatusReportOrders = parameterCombinations.keys()
            jobStatusReportOrders.sort()
            for jobStatusReportOrder in jobStatusReportOrders:
               for parameterCombination in parameterCombinations[jobStatusReportOrder]:
                  csvWriter.writerow(parameterCombination)
            fpCSVOut.close()
            copyTmpFile = True
         fpCSVIn.close()

      if copyTmpFile:
         os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])


   @staticmethod
   def __which(program):
      def isExe(fpath):
         return(os.path.isfile(fpath) and os.access(fpath,os.X_OK))

      fpath,fname = os.path.split(program)
      if fpath:
         if isExe(program):
            return(program)
      else:
         for path in os.environ["PATH"].split(os.pathsep):
            exePath = os.path.join(path,program)
            if isExe(exePath):
               return(exePath)

      return(None)


   def startLocalParametric(self):
      exitCode = 0
      useEnvironment = ""
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = filter(reChoice.search,os.environ.keys())
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         useEnvironment = ". /etc/environ.sh\n"
         if "ENVIRON_CONFIG_DIRS" in os.environ:
            useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
         for environmentVar in environmentVars:
            useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

      environment = ""
      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               log("Invalid environment variable %s specified." % (environmentVariableValue))
               if not exitCode:
                  exitCode = 1

            if   environmentVariable == "IGNORESTATECHECK":
               pass
            elif environmentVariable == "IGNOREPROBECHECK":
               pass
            elif environmentVariable == "X509_SUBMIT_PROXY":
               pass
            elif environmentVariable == "USER_LOG_PATH":
               pass
            elif environmentVariable == "USER_JOBID_PATH":
               pass
            else:
               environment += environmentVariable + "=" + value + " "
         environment = environment.strip()

      localJobId = "%08d" % (self.serverData['jobId'])
      clientInputFiles = []
      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            clientInputFiles.append(inputFile.replace(' ','\ '))

      self.clientData['enteredCommand'] = self.commandParser.getEnteredCommand()
      enteredCommandArguments = self.commandParser.getEnteredCommandArguments()

      self.clientData['jobPath'] = os.path.join(os.getcwd(),localJobId)
      if not os.path.isdir(self.clientData['jobPath']):
         os.mkdir(self.clientData['jobPath'])
      self.clientData['jobScanPath'] = os.path.join(self.clientData['jobPath'],'parameterCombinations.csv')
      self.commandParser.writeParameterCombinations(self.clientData['jobScanPath'])

      inputError = False
      executableExistsError = False
      nInstanceIdDigits = max(2,int(math.log10(self.commandParser.getParameterCombinationCount())+1))
      instance = 0
      for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.clientData['jobScanPath']):
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)
         self.clientData['jobStates'][instance] = 'setup'

         instanceInputsPath = os.path.join(self.clientData['jobPath'],instanceId)
         if not os.path.isdir(instanceInputsPath):
            os.makedirs(instanceInputsPath)

         exitCode = 0

         enteredCommand = self.commandParser.getEnteredCommand()
         template = ParameterTemplate(enteredCommand)
         try:
            enteredCommand = template.substitute(substitutions)
         except KeyError,err:
            sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
            inputError = True

         inputFiles = []
         for inputFile in clientInputFiles:
            template = ParameterTemplate(inputFile)
            try:
               inputFile = template.substitute(substitutions)
            except KeyError,err:
               sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
               inputError = True
            inputFiles.append(inputFile)

         if len(enteredCommandArguments) > 1:
            for arg in enteredCommandArguments[1:]:
               if not arg.startswith('-'):
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                     inputError = True
                  if os.path.isfile(arg.replace('\ ',' ').replace('@:','')):
                     if not arg in inputFiles:
                        inputFiles.append(arg)

         substitutedInputFiles = []
         for inputFile in inputFiles:
            if '@:' in inputFile:
               inputFile = inputFile.replace('@:','')
               fpInputFile = open(inputFile,'r')
               if fpInputFile:
                  inputText = fpInputFile.readlines()
                  fpInputFile.close()
                  template = ParameterTemplate(''.join(inputText))
                  try:
                     inputText = template.substitute(substitutions)
                  except KeyError,err:
                     sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                     inputError = True
                  inputName = os.path.basename(inputFile)
                  inputPath = os.path.join(instanceInputsPath,inputName)
                  fpInputPath = open(inputPath,'w')
                  if fpInputPath:
                     fpInputPath.writelines(inputText)
                     fpInputPath.close()
                     substitutedInputFiles.append(inputPath)
            else:
               substitutedInputFiles.append(inputFile)
         del inputFiles
         inputFiles = substitutedInputFiles

         remoteArgs = []

         executable = self.__which(enteredCommandArguments[0])
         if executable:
            executable = os.path.realpath(executable)
            remoteArgs.append(executable)
         else:
            if not executableExistsError:
               sys.stderr.write("Specified command %s is not in your PATH\n" % (enteredCommandArguments[0]))
            inputError = True
            # allow completion of input checks
            remoteArgs.append(enteredCommandArguments[0])
            executable = enteredCommandArguments[0]
            executableExistsError = True

         for arg in enteredCommandArguments[1:]:
            if arg.startswith('-'):
               remoteArgs.append(arg)
            else:
               template = ParameterTemplate(arg)
               try:
                  arg = template.substitute(substitutions)
               except KeyError,err:
                  sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                  inputError = True
               if '@:' in arg:
                  if os.path.isfile(arg.replace('\ ',' ').replace('@:','')):
                     inputName = os.path.basename(arg.replace('@:',''))
                     remoteArgs.append(inputName)
               else:
                  if os.path.isfile(arg.replace('\ ',' ')):
                     rarg = os.path.abspath(arg)
                     remoteArgs.append(rarg)
                  else:
                     remoteArgs.append(arg)

         remoteCommand = " ".join(remoteArgs)
         arguments     = " ".join(remoteArgs[1:])
         arguments     = arguments.strip()

         timeHistoryLogs = {}
         timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,localJobId,instanceId)
         timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,localJobId,instanceId)
         timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,localJobId,instanceId)
         timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,localJobId,instanceId)
         timeHistoryLogs['exitCode']             = "%s.%s_%s" % (EXITCODE,localJobId,instanceId)

         localBatchAppScript = LocalBatchAppScript(localJobId,instanceId,
                                                   executable,arguments,useEnvironment,environment,
                                                   timeHistoryLogs)
         appScriptName,appScript = localBatchAppScript.buildAppScript()
         appScriptPath = os.path.join(self.clientData['jobPath'],instanceId,appScriptName)

         fpAppScript = open(appScriptPath,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            self.clientData['filesToRemove'].append(appScriptPath)
         else:
            log("could not open %s for writing" % (appScriptPath))
            exitCode = 1
         del localBatchAppScript

         stdFile = os.path.join(self.clientData['jobPath'],instanceId,"%s_%s.stderr" % (localJobId,instanceId))
         self.clientData['emptyFilesToRemove'].append(stdFile)

      nInstances      = instance
      executeInstance = nInstances+1

      timeHistoryLogs = {}
      timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,localJobId,"0")
      timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,localJobId,"0")
      timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,localJobId,"0")
      timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,localJobId,"0")

      pegasusTemplates = {}
      if self.serverData['pegasusRC'] != "":
         pegasusTemplates['rc'] = self.serverData['pegasusRC']
      if self.serverData['pegasusSites'] != "":
         pegasusTemplates['sites'] = self.serverData['pegasusSites']

      self.localWorkflowPEGASUS = LocalWorkflowPEGASUS(self.clientData['pegasusVersion'],self.clientData['pegasusHome'],
                                                       localJobId,pegasusTemplates,timeHistoryLogs)
      batchScriptName,batchScript = self.localWorkflowPEGASUS.buildWorkflowScript()
      batchScriptPath = os.path.join(self.clientData['jobPath'],batchScriptName)

      fpBatchScript = open(batchScriptPath,'w')
      if fpBatchScript:
         fpBatchScript.write(batchScript)
         fpBatchScript.close()
         os.chmod(batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         self.clientData['filesToRemove'].append(batchScriptPath)
      else:
         log("could not open %s for writing" % (batchScriptPath))
         exitCode = 1

      enteredCommand = self.commandParser.getEnteredCommand()

      stdFile = "%s.stderr" % (localJobId)
      self.clientData['emptyFilesToRemove'].append(stdFile)

      self.clientData['startDate']      = time.strftime("%a %b %d %X %Z %Y")
      self.clientData['localStartTime'] = time.time()
      self.__updateWorkflowStatusSheet()

      if inputError:
         self.exit(1)
      else:
         child = subprocess.Popen(batchScriptPath,shell=False,bufsize=1,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr

         self.serverConnection.postServerMessage('localWait\n')

         if not sys.stdin in self.clientData['inputObjects']:
            self.clientData['inputObjects'][sys.stdin] = '#STDIN#'

         self.clientData['isClientTTY'] = sys.stdin.isatty() and sys.stdout.isatty()
         if self.clientData['isClientTTY']:
            self.jobScanner = JobScanner(self.clientData['jobScanPath'])
            self.jobScanner.start()


   def startLocalExecute(self):
      self.clientData['localStartTime'] = time.time()
      commandArguments = self.commandParser.getEnteredCommandArguments()
      os.environ['SUBMIT_JOB'] = "%s" % (self.serverData['jobId'])
      try:
         child = subprocess.Popen(commandArguments,shell=False,bufsize=1024,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  env=os.environ,
                                  close_fds=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr
         self.serverConnection.postServerMessage('localWait\n')
      except OSError,err:
         sys.stderr.write("Failed to invoke %s: %s\n" % (commandArguments[0],err.args[1]))
         self.exit(err.args[0])


   def startLocal(self):
      if self.clientData['isParametric']:
         self.startLocalParametric()
      else:
         self.startLocalExecute()


   def finishLocalParametric(self):
      localJobId = "%08d" % (self.serverData['jobId'])
      parameterCombinationCount = self.commandParser.getParameterCombinationCount()
      if parameterCombinationCount > 0:
         nInstanceIdDigits = max(2,int(math.log10(parameterCombinationCount)+1))
         for instance in xrange(1,parameterCombinationCount+1):
            instanceId = str(instance).zfill(nInstanceIdDigits)

            jobStatistic = JobStatistic(1)
            timeFile = "%s.%s_%s" % (TIMESTAMPSTART,localJobId,instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobStartedTime',timePath)
            timeFile = "%s.%s_%s" % (TIMESTAMPFINISH,localJobId,instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobFinshedTime',timePath)
            timerFile = "%s.%s_%s" % (TIMERESULTS,localJobId,instanceId)
            timerPath = os.path.join(self.clientData['jobPath'],instanceId,timerFile)
            jobStatistic.recordTimer(timerPath)
            jobStatistic.setWaitingTime()
            jobStatistic.setElapsedRunTime()

            exitCodeFile = "%s.%s_%s" % (EXITCODE,localJobId,instanceId)
            exitCodePath = os.path.join(self.clientData['jobPath'],instanceId,exitCodeFile)
            if(os.path.exists(exitCodePath)):
               os.remove(exitCodePath)

            exitCode       = jobStatistic['exitCode']
            cpuTime        = jobStatistic['userTime']+jobStatistic['sysTime']
            elapsedRunTime = jobStatistic['elapsedRunTime']

            workDirectoryName = os.path.join(self.clientData['jobPath'],'work')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            workDirectoryName = os.path.join(self.clientData['jobPath'],'scratch')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)

            self.serverConnection.postServerMessage('localMetrics %d %d %f %f\n' % (instance,exitCode,cpuTime,elapsedRunTime))
            del jobStatistic

         if not self.clientData['isClientTTY']:
            sys.stdout.write("Simulations complete. Results are stored in directory %s\n" % (self.clientData['jobPath']))


   def finishLocal(self):
      if self.clientData['isParametric']:
         self.finishLocalParametric()

      for fileToRemove in self.clientData['filesToRemove']:
         if(os.path.exists(fileToRemove)):
            os.remove(fileToRemove)

      for fileToRemove in self.clientData['emptyFilesToRemove']:
         if(os.path.exists(fileToRemove)):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)


   def scanLocalOutputFile(self,
                           outputFileObject):
      outChunk = os.read(outputFileObject.fileno(),1024)
      if len(outChunk) > 0:
         if self.clientData['isParametric']:
            update = False
            jobStartPattern    = re.compile('.* Executing JOB ([^ ]*) *-n *([^ ]*) *-N *([^ ]*).*/([^ ]*)')
            jobCompletePattern = re.compile('.* JOB *[0-9]*_[0-9]*\.sh_([0-9]*_[0-9]*)')
            for record in outChunk.split('\n'):
               matchObj = jobStartPattern.match(record)
               if matchObj:
                  if matchObj.group(2) == matchObj.group(4):
                     instance = int(matchObj.group(3).split('_')[-1])
                     self.clientData['jobStates'][instance] = 'executing'
                     update = True
               else:
                  matchObj = jobCompletePattern.match(record)
                  if matchObj:
                     instance = int(matchObj.group(1).split('_')[-1])
                     instanceExitCode = 0
                     exitCodeFile = "%s.%s" % (EXITCODE,matchObj.group(1))
                     exitCodePath = os.path.join(self.clientData['jobPath'],matchObj.group(1).split('_')[-1],exitCodeFile)
                     if(os.path.exists(exitCodePath)):
                        fpExitCode = open(exitCodePath,'r')
                        if fpExitCode:
                           instanceExitCode = int(fpExitCode.readline())
                           fpExitCode.close()
                     if instanceExitCode:
                        self.clientData['jobStates'][instance] = 'failed'
                     else:
                        self.clientData['jobStates'][instance] = 'finished'
                     update = True
            if update:
               self.__updateWorkflowStatusSheet()
         else:
            if outputFileObject == self.clientData['childReadStdout']:
               sys.stdout.write(outChunk)
               sys.stdout.flush()
            if outputFileObject == self.clientData['childReadStderr']:
               sys.stderr.write(outChunk)
               sys.stderr.flush()
      else:
         self.closeFile(outputFileObject)


   def mapInputFiles(self):
      if not self.clientData['showHelp']:
         inputFiles = {}
         enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
         if self.clientData['isParametric']:
            clientInputFiles = {}
            if self.commandParser.getOption('inputFiles'):
               for inputFile in self.commandParser.getOption('inputFiles'):
                  clientInputFiles[inputFile] = inputFile.replace(' ','\ ')

            for substitutions in self.commandParser.getNextParameterCombination():
               for inputFile in clientInputFiles:
                  template = ParameterTemplate(clientInputFiles[inputFile])
                  actualInputFile = template.substitute(substitutions).replace('@:','')
                  if not actualInputFile in inputFiles:
                     inputFiles[actualInputFile] = inputFile

               if len(enteredCommandArguments) > 0:
                  for arg in enteredCommandArguments:
                     if not arg.startswith('-'):
                        template = ParameterTemplate(arg)
                        actualArg = template.substitute(substitutions).replace('\ ',' ').replace('@:','')
                        if os.path.isfile(actualArg) or os.path.isdir(actualArg):
                           if not actualArg in inputFiles:
                              inputFiles[actualArg] = arg
         else:
            if self.commandParser.getOption('inputFiles'):
               for inputFile in self.commandParser.getOption('inputFiles'):
                  inputFiles[inputFile] = inputFile.replace(' ','\ ')

            if len(enteredCommandArguments) > 0:
               for arg in enteredCommandArguments:
                  if not arg.startswith('-'):
                     actualArg = arg.replace('\ ',' ')
                     if os.path.isfile(actualArg) or os.path.isdir(actualArg):
                        if not actualArg in inputFiles:
                           inputFiles[actualArg] = arg

         for inputFile in inputFiles:
            try:
               inode = os.lstat(inputFile).st_ino
               self.serverConnection.postServerMessagesBySize('inputFile %ld' % (inode),[inputFile,inputFiles[inputFile]])
            except OSError,err:
               sys.stderr.write("%s: %s\n" % (inputFile,err.args[1]))
               self.exit(err.args[0])

      self.serverConnection.postServerMessage('inputFileInodesSent\n')


   def wait(self):
      if self.serverConnection.isConnected():
         self.serverConnection.postServerMessage('exit\n')


   def processServerRequests(self):
      message = self.serverConnection.pullServerMessage(0)
      while message:
         args = message.split()
         log("server request = %s" % (args[0]))

         if   args[0] == 'session':
            self.serverData['session'] = args[1]
         elif args[0] == 'message':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  log(messageText)
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'authz':
            self.serverData['authz'] = int(args[1])
            if self.serverData['authz'] == 0:
               log("Session authentication failed")
               self.clientIdAuthAttributes.clearSessionToken()
               self.clientIdAuthAttributes.getUserAttributes()
               if self.clientIdAuthAttributes.haveIdAuthAttributes():
                  self.signon()
            else:
               if self.clientData['localExecution']:
                  self.serverConnection.postServerMessage('startLocal\n')
               else:
                  self.serverConnection.postServerMessage('setupRemote\n')
         elif args[0] == 'writeStdout':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  sys.stdout.write(messageText)
                  sys.stdout.flush()
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'writeStderr':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  sys.stderr.write(messageText)
                  sys.stderr.flush()
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'write':
            serverOutputFile  = args[1]
            textLength = int(args[2])
            if textLength > 0:
               text = self.serverConnection.pullServerMessage(textLength)
               if len(text) > 0:
                  found = False
                  if not found:
                     for importObject in self.clientData['importObjects']:
                        if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                           self.clientData['importObjects'][importObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     log("File object not found to match %s" % (serverOutputFile))
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'close':
            serverOutputFile  = args[1]
            found = False
            for importObject in self.clientData['importObjects']:
               if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                  self.clientData['closePendingObjects'].append(importObject)
                  found = True
                  break
            if not found:
               log("File object not found to match %s" % (serverOutputFile))
         elif args[0] == 'jobId':
            self.serverData['jobId'] = int(args[1])
            if self.serverData['jobId'] == 0:
               sys.exit(1)
            if self.clientData['reportMetrics']:
               sys.stdout.write("=SUBMIT-METRICS=> job=%d\n" % (self.serverData['jobId']))
         elif args[0] == 'event':
            self.serverData['event'] = args[1]
         elif args[0] == 'pegasusRC':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  self.serverData['pegasusRC'] = messageText
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'pegasusSites':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  self.serverData['pegasusSites'] = messageText
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'jobScanUpdate':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullServerMessage(messageLength)
               if len(messageText) > 0:
                  if self.clientData['jobScanPath'] == "":
                     jobScanPath = os.path.join(self.clientData['workDirectory'],
                                                "%08d" % (self.serverData['jobId']),
                                                "parameterCombinations.csv")
                     self.clientData['jobScanPath'] = jobScanPath
                  try:
                     parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
                     parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
                     if '.' in parameterCombinationsBase:
                        parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
                     tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
                     tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
                     fpJobScan = open(tmpParameterCombinationsPath,'w')
                     if fpJobScan:
                        fpJobScan.write(messageText)
                        fpJobScan.close()
                        os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])
                  except:
                     pass
                  if os.path.exists(self.clientData['jobScanPath']):
                     if not self.jobScanner:
                        self.jobScanner = JobScanner(self.clientData['jobScanPath'])
                        self.jobScanner.start()
               else:
                  self.serverConnection.pushServerMessage(message + '\n')
                  break
         elif args[0] == 'serverReadyForIO':
            if self.clientData['localExecution']:
               self.startLocal()
            else:
               self.clientData['inputObjects'][sys.stdin] = '#STDIN#'
         elif args[0] == 'serverReadyForInputMapping':
            self.mapInputFiles()
         elif args[0] == 'addExportFile':
            clientInputFile = args[1]
            self.clientData['transferFiles'].append(clientInputFile)
         elif args[0] == 'exportFiles':
            transferTarFile = args[1]
            transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
            tarCommand = buildCreateTarCommand(transferTarPath,self.clientData['transferFiles'])
            log("command = " + tarCommand)
            try:
               child = subprocess.Popen(tarCommand,shell=True,
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        close_fds=True)
               tarStdOutput,tarStdError = child.communicate()
               tarExitStatus = child.returncode
               if tarExitStatus != 0:
                  log("Failed to write transfer tarfile %s" % (transferTarPath))
                  if tarStdOutput:
                     log(tarStdOutput)
                  if tarStdError:
                     log(tarStdError)
                  exitCode = 1
               else:
                  fp = open(transferTarPath,'r')
                  if fp:
                     self.clientData['exportObjects'][fp] = transferTarFile
                  else:
                     log("input file open failed: %s" % (transferTarPath))
            except OSError,err:
               log("Failed to to create transfer tarfile %s" % (transferTarPath))
               log(err.args[1])
         elif args[0] == 'exportFilesComplete':
            self.serverConnection.postServerMessage('startRemote\n')
         elif args[0] == 'importFile':
            transferTarFile = args[1]
            if transferTarFile == "None":
               self.serverConnection.postServerMessage('importFilesComplete\n')
            else:
               transferTarPath = os.path.join(self.clientData['workDirectory'],transferTarFile)
               try:
                  fp = open(transferTarPath,'w')
                  if fp:
                     self.clientData['importObjects'][fp]           = {}
                     self.clientData['importObjects'][fp]['path']   = transferTarFile
                     self.clientData['importObjects'][fp]['buffer'] = ""
                  else:
                     log("input file open failed: %s" % (transferTarPath))
               except OSError,err:
                  log("Failed to to create transfer tarfile %s" % (transferTarPath))
                  log(err.args[1])
         elif args[0] == 'serverExit':
            exitCode = int(args[1])
            self.clientData['readyToExit'] = True
            self.exit(exitCode)
         elif args[0] == 'exit':
            exitCode = int(args[1])
            self.exit(exitCode)
         elif args[0] == 'wait':
            self.wait()
         else:
            log("Discarded server message: %s" % (message))

         message = self.serverConnection.pullServerMessage(0)


   def run(self):
      while True:
         try:
            serverReader,inputObjects,exportObjects,childProcessInputObjects = self.getInputObjects()
            serverWriter,importObjects                                       = self.getOutputObjects()
            if self.clientData['childHasExited'] or self.clientData['readyToExit']:
               selectTimeout = 0.1
            elif self.jobScanner:
               selectTimeout = 10.
            else:
               selectTimeout = self.clientData['selectTimeout']
            readyReaders,readyWriters,readyExceptions = select.select(serverReader+inputObjects+exportObjects+childProcessInputObjects,
                                                                      serverWriter+importObjects,[],
                                                                      selectTimeout)
         except select.error,err:
            readyReaders = []
            readyWriters = []
#
# If the timeout has occurred (nothing to read/write) send a keepalive.
#
         if readyReaders == [] and readyWriters == []:
            self.serverConnection.postServerMessage('null\n')
            if   self.jobScanner:
               if self.jobScanner.processInput():
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],"%08d" % (self.serverData['jobId']))
                  sys.stdout.write("Simulations complete. Results are stored in directory %s\n" % (resultsPath))
            elif self.clientData['readyToExit']:
               self.exit()

         for readyReader in readyReaders:
            if   readyReader in serverReader:
               self.serverConnection.receiveServerMessage()
            elif readyReader in childProcessInputObjects:
               self.scanLocalOutputFile(readyReader)
            elif self.jobScanner and readyReader == sys.stdin:
               if self.jobScanner.processInput():
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],"%08d" % (self.serverData['jobId']))
                  sys.stdout.write("Simulations complete. Results are stored in directory %s\n" % (resultsPath))
                  os.kill(os.getpid(),signal.SIGINT)
            else:
               self.readFile(readyReader)

         self.processServerRequests()

         for readyWriter in readyWriters:
            if   readyWriter in serverWriter:
               self.serverConnection.sendServerMessage()
            elif readyWriter in importObjects:
               self.writeFile(readyWriter)
         self.checkClosePendingObjects()

         if self.clientData['childPid']:
            try:
               pid,exitCode = os.waitpid(self.clientData['childPid'],os.WNOHANG)
            except:
               pid = 0

            if pid != 0:
               exitCode = exitCode >> 8
               self.clientData['childPid'] = None
               self.clientData['exitCode'] = exitCode
               rusage = resource.getrusage(resource.RUSAGE_CHILDREN)
               self.clientData['localFinishTime'] = time.time()
               realTime = self.clientData['localFinishTime']-self.clientData['localStartTime']
               cpuTime  = rusage[0] + rusage[1]

               if self.clientData['reportMetrics']:
                  message =  "=SUBMIT-METRICS=>"
                  message += " job=%d" % (self.serverData['jobId'])
                  message += " venue=local"
                  message += " status=%d" % (self.clientData['exitCode'])
                  message += " cputime=%f" % (cpuTime)
                  message += " realtime=%f" % (realTime)
                  sys.stdout.write("%s\n" % (message))

               self.connect()
               self.serverConnection.postServerMessage('pwd %s %ld\n' % (self.clientData['workDirectory'],
                                                                         self.clientData['workDirectoryInode']))
               self.serverConnection.postServerMessage('jobId %d\n' % (self.serverData['jobId']))
               self.signon()

               try:
                  submitApplicationRevision = os.environ["SUBMIT_APPLICATION_REVISION"]
               except:
                  submitApplicationRevision = ""
               if submitApplicationRevision != "":
                  self.serverConnection.postServerMessage('event %s\n' % (submitApplicationRevision))
               else:
                  self.serverConnection.postServerMessage('event %s\n' % (self.serverData['event']))

               self.finishLocal()

               self.serverConnection.postServerMessage('localExit %d %f %f\n' % (exitCode,cpuTime,realTime))
               self.clientData['readyToExit'] = True


