# @package      hubzero-submit-common
# @file         LocalBatchFACTORY.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import re

class LocalBatchFACTORY:
   def __init__(self,
                hubUserName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                environment,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                wallTime):
      self.hubUserName        = hubUserName
      self.localJobId         = localJobId
      self.instanceId         = instanceId
      self.instanceDirectory     = instanceDirectory
      self.stageFiles         = siteInfo['stageFiles'] and not siteInfo['sharedUserSpace']
      self.appScriptName      = appScriptName
      self.environment        = environment
      self.isMultiCoreRequest = isMultiCoreRequest
      self.computationMode    = managerInfo['computationMode']
      self.wallTime           = str(int(wallTime))

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeSerialNotStagedTemplate(self):
      return """# LocalBatchFACTORY:makeSerialNotStagedTemplate
universe = vanilla
executable = INSTANCEDIRECTORY/APPSCRIPTNAME
transfer_executable = true
transfer_input_files = JOBID_input.tar.gz
transfer_output_files = JOBID_output.tar.gz
environment = "ENVIRONMENT"
output = INSTANCEDIRECTORY/factory_JOBID.stdout
error = INSTANCEDIRECTORY/factory_JOBID.stderr
log = INSTANCEDIRECTORY/factory_JOBID.log
should_transfer_files = YES
when_to_transfer_output = ON_EXIT
notification = never
+FactoryJob = true
x509userproxy = /tmp/hub-proxy.USER
+FactoryRefreshProxy = true
periodic_release = (CurrentTime > (EnteredCurrentStatus + 300)) && ((HoldReasonCode > 5) && (HoldReasonCode < 15))
# (Job is Held AND Error && # Holds >= 3)
periodic_remove = (JobStatus == 5) && ((HoldReasonCode > 5) && (HoldReasonCode < 15)) && (NumShadowStarts >= 3)
queue
"""


   def __makeSerialStagedTemplate(self):
      return """# LocalBatchFACTORY:makeSerialStagedTemplate
universe = vanilla
executable = APPSCRIPTNAME
transfer_executable = false
environment = "ENVIRONMENT"
output = factory_JOBID.stdout
error = factory_JOBID.stderr
log = factory_JOBID.log
notification = never
+FactoryJob = true
x509userproxy = /tmp/hub-proxy.USER
+FactoryRefreshProxy = true
periodic_release = (CurrentTime > (EnteredCurrentStatus + 300)) && ((HoldReasonCode > 5) && (HoldReasonCode < 15))
# (Job is Held AND Error && # Holds >= 3)
periodic_remove = (JobStatus == 5) && ((HoldReasonCode > 5) && (HoldReasonCode < 15)) && (NumShadowStarts >= 3)
queue
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_user              = re.compile("USER")
      re_jobid             = re.compile("JOBID")
      re_appScriptName     = re.compile("APPSCRIPTNAME")
      re_walltime          = re.compile("WALLTIME")
      re_environment       = re.compile("ENVIRONMENT")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")

      if self.stageFiles:
         template = self.__makeSerialStagedTemplate()
      else:
         template = self.__makeSerialNotStagedTemplate()

      template = re_user.sub(self.hubUserName,template)
      template = re_jobid.sub("%s_%s" % (self.localJobId,self.instanceId),template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_walltime.sub(self.wallTime,template)
      template = re_environment.sub(self.environment,template)
      template = re_instanceDirectory.sub(self.instanceDirectory,template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = ""
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
      else:
         batchLogName = "factory_%s_%s.log" % (self.localJobId,self.instanceId)
         batchScript = self.__buildSerialFile()
         batchScriptName = "%s_%s.factory" % (self.localJobId,self.instanceId)

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


