# @package      hubzero-submit-common
# @file         JobDistributor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import sys
import os
import re
import signal
import select
import time
import subprocess
import random
import traceback
import math
import copy
import shutil

from hubzero.submit.LogMessage               import openLog, logJobId as log, logSetJobId
from hubzero.submit.SitesInfo                import SitesInfo
from hubzero.submit.ToolsInfo                import ToolsInfo
from hubzero.submit.ManagersInfo             import ManagersInfo
from hubzero.submit.TunnelsInfo              import TunnelsInfo
from hubzero.submit.EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from hubzero.submit.CommandParser            import CommandParser
from hubzero.submit.VenueMechanismLocal      import VenueMechanismLocal
from hubzero.submit.VenueMechanismSsh        import VenueMechanismSsh
from hubzero.submit.VenueMechanismGsiSsh     import VenueMechanismGsiSsh
from hubzero.submit.RemoteJobMonitor         import RemoteJobMonitor
from hubzero.submit.RemoteProbeMonitor       import RemoteProbeMonitor
from hubzero.submit.RemoteTunnelMonitor      import RemoteTunnelMonitor
from hubzero.submit.TimeConversion           import hhmmssTomin
from hubzero.submit.GroupMembership          import isSubmitMember
from hubzero.submit.TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand
from hubzero.submit.JobStatistic             import JobStatistic
from hubzero.submit.ParameterTemplate        import ParameterTemplate

HUBLOGFILE               = "distributor.log"
HUBGRIDJOBIDLOGFILE      = "gridjobid.log"
HUBGRIDJOBHISTORYLOGFILE = "gridjobhistory.log"

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

LOCALJOBID = ".__local_jobid"

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"

JOBGRIDRESOURCE = ".__grid_resource"
JOBGRIDHISTORY  = ".__grid_history"
JOBGRIDJOBID    = ".__grid_jobid"

class JobDistributor:
   def __init__(self,
                distributorVersion,
                installedDirectory,
                logDirectory,
                jobMonitorHost,
                jobMonitorPort,
                probeMonitorHost,
                probeMonitorPort,
                tunnelMonitorHost,
                tunnelMonitorPort,
                cloudMonitorHost,
                cloudMonitorPort,
                allowedVenueMechanisms):

      self.jobId                    = 0
      self.jobs                     = {}
      self.jobStatistics            = {}
      self.fpUserLogPath            = None
      self.abortAttempted           = False
      self.disableStateCheck        = False
      if distributorVersion == 'dev':
         self.disableStateCheck     = True
      self.disableProbeCheck        = True
      self.quotaLimit               = -1
      self.childPid                 = 0
      self.bufferSize               = 4096
      self.hubLogPath               = os.path.join(logDirectory,HUBLOGFILE)
      self.hubGridJobIdLogPath      = os.path.join(logDirectory,HUBGRIDJOBIDLOGFILE)
      self.hubGridJobHistoryLogPath = os.path.join(logDirectory,HUBGRIDJOBHISTORYLOGFILE)
      self.dataDirectory            = installedDirectory
      self.binDirectory             = os.path.join(installedDirectory,'bin')
      self.distributorVersion       = distributorVersion
      self.jobMonitorHost           = jobMonitorHost
      self.jobMonitorPort           = jobMonitorPort
      self.probeMonitorHost         = probeMonitorHost
      self.probeMonitorPort         = probeMonitorPort
      self.tunnelMonitorHost        = tunnelMonitorHost
      self.tunnelMonitorPort        = tunnelMonitorPort
      self.cloudMonitorHost         = cloudMonitorHost
      self.cloudMonitorPort         = cloudMonitorPort
      self.allowedVenueMechanisms   = allowedVenueMechanisms
      self.runjobExists             = False

      self.batchCommands = {}
      self.batchCommands['receiveInput']    = RECEIVEINPUTCOMMAND
      self.batchCommands['submitBatchJob']  = SUBMITBATCHJOBCOMMAND
      self.batchCommands['transmitResults'] = TRANSMITRESULTSCOMMAND
      self.batchCommands['cleanupJob']      = CLEANUPJOBCOMMAND
      self.batchCommands['killBatchJob']    = KILLBATCHJOBCOMMAND

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.waitForJobsInfo = {}

      self.remoteMonitors = {}
      self.remoteMonitors['job']    = None
      self.remoteMonitors['probe']  = None
      self.remoteMonitors['tunnel'] = None
      self.remoteMonitors['cloud']  = None

      self.sitesInfo                = None
      self.cloudsInfo               = None
      self.tunnelsInfo              = None
      self.toolsInfo                = None
      self.managersInfo             = None
      self.environmentWhitelistInfo = None

      self.successfulInstance    = None
      self.maximumSelectedSites  = 3
      self.userDestinations      = []
      self.inputFiles            = []
      self.outputFiles           = []
      self.isMultiCoreRequest    = False
      self.nCpus                 = 1
      self.ppn                   = ""
      self.wallTime              = 60
      self.environment           = ""
      self.managerSpecified      = False
      self.managerInfo           = {}
      self.x509SubmitProxy       = ""
      self.stdinput              = ""
      self.isParametric          = False
      self.parameterNames        = []
      self.parameterCombinations = None

      self.commandParser           = None
      self.enteredCommandArguments = []
      self.showHelp                = False

      self.submitVenues   = None
      self.localJobId     = None
      self.hubUserId      = None
      self.hubUserName    = None
      self.isClientTTY    = True
      self.pegasusVersion = None
      self.pegasusHome    = None

      openLog(self.hubLogPath)

      self.getUserQuota()

      self.abortGlobal = {}
      self.abortGlobal['abortAttempted'] = self.abortAttempted
      self.abortGlobal['abortSignal']    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def setEnvironment(self,
                      pbsRoot,
                      condorRoot,
                      condorConfig,
                      runjobRoot):
      if pbsRoot != "" and pbsRoot != "REMOTE":
         os.environ['PATH'] = os.path.join(pbsRoot,'bin') + ':' + os.environ['PATH']
      if condorRoot != "" and condorRoot != "REMOTE":
         os.environ['PATH'] = os.path.join(condorRoot,'bin') + ':' + \
                              os.path.join(condorRoot,'sbin') + ':' + os.environ['PATH']
         if condorConfig != "":
            os.environ['CONDOR_CONFIG'] = condorConfig
      if runjobRoot != "":
         if runjobRoot == "REMOTE":
            self.runjobExists = True
         else:
            runjobScriptPath = os.path.join(runjobRoot,'bin','runjob.sh')
            if(os.path.exists(runjobScriptPath)):
               os.environ['PATH'] = os.path.join(runjobRoot,'bin') + ':' + os.environ['PATH']
               self.runjobExists = True

      os.environ['PATH'] = self.binDirectory + ':' + os.environ['PATH']


   def getEnvironment(self):
      self.hubUserName    = os.getenv("USER")
      self.hubUserId      = os.getuid()
      self.submitVenues   = os.getenv("SUBMITVENUES")
      self.isClientTTY    = bool(int(os.getenv("SUBMIT_ISCLIENTTTY",'0')))
      self.pegasusVersion = os.environ["PEGASUS_VERSION"]
      self.pegasusHome    = os.environ["PEGASUS_HOME"]


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      exitCode = 0
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.showHelp = self.commandParser.getOption('help') or \
                      self.commandParser.getOption('helpManagers') or \
                      self.commandParser.getOption('helpVenues') or \
                      self.commandParser.getOption('helpTools')
      if not self.showHelp:
         if not self.commandParser.getOption('quotaCheck'):
            self.quotaLimit = -1

         exitCode = self.commandParser.setSweepParameters()
         if exitCode == 0:
            exitCode = self.commandParser.setCSVDataParameters()
            if exitCode == 0:
               if self.commandParser.getParameterCombinationCount() > 0:
                  self.isParametric = True

               if not self.showHelp:
                  enteredCommand = self.commandParser.getEnteredCommand()
                  if enteredCommand == "":
                     log("Command must be supplied")
                     sys.stderr.write("Command must be supplied\n")
                     sys.stderr.flush()
                     if not exitCode:
                        exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      def getJobId(dataDirectory):
         nextLocalJobId = 0

         exitCode = 0
         localJobIdFileName = os.path.join(dataDirectory,"localJobId.dat")
         if not os.path.isfile(localJobIdFileName):
            fpLocalJobId = open(localJobIdFileName,'w')
            if fpLocalJobId:
               fpLocalJobId.write('0')
               fpLocalJobId.close()
            else:
               log("could not open %s for creating" % (localJobIdFileName))
               sys.stderr.write("could not open %s for creating\n" % (localJobIdFileName))
               sys.stderr.flush()
               exitCode = 1

         if not exitCode:
            fpLocalJobId = open(localJobIdFileName,'r')
            if fpLocalJobId:
               previousLocalJobId = int(fpLocalJobId.readline().strip())
               fpLocalJobId.close()
               nextLocalJobId = previousLocalJobId+1
               fpLocalJobId = open(localJobIdFileName,'w')
               if fpLocalJobId:
                  fpLocalJobId.write('%d' % (nextLocalJobId))
                  fpLocalJobId.close()
               else:
                  log("could not open %s for writing" % (localJobIdFileName))
                  sys.stderr.write("could not open %s for writing\n" % (localJobIdFileName))
                  sys.stderr.flush()
            else:
               log("could not open %s for reading" % (localJobIdFileName))
               sys.stderr.write("could not open %s for reading\n" % (localJobIdFileName))
               sys.stderr.flush()

         return(nextLocalJobId)

      exitCode = 0

      if not self.showHelp:
         if self.commandParser.getOption('jobId'):
            self.jobId = self.commandParser.getOption('jobId')
         if self.jobId == 0:
            self.jobId = getJobId(self.dataDirectory)

      self.localJobId = "%08d" % (self.jobId)
      logSetJobId(self.jobId)

      if not self.showHelp:
         if self.jobId == 0:
            exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setInfo(self):
      self.sitesInfo = SitesInfo(self.dataDirectory,"sites.dat",self.allowedVenueMechanisms)
      self.cloudsInfo = None
      self.tunnelsInfo = TunnelsInfo(self.dataDirectory,"tunnels.dat")

      self.toolsInfo = ToolsInfo(self.dataDirectory,"tools.dat")
      self.toolsInfo.applyUserRestriction(self.hubUserName)
      self.toolsInfo.applyGroupRestriction()

      self.managersInfo = ManagersInfo(self.dataDirectory,"managers.dat")

      self.environmentWhitelistInfo = EnvironmentWhitelistInfo(self.dataDirectory,"environmentwhitelist.dat")


   def showUsage(self):
      exitCode = 0
      if self.showHelp:
         if self.commandParser.getOption('help'):
            self.commandParser.showUsage()

         if self.commandParser.getOption('helpVenues'):
            venues = []
#           if self.runjobExists:
#              if self.disableStateCheck:
#                 condorStatusCommand = "condor_status -format '%s\t' Name"
#              else:
#                 condorStatusCommand = "condor_status -avail -format '%s\t' Name"
#              log("command = " + condorStatusCommand)
#              exitStatus,stdOutput,stdError = self.executeLaunchCommand(condorStatusCommand)
#              if exitStatus == 0:
#                 venues = stdOutput.split()
            for site in self.sitesInfo.getEnabledSites():
               venues.append(site)

            if len(venues) > 0:
               venues.sort()
               sys.stdout.write("\nCurrently available VENUES are:\n")
               for venue in venues:
                  sys.stdout.write("   %s\n" % (venue))
               sys.stdout.flush()
            del venues

         if self.commandParser.getOption('helpTools'):
            executableToolNames = self.toolsInfo.getToolNames()
            if len(executableToolNames) > 0:
               executableToolNames.sort()
               sys.stdout.write("\nCurrently available TOOLs are:\n")
               for executableToolName in executableToolNames:
                  sys.stdout.write("   %s\n" % (executableToolName))
               sys.stdout.flush()
            del executableToolNames

         if self.commandParser.getOption('helpManagers'):
            validManagers = self.managersInfo.getManagerNames()
            if len(validManagers) > 0:
               validManagers.sort()
               sys.stdout.write("\nCurrently available MANAGERs are:\n")
               for validManager in validManagers:
                  sys.stdout.write("   %s\n" % (validManager))
               sys.stdout.flush()
            del validManagers

         exitCode = 1
         self.jobStatistics[0]['jobSubmissionMechanism'] = "help"
         self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def reportExitCondition(self):
      if len(self.jobs) == 0:
         jobIndices = self.jobStatistics.keys()
         maximumJobIndex = max(jobIndices)

         try:
            mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
            remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
            remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
            exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
            cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
            elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
            waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
            nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']
            event          = self.jobStatistics[maximumJobIndex]['event']

            message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                      (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
            if event:
               message += " event=%s" % (event)
            log(message)
            message += "\n"
            if self.fpUserLogPath:
               self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
               self.fpUserLogPath.flush()
            os.write(3,message)
         except:
            pass
      else:
         instanceIndex = 0
         for instance in self.jobs:
            if (not self.successfulInstance) or \
               (self.successfulInstance and (instance != self.successfulInstance)):
               jobIndices = self.jobs[instance].jobStatistics.keys()
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                        if jobIndex != maximumJobIndex:
                           if exitCode == 0:
                              exitCode = 65533
                        cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                        event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                        instanceIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                 (instanceIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        if event:
                           message += " event=%s" % (event)
                        log(message)
                        message += "\n"
                        if self.fpUserLogPath:
                           self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                           self.fpUserLogPath.flush()
                        os.write(3,message)
               except:
                  pass

         if self.successfulInstance:
            instance = self.successfulInstance
            jobIndices = self.jobs[instance].jobStatistics.keys()
            jobIndices.sort()
            maximumJobIndex = max(jobIndices)

            try:
               for jobIndex in jobIndices:
                  if jobIndex != 0 or maximumJobIndex == 0:
                     mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                     remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                     remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                     exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                     if self.jobs[instance].remoteBatchSystem == 'PEGASUS':
                        if mechanism == "" or mechanism == 'Unknown':
                           mechanism = 'PEGASUS'
                     else:
                        if jobIndex != maximumJobIndex:
                           if exitCode == 0:
                              exitCode = 65533
                     cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                      self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                     elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                     waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                     nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                     event          = self.jobs[instance].jobStatistics[jobIndex]['event']

                     instanceIndex += 1
                     message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                               (instanceIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                     if event:
                        message += " event=%s" % (event)
                     log(message)
                     message += "\n"
                     if self.fpUserLogPath:
                        self.fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),self.jobId,message))
                        self.fpUserLogPath.flush()
                     os.write(3,message)
            except:
               pass

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortGlobal['abortAttempted'] = self.abortAttempted
         self.abortGlobal['abortSignal']    = signalNumber

         for instance in self.waitForJobsInfo:
            if 'isBatchJob' in self.waitForJobsInfo[instance]:
               if not self.waitForJobsInfo[instance]['isBatchJob']:
                  if self.waitForJobsInfo[instance]['recentJobStatus'] == 'R':
                     self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGINT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      log("Received SIGHUP!")
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGQUIT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGABRT!")
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      log("Received SIGTERM!")
      self.sigGEN_handler(signalNumber,frame)


   def getUserQuota(self):
      quotaCommand = "quota -w | tail -n 1"
      log("quotaCommand = " + quotaCommand)
      exitStatus,stdOutput,stdError = self.executeCommand(quotaCommand)
      if exitStatus == 0:
         try:
            quotaResults = stdOutput.split()
            if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
               currentUsage = int(quotaResults[1].strip("*"))
               self.quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
         except:
            log(stdOutput)
            log(stdError)
            log(traceback.format_exc())


   def getStdInputFile(self,
                       inputFiles):
      stdInputFile = '/dev/null'
      toCheck = []
      stdinFd = sys.stdin.fileno()
      toCheck.append(stdinFd)
      try:
         ready = select.select(toCheck,[],[],0.) # wait for input
      except select.error,err:
         ready = {}
         ready[0] = []
      if stdinFd in ready[0]:
         content = sys.stdin.read()
         if content != "":
            stdInputFile = ".__%08d.stdin" % (self.jobId)
            stdinFile = open(stdInputFile,'w')
            stdinFile.write(content)
            stdinFile.close()
            inputFiles.append(stdInputFile)
         del content

      return(stdInputFile)


   def executeCommand(self,
                      command,
                      streamOutput=False):
      child = subprocess.Popen(command,shell=True,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               sys.stdout.write(outChunk)
               sys.stdout.flush()

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               sys.stderr.write(errChunk)
               sys.stderr.flush()

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            log("%s failed w/ signal %d" % (command,os.WTERMSIG(err)))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            log("%s failed w/ exit code %d" % (command,err))
         if not streamOutput:
            log("%s" % ("".join(errData)))

      return(err,"".join(outData),"".join(errData))


   def executeLaunchCommand(self,
                            launchCommand,
                            streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

      while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      return(exitStatus,stdOutput,stdError)


   def buildJobDescription(self):
      exitCode = 0

      userDestinations = []
      inputFiles       = []
      nCpus            = 1
      ppn              = ""
      wallTime         = 60
      environment      = ""
      x509SubmitProxy  = ""

      if self.submitVenues == None:
         if self.commandParser.getOption('destinations'):
            userDestinations = self.commandParser.getOption('destinations')
      else:
         userDestinations = self.submitVenues.split(',')
      if len(userDestinations) > 0:
         reasonsDenied = self.sitesInfo.purgeDisabledSites(userDestinations)
         if len(userDestinations) == 0:
            errorMessage = "Access to all sites listed as possible execution hosts has been denied.\n"
            for userDestination in reasonsDenied:
               errorMessage += "   %20s %s\n" % (userDestination,reasonsDenied[userDestination])
            errorMessage += "Please select another site or attempt execution at a later time."
            log(errorMessage)
            sys.stderr.write(errorMessage + "\n")
            sys.stderr.flush()
            exitCode = 7
         del reasonsDenied

      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            inputFiles.append(inputFile.replace(' ','\ '))
      self.stdinput = self.getStdInputFile(inputFiles)

      if self.commandParser.getOption('outputFiles'):
         for outputFile in self.commandParser.getOption('outputFiles'):
            self.outputFiles.append(outputFile.replace(' ','\ '))

      if self.commandParser.getOption('manager'):
         manager = self.commandParser.getOption('manager')[0]
         if self.managersInfo.managerExists(manager):
            self.managerSpecified = True
            self.managerInfo      = self.managersInfo.getManagerInfo(manager)
         else:
            validManagers = self.managersInfo.getManagerNames()
            log("Invalid manager %s specified. Valid managers are %s" % (manager,validManagers))
            sys.stderr.write("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
            sys.stderr.flush()
            del validManagers
            if not exitCode:
               exitCode = 1
      else:
         self.managerInfo = self.managersInfo.getDefaultManagerInfo('serial')

      if self.commandParser.getOption('nCpus'):
         self.isMultiCoreRequest = True
         nCpus = int(self.commandParser.getOption('nCpus'))

      if self.commandParser.getOption('ppn'):
         if self.isMultiCoreRequest:
            ppn = str(self.commandParser.getOption('ppn'))

      if self.isParametric:
         self.maximumSelectedSites = 1
      else:
         if self.commandParser.getOption('nRedundant'):
            self.maximumSelectedSites = self.commandParser.getOption('nRedundant')

      if self.commandParser.getOption('wallTime'):
         if ":" in self.commandParser.getOption('wallTime'):
            wallTime = hhmmssTomin(self.commandParser.getOption('wallTime'))
            if wallTime < 0.:
               log("wallTime must be > 0")
               sys.stderr.write("wallTime must be > 0\n")
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
         else:
            try:
               wallTime = float(self.commandParser.getOption('wallTime'))
            except ValueError:
               log("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes")
               sys.stderr.write("Excepted estimated walltime formats are colon separated fields (hh:mm:ss) or minutes\n")
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1

      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               log("Invalid environment variable %s specified." % (environmentVariableValue))
               sys.stderr.write("Invalid environment variable %s specified.\n" % (environmentVariableValue))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1

            if   environmentVariable == "IGNORESTATECHECK":
               self.disableStateCheck = True
            elif environmentVariable == "IGNOREPROBECHECK":
               self.disableProbeCheck = True
            elif environmentVariable == "X509_SUBMIT_PROXY":
               x509SubmitProxy = value
            elif environmentVariable == "USER_LOG_PATH":
               try:
                  self.fpUserLogPath = open(value,'a')
               except:
                  log("User log %s could not be opened." % (value))
                  sys.stderr.write("User log %s could not be opened.\n" % (value))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 1
            elif environmentVariable == "USER_JOBID_PATH":
               try:
                  fpUserJobId = open(value,'w')
                  if fpUserJobId:
                     fpUserJobId.write("%d\n" % (self.jobId))
                     fpUserJobId.close()
               except:
                  log("User jobId file %s could not be created." % (value))
                  sys.stderr.write("User jobId file %s could not be created.\n" % (value))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 1
            else:
               if self.environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
                  environment += environmentVariable + "=" + value + " "
               else:
                  log("Environment variable %s could not be set." % (environmentVariable))
                  sys.stderr.write("Environment variable %s could not be set.\n" % (environmentVariable))
                  sys.stderr.flush()
         environment = environment.strip()

      if self.isMultiCoreRequest:
         if not self.managerSpecified:
            self.managerInfo = self.managersInfo.getDefaultManagerInfo('mpi')

      if not self.disableProbeCheck:
         ignoreProbeSites = self.sitesInfo.getIgnoreProbeSites()
         self.remoteMonitors['probe'] = RemoteProbeMonitor(self.probeMonitorHost,self.probeMonitorPort,
                                                           ignoreProbeSites=ignoreProbeSites)

      if len(userDestinations) > 0:
         goToGrid = self.sitesInfo.purgeOfflineSites(userDestinations,self.remoteMonitors['probe'])
         if len(userDestinations) == 0 and not goToGrid:
            errorMessage = "All sites listed as possible execution hosts are out of service.\n" + \
                           "Please select another site or attempt execution at a later time."
            log(errorMessage)
            sys.stderr.write(errorMessage + "\n")
            sys.stderr.flush()
            exitCode = 8

      nUserDestinations = len(userDestinations)
      self.sitesInfo.purgeResourceLimitedSites(userDestinations,nCpus,wallTime)
      if nUserDestinations > 0 and len(userDestinations) == 0:
         errorMessage = "All sites listed as possible execution hosts are unable to meet\n" + \
                        "the specified resource requirements.  Please select another site."
         log(errorMessage)
         sys.stderr.write(errorMessage + "\n")
         sys.stderr.flush()
         exitCode = 10

      self.userDestinations = copy.copy(userDestinations)
      self.inputFiles       = copy.copy(inputFiles)
      self.nCpus            = copy.copy(nCpus)
      self.ppn              = copy.copy(ppn)
      self.wallTime         = copy.copy(wallTime)
      self.environment      = copy.copy(environment)
      self.x509SubmitProxy  = copy.copy(x509SubmitProxy)

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def buildRedundantJobInstances(self):
      class CommandError(Exception):
         pass

      class InstancesError(Exception):
         pass

      class InstanceError(Exception):
         pass

      exitCode = 0
      enteredCommand = self.commandParser.getEnteredCommand()

      try:
         self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
         if len(self.enteredCommandArguments) > 1:
            for arg in self.enteredCommandArguments[1:]:
               if not arg.startswith('-'):
                  if os.path.isfile(arg.replace('\ ',' ')):
                     if not arg in self.inputFiles:
                        self.inputFiles.append(arg)

         executable         = self.enteredCommandArguments[0]
         transferExecutable = True
         if "/" in executable:
            if   not os.path.isfile(executable):
               log("Missing executable: %s" % (executable))
               sys.stderr.write("Missing executable: %s\n" % (executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
                  raise CommandError
            elif not os.access(executable,os.X_OK):
               log("Permission denied on executable: %s" % (executable))
               sys.stderr.write("Permission denied on executable: %s\n" % (executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
                  raise CommandError
         else:
            transferExecutable = False

         if transferExecutable:
            if not executable.startswith("/apps/"):
               if not isSubmitMember():
                  log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
                  sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 9
                     raise CommandError
         else:
            applicationIsStaged = False
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  applicationIsStaged = True
            if not applicationIsStaged and not isSubmitMember():
               log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
               sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 9
                  raise CommandError

      except CommandError:
         self.jobStatistics[0]['exitCode'] = exitCode
         return(exitCode)

      userDestinations = copy.copy(self.userDestinations)
      x509SubmitProxy  = copy.copy(self.x509SubmitProxy)

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination  = ""
      destinations = []

      try:
         if not transferExecutable:
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,self.remoteMonitors['probe'])
                  toolInfo = self.toolsInfo.selectTool(executable,expandedUserDestinations,self.remoteMonitors['probe'])
                  del expandedUserDestinations
                  if len(toolInfo['destinations']) == 0:
                     log("The combination of destination and executable is not properly specified.")
                     sys.stderr.write("The combination of destination and executable is not properly specified.\n")
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstancesError

                  if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
                     if self.managersInfo.managerExists(toolInfo['remoteManager']):
                        self.managerSpecified = True
                        self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable))
                        sys.stderr.write("Invalid manager %s specified for tool %s.\n" % (toolInfo['remoteManager'],executable))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstancesError
                  executable = toolInfo['executablePath']
               else:
                  log("Access to %s is restricted by user or group" % (executable))
                  sys.stderr.write("Access to %s is restricted by user or group\n" % (executable))
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstancesError

         instanceDestinations = []

         if   len(toolInfo['destinations']) == 1:
# if tool is staged and only one site is available
            destination = toolInfo['destinations'][0]
            instanceDestinations.append(destination)
         elif len(toolInfo['destinations']) == 0:
# if tool is not staged and only one user requested site is available, userDestinations reduced to grid sites
            destinations = self.sitesInfo.selectSites(userDestinations)
            if len(destinations) == 1:
               destination = destinations[0]
               instanceDestinations.append(destination)

         if destination == "":
            if len(toolInfo['destinations']) > 1:
# if tool is staged pick non-grid site
               nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','RUNJOB',toolInfo['destinations'])
               destinations = self.sitesInfo.selectSites(nonRunjobSites)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]
                  instanceDestinations.append(destination)

         if destination == "":
            if len(userDestinations) > 1:
# if user requested non-grid site is available
               nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','RUNJOB',userDestinations)
               destinations = self.sitesInfo.selectSites(nonRunjobSites)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]
                  instanceDestinations.append(destination)

         if   (self.sitesInfo.siteExists(destination)) and \
               self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
# selected a non-grid site
            pass
         elif (self.sitesInfo.siteExists(destination)) and \
               self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
# selected a grid site
            pass
         else:
            if   self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
# defer to tool or user specified runjob grid site selection
               if len(instanceDestinations) == 0:
                  instanceDestinations.append('2BeDetermined')
            else:
# select site from those declared with undeclaredSiteSelectionWeight
               enabledSites = self.sitesInfo.getEnabledSites()
               if self.remoteMonitors['probe']:
                  self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
               arbitraryExecutableRequired = not executable.startswith("/apps/")
               destinations = self.sitesInfo.selectUndeclaredSites(enabledSites, \
                                                                   arbitraryExecutableRequired, \
                                                                   self.maximumSelectedSites)
               if len(destinations) > 0:
                  for instanceDestination in destinations:
                     instanceDestinations.append(instanceDestination)
               else:
                  log("Currently no DESTINATIONs are available")
                  sys.stderr.write("Currently no DESTINATIONs are available\n")
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstancesError

         self.remoteMonitors['job']    = RemoteJobMonitor(self.jobMonitorHost,self.jobMonitorPort)
         self.remoteMonitors['tunnel'] = RemoteTunnelMonitor(self.tunnelMonitorHost,self.tunnelMonitorPort)
         self.remoteMonitors['cloud']  = None

         filesToRemove      = []
         emptyFilesToRemove = []
         instance = 0
         for destination in instanceDestinations:
            instance += 1
            instanceId = str(instance).zfill(2)
            exitCode         = 0
            inputFiles       = copy.copy(self.inputFiles)
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = ""
            cloudInstanceId = ""

            try:
               if   (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') != 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  if self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
                     gridsite = self.sitesInfo.getSitePegasusSite(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination))
                        sys.stderr.write("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],destination))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               elif (self.sitesInfo.siteExists(destination)) and \
                     self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'RUNJOB':
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  runjobVenues = self.sitesInfo.getSiteMembers(destination)
                  gridsite     = " ".join(runjobVenues)
               else:
                  if   self.runjobExists and (len(toolInfo['destinations']) > 0 or len(userDestinations) > 0 or destination != ""):
                     siteInfo = self.sitesInfo.getSiteInfo('grid')
                     if   len(toolInfo['destinations']) > 0:
                        gridsite = " ".join(toolInfo['destinations'])
                     elif len(userDestinations) > 0:
                        gridsite = " ".join(userDestinations)
                     else:
                        gridsite = destination
                  elif self.runjobExists or transferExecutable:
                     log("Invalid destination selection")
                     sys.stderr.write("Invalid destination selection\n")
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError

               if transferExecutable:
                  if not executable.startswith("/apps/"):
                     if not siteInfo['arbitraryExecutableAllowed']:
                        errorMessage = "submission of arbitrary executables is not permitted to selected site." + \
                                       "  Attempting to execute:"
                        log("%s %s" % (errorMessage,executable))
                        sys.stderr.write("%s %s\n" % (errorMessage,executable))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  log("command = " + tarCommand)
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     sys.stderr.write(tarStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  remoteArgs.append(self.enteredCommandArguments[0])
                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        rarg = arg
                        for tarFile in tarFiles:
                           if re.match(".*" + re.escape(tarFile) + "$",arg):
                              rarg = "./" + tarFile
                              break
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(remoteArgs[1:])
                  arguments = arguments.strip()
               else:
                  stageInTarFile = ""
                  remoteCommand = " ".join(self.enteredCommandArguments)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(self.enteredCommandArguments[1:])
                  arguments = arguments.strip()

               useEnvironment = ""
               if siteInfo['passUseEnvironment']:
                  reChoice = re.compile(".*_CHOICE$")
                  environmentVars = filter(reChoice.search,os.environ.keys())
                  if "PROMPT_CHOICE" in environmentVars:
                     environmentVars.remove("PROMPT_CHOICE")
                  if "EDITOR_CHOICE" in environmentVars:
                     environmentVars.remove("EDITOR_CHOICE")
                  if len(environmentVars) > 0:
                     useEnvironment = ". /apps/environ/.setup.sh\n"
                     if "ENVIRON_CONFIG_DIRS" in os.environ:
                        useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
                     for environmentVar in environmentVars:
                        useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        log("Tool environment variable must be of the form Variable = Value.")
                        sys.stderr.write("Tool environment variable must be of the form Variable = Value.\n")
                        sys.stderr.flush()

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,
                                                            self.batchCommands,
                                                            self.isParametric,self.localJobId,instanceId,destination,
                                                            enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,managerInfo,
                                                            nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,
                                                          self.batchCommands,
                                                          self.isParametric,self.localJobId,instanceId,destination,
                                                          enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,managerInfo,
                                                          nCpus,nNodes,ppn,wallTime,
                                                          x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'gsissh':
                  self.jobs[instance] = VenueMechanismGsiSsh(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,
                                                             self.batchCommands,
                                                             self.isParametric,self.localJobId,instanceId,destination,
                                                             enteredCommand,x509SubmitProxy,
                                                             stageInTarFile,
                                                             transferExecutable,executable,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,managerInfo,
                                                             nCpus,nNodes,ppn,wallTime,
                                                             self.quotaLimit,
                                                             timeHistoryLogs)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     log("command = " + tarCommand)
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        log(tarStdError)
                        log(tarStdOutput)
                        sys.stderr.write(tarStdError)
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  log("command = " + niceCommand)
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     log(gzipStdError)
                     log(gzipStdOutput)
                     sys.stderr.write(gzipStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
               self.waitForJobsInfo[instance]['isBatchJob']            = waitForJobInfo['isBatchJob']
               self.waitForJobsInfo[instance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
               self.waitForJobsInfo[instance]['remoteBatchSystem']     = self.jobs[instance].remoteBatchSystem
               self.waitForJobsInfo[instance]['instanceDirectory']     = self.jobs[instance].instanceDirectory

            except InstanceError:
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

            stdFile = "%s.stderr" % (self.localJobId)
            emptyFilesToRemove.append(stdFile)
            for batchQueue in 'pbs','lsf','ll','slurm','condor','factory':
               for outFile in 'stdout','stderr':
                  batchQueueOutputFile = "%s_%s_%s.%s" % (batchQueue,self.localJobId,instanceId,outFile)
                  emptyFilesToRemove.append(batchQueueOutputFile)

         nonBatchJobRan = False
         markedForDeletion = []
         for instance in self.waitForJobsInfo:
            if not self.waitForJobsInfo[instance]['isBatchJob']:
               if not nonBatchJobRan:
                  if self.jobs[instance].sendFiles():
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'R'
                     if self.jobs[instance].executeJob():
                        self.jobs[instance].retrieveFiles()
                        self.jobs[instance].cleanupFiles()
                  self.jobs[instance].recordJobStatistics()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
                  self.successfulInstance = instance
                  nonBatchJobRan = True
               else:
                  markedForDeletion.append(instance)

         if nonBatchJobRan:
            for instance in markedForDeletion:
               del self.jobs[instance]
         del markedForDeletion

         jobsRunning = False
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['isBatchJob']:
               if self.jobs[instance].sendFiles():
                  if self.jobs[instance].executeJob():
                     waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
                     self.waitForJobsInfo[instance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                     self.waitForJobsInfo[instance]['knownSite']       = waitForJobInfo['knownSite']
                     self.waitForJobsInfo[instance]['recentJobStatus'] = '?'
                     jobsRunning = True
                  else:
                     self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               else:
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

         while jobsRunning and (not self.successfulInstance) and (not self.abortGlobal['abortAttempted']):
            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForBatchJobs(self.waitForJobsInfo,self.abortGlobal)
            for instance in self.waitForJobsInfo:
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
            for instance in completeRemoteJobIndexes:
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance
            if len(completeRemoteJobIndexes) == 0:
               jobsRunning = False

         if self.abortGlobal['abortAttempted']:
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                  self.jobs[instance].killScripts(self.abortGlobal['abortSignal'])
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            for instance in self.waitForJobsInfo:
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'

         if self.successfulInstance:
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                  self.jobs[instance].killScripts(signal.SIGUSR1)
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            for instance in self.waitForJobsInfo:
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
            for instance in self.waitForJobsInfo:
               if   self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               elif instance != self.successfulInstance:
                  self.jobs[instance].cleanupFiles()

         if self.successfulInstance:
            self.jobs[self.successfulInstance].recoverFiles(False)
            fileToMove = "%s_%02d.stdout" % (self.localJobId,self.successfulInstance)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.stdout" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.stderr" % (self.localJobId,self.successfulInstance)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.stderr" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
            fileToMove = "%s_%02d.SUCCESS" % (self.localJobId,self.successfulInstance)
            if(os.path.exists(fileToMove)):
               fileNewName = "%s.SUCCESS" % (self.localJobId)
               os.rename(fileToMove,fileNewName)
         else:
            for instance in self.waitForJobsInfo:
               self.jobs[instance].recoverStdFiles()
            if len(self.jobs) > 0:
               recoverInstance = random.choice(xrange(len(self.jobs)))+1
               self.jobs[recoverInstance].recoverFiles(True)

         for instance in self.waitForJobsInfo:
            self.jobs[instance].removeInstanceDirectory()

         for fileToRemove in filesToRemove:
            if(os.path.exists(fileToRemove)):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if(os.path.exists(fileToRemove)):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

      except InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildParametricJobInstances(self):
      class CommandError(Exception):
         pass

      class InstancesError(Exception):
         pass

      class InstanceError(Exception):
         pass

      exitCode = 0
      self.enteredCommandArguments = self.commandParser.getEnteredCommandArguments()

      try:
         executable         = self.enteredCommandArguments[0]
         transferExecutable = True
         if "/" in executable:
            if   not os.path.isfile(executable):
               log("Missing executable: %s" % (executable))
               sys.stderr.write("Missing executable: %s\n" % (executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
                  raise CommandError
            elif not os.access(executable,os.X_OK):
               log("Permission denied on executable: %s" % (executable))
               sys.stderr.write("Permission denied on executable: %s\n" % (executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 1
                  raise CommandError
         else:
            transferExecutable = False

         if transferExecutable:
            if not executable.startswith("/apps/"):
               if not isSubmitMember():
                  log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
                  sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
                  sys.stderr.flush()
                  if not exitCode:
                     exitCode = 9
                     raise CommandError
         else:
            applicationIsStaged = False
            if self.toolsInfo.isExecutableTool(executable):
               if self.toolsInfo.isPermissionGranted(executable):
                  applicationIsStaged = True
            if not applicationIsStaged and not isSubmitMember():
               log("%s not member of group 'submit' - Attempting to execute: %s" % (self.hubUserName,executable))
               sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (self.hubUserName,executable))
               sys.stderr.flush()
               if not exitCode:
                  exitCode = 9
                  raise CommandError

         userDestinations = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',self.userDestinations)
         if len(self.userDestinations) > 0 and len(userDestinations) == 0:
            log("None of the requested venues support Pegasus job execution")
            sys.stderr.write("None of the requested venues support Pegasus job execution\n")
            sys.stderr.flush()
            if not exitCode:
               exitCode = 8
               raise CommandError

      except CommandError:
         self.jobStatistics[0]['exitCode'] = exitCode
         return(exitCode)

      toolInfo = self.toolsInfo.getDefaultToolInfo()
      destination  = ""
      destinations = []

      try:
         if self.toolsInfo.isExecutableTool(executable):
            if self.toolsInfo.isPermissionGranted(executable):
               expandedUserDestinations = self.sitesInfo.getExpandedSiteNames(userDestinations,self.remoteMonitors['probe'])
               toolInfo = self.toolsInfo.selectTool(executable,expandedUserDestinations,self.remoteMonitors['probe'])
               del expandedUserDestinations
               if len(toolInfo['destinations']) == 0:
                  log("The combination of destination and executable is not properly specified.")
                  sys.stderr.write("The combination of destination and executable is not properly specified.\n")
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstancesError

               if not self.managerSpecified and (toolInfo['remoteManager'] != ""):
                  if self.managersInfo.managerExists(toolInfo['remoteManager']):
                     self.managerSpecified = True
                     self.managerInfo      = self.managersInfo.getManagerInfo(toolInfo['remoteManager'])
                  else:
                     log("Invalid manager %s specified for tool %s." % (toolInfo['remoteManager'],executable))
                     sys.stderr.write("Invalid manager %s specified for tool %s.\n" % (toolInfo['remoteManager'],executable))
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstancesError
               executable = toolInfo['executablePath']
            else:
               log("Access to %s is restricted by user or group" % (executable))
               sys.stderr.write("Access to %s is restricted by user or group\n" % (executable))
               sys.stderr.flush()
               exitCode = 1
               raise InstancesError

         if   len(toolInfo['destinations']) == 1:
# if tool is staged and only one site is available
            destination = toolInfo['destinations'][0]
         elif len(toolInfo['destinations']) == 0:
# if tool is not staged and only one user requested site is available, userDestinations reduced to pegasus sites
            destinations = self.sitesInfo.selectSites(userDestinations)
            if len(destinations) == 1:
               destination = destinations[0]

         if destination == "":
            if len(toolInfo['destinations']) > 1:
# if tool is staged pick site
               destinations = self.sitesInfo.selectSites(toolInfo['destinations'])
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]

         if destination == "":
            if len(toolInfo['destinations']) > 1:
# if tool is staged pick site
               nonRunjobSites = self.sitesInfo.getSitesWithoutKeyValue('remoteBatchSystem','RUNJOB',toolInfo['destinations'])
               destinations = self.sitesInfo.selectSites(nonRunjobSites)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]

         if destination == "":
            if len(userDestinations) > 1:
# if user requested site is available
               destinations = self.sitesInfo.selectSites(userDestinations)
               if len(destinations) > 0:
                  destinationIndex = random.randint(0,len(destinations)-1)
                  destination = destinations[destinationIndex]

         if (self.sitesInfo.siteExists(destination)) and \
             self.sitesInfo.getSiteKeyValue(destination,'remoteBatchSystem') == 'PEGASUS':
# selected a pegasus site
            pass
         else:
# select site from those declared with undeclaredSiteSelectionWeight
            enabledSites = self.sitesInfo.getEnabledSites()
            if self.remoteMonitors['probe']:
               self.remoteMonitors['probe'].purgeOfflineSites(enabledSites)
            pegasusSites = self.sitesInfo.getSitesWithKeyValue('remoteBatchSystem','PEGASUS',enabledSites)
            arbitraryExecutableRequired = transferExecutable and not executable.startswith("/apps/")
            destinations = self.sitesInfo.selectUndeclaredSites(pegasusSites,
                                                                arbitraryExecutableRequired,
                                                                self.maximumSelectedSites)
            if len(destinations) > 0:
               destination = destinations[0]
            else:
               log("Currently no DESTINATIONs are available")
               sys.stderr.write("Currently no DESTINATIONs are available\n")
               sys.stderr.flush()
               exitCode = 1
               raise InstancesError

         self.remoteMonitors['job']    = RemoteJobMonitor(self.jobMonitorHost,self.jobMonitorPort)
         self.remoteMonitors['tunnel'] = RemoteTunnelMonitor(self.tunnelMonitorHost,self.tunnelMonitorPort)

         filesToRemove      = []
         emptyFilesToRemove = []
         jobPath = os.path.join(os.getcwd(),self.localJobId)
         if not os.path.isdir(jobPath):
            os.mkdir(jobPath)
         parameterCombinationsPath = os.path.join(jobPath,'remoteParameterCombinations.csv')
         self.commandParser.writeParameterCombinations(parameterCombinationsPath)

         instanceError = False
         inputsPath        = os.path.join(os.getcwd(),self.localJobId,'InputFiles')
         nInstanceIdDigits = max(2,int(math.log10(self.commandParser.getParameterCombinationCount())+1))
         instance = 0
#        for parameterCombination in self.parameterCombinations:
         for substitutions in self.commandParser.getNextParameterCombinationFromCSV(parameterCombinationsPath):
            instance += 1
            instanceId = str(instance).zfill(nInstanceIdDigits)
            exitCode         = 0
            nCpus            = copy.copy(self.nCpus)
            ppn              = copy.copy(self.ppn)
            wallTime         = copy.copy(self.wallTime)
            environment      = copy.copy(self.environment)
            managerSpecified = copy.copy(self.managerSpecified)
            managerInfo      = copy.copy(self.managerInfo)

#           substitutions = {}
#           for parameterName,parameterValue in zip(self.parameterNames,parameterCombination):
#              substitutions[parameterName] = parameterValue

            enteredCommand = self.commandParser.getEnteredCommand()
            template = ParameterTemplate(enteredCommand)
            try:
               enteredCommand = template.substitute(substitutions)
            except KeyError,err:
               sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
               sys.stderr.flush()
               exitCode = 1
               raise InstanceError
            log(enteredCommand)

            inputFiles = []
            for inputFile in self.inputFiles:
               template = ParameterTemplate(inputFile)
               try:
                  inputFile = template.substitute(substitutions)
               except KeyError,err:
                  sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                  sys.stderr.flush()
                  exitCode = 1
                  raise InstanceError
               inputFiles.append(inputFile)

            if len(self.enteredCommandArguments) > 1:
               for arg in self.enteredCommandArguments[1:]:
                  if not arg.startswith('-'):
                     template = ParameterTemplate(arg)
                     try:
                        arg = template.substitute(substitutions)
                     except KeyError,err:
                        sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError
                     if os.path.isfile(arg.replace('\ ',' ').replace('@:','')):
                        if not arg in inputFiles:
                           inputFiles.append(arg)

            siteInfo = self.sitesInfo.getDefaultSiteInfo()

            gridsite        = "Unknown"
            nNodes          = "1"
            arguments       = ""
            cloudInstanceId = ""

            try:
               if self.sitesInfo.siteExists(destination):
                  siteInfo = self.sitesInfo.getSiteInfo(destination)
                  gridsite = self.sitesInfo.getSitePegasusSite(destination)
                  if not managerSpecified and siteInfo['remoteManager'] != "":
                     if self.managersInfo.managerExists(siteInfo['remoteManager']):
                        managerSpecified = True
                        managerInfo      = self.managersInfo.getManagerInfo(siteInfo['remoteManager'])
                     else:
                        log("Invalid manager %s specified for site %s." % (siteInfo['remoteManager'],destination))
                        sys.stderr.write("Invalid manager %s specified for site %s.\n" % (siteInfo['remoteManager'],destination))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError
                  if ppn == "":
                     ppn = siteInfo['remotePpn']
                  nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
                  if int(nNodes) == 1:
                     if int(nCpus) < int(ppn):
                        ppn = str(nCpus)
               else:
                  if transferExecutable:
                     log("Invalid destination selection")
                     sys.stderr.write("Invalid destination selection\n")
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError

               if transferExecutable:
                  if not executable.startswith("/apps/"):
                     if not siteInfo['arbitraryExecutableAllowed']:
                        errorMessage = "submission of arbitrary executables is not permitted to selected site." + \
                                       "  Attempting to execute:"
                        log("%s %s" % (errorMessage,executable))
                        sys.stderr.write("%s %s\n" % (errorMessage,executable))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError

               localJobIdFile = "%s.%s_%s" % (LOCALJOBID,self.localJobId,instanceId)
               self.executeCommand("touch " + localJobIdFile)
               inputFiles.append(localJobIdFile)
               filesToRemove.append(localJobIdFile)

               if transferExecutable:
                  inputFiles.append(executable)

               substitutedInputFiles = []
               for inputFile in inputFiles:
                  if '@:' in inputFile:
                     inputFile = inputFile.replace('@:','')
                     fpInputFile = open(inputFile,'r')
                     if fpInputFile:
                        inputText = fpInputFile.readlines()
                        fpInputFile.close()
                        template = ParameterTemplate(''.join(inputText))
                        try:
                           inputText = template.substitute(substitutions)
                        except KeyError,err:
                           sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                           sys.stderr.flush()
                           exitCode = 1
                           raise InstanceError
                        instanceInputsPath = os.path.join(inputsPath,instanceId)
                        if not os.path.isdir(instanceInputsPath):
                           os.makedirs(instanceInputsPath)
                        inputName = os.path.basename(inputFile)
                        inputPath = os.path.join(instanceInputsPath,inputName)
                        fpInputPath = open(inputPath,'w')
                        if fpInputPath:
                           fpInputPath.writelines(inputText)
                           fpInputPath.close()
                           substitutedInputFiles.append(inputPath)
                  else:
                     substitutedInputFiles.append(inputFile)
               del inputFiles
               inputFiles = substitutedInputFiles

               if siteInfo['stageFiles']:
                  stageInTarFile = "%s_%s_input.tar" % (self.localJobId,instanceId)
                  tarCommand = buildCreateTarCommand(stageInTarFile,inputFiles)
                  log("command = " + tarCommand)
                  tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                  if tarExitStatus:
                     log(tarStdError)
                     log(tarStdOutput)
                     sys.stderr.write(tarStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError

                  filesToRemove.append(stageInTarFile)
                  remoteArgs = []
                  arg = self.enteredCommandArguments[0]
                  template = ParameterTemplate(arg)
                  try:
                     rarg = template.substitute(substitutions)
                  except KeyError,err:
                     sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError
                  remoteArgs.append(rarg)
                  if len(self.enteredCommandArguments) > 1:
                     tarFiles = tarStdOutput.strip().split('\n')
                     for arg in self.enteredCommandArguments[1:]:
                        template = ParameterTemplate(arg)
                        try:
                           rarg = template.substitute(substitutions)
                        except KeyError,err:
                           sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                           sys.stderr.flush()
                           exitCode = 1
                           raise InstanceError
                        for tarFile in tarFiles:
                           if re.match(".*" + re.escape(tarFile) + "$",arg):
                              rarg = "./" + tarFile
                              break
                        remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(remoteArgs[1:])
                  arguments = arguments.strip()
               else:
                  stageInTarFile = ""
                  remoteArgs = []
                  for arg in self.enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     try:
                        rarg = template.substitute(substitutions)
                     except KeyError,err:
                        sys.stderr.write("Pattern substitution failed for @@%s\n" % (err[0]))
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError
                     remoteArgs.append(rarg)

                  remoteCommand = " ".join(remoteArgs)
                  log("remoteCommand " + remoteCommand)

                  arguments = " ".join(remoteArgs[1:])
                  arguments = arguments.strip()

               useEnvironment = ""
               if siteInfo['passUseEnvironment']:
                  reChoice = re.compile(".*_CHOICE$")
                  environmentVars = filter(reChoice.search,os.environ.keys())
                  if "PROMPT_CHOICE" in environmentVars:
                     environmentVars.remove("PROMPT_CHOICE")
                  if "EDITOR_CHOICE" in environmentVars:
                     environmentVars.remove("EDITOR_CHOICE")
                  if len(environmentVars) > 0:
                     useEnvironment  = "ENVIRON_CONFIG_DIRS=\"\"\n"
                     useEnvironment += ". /apps/environ/.setup.sh\n"
                     if "ENVIRON_CONFIG_DIRS" in os.environ:
                        useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
                     for environmentVar in environmentVars:
                        useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

               managedEnvironment = managerInfo['environment']
               if len(managedEnvironment) > 0:
                  for environmentVariableValue in managedEnvironment:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if   nParts == 1:
                        environmentVariable = environmentVariableValue.strip()
                        if environmentVariable in os.environ:
                           value = os.environ[environmentVariable]
                     elif nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           if environmentVariable in os.environ:
                              value = os.environ[environmentVariable]

                     if environmentVariable != "" and value != "":
                        if environment.find(environmentVariable + '=') < 0:
                           environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               if len(toolInfo['environment']) > 0:
                  for environmentVariableValue in toolInfo['environment']:
                     environmentVariable = ""
                     value               = ""
                     nParts = len(environmentVariableValue.split('='))
                     if nParts == 2:
                        environmentVariable,value = environmentVariableValue.split('=')
                        environmentVariable = environmentVariable.strip()
                        value               = value.strip()
                        if value == "":
                           value = " "
                     else:
                        log("Tool environment variable must be of the form Variable = Value.")
                        sys.stderr.write("Tool environment variable must be of the form Variable = Value.\n")
                        sys.stderr.flush()

                     if environmentVariable != "" and value != "":
                        environment += environmentVariable + "=" + value + " "
                  environment = environment.strip()

               timeHistoryLogs = {}
               timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
               timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
               timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,instanceId)
               timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,instanceId)
               timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,instanceId)
               timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,instanceId)
               timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,instanceId)
               timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,instanceId)
               timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,instanceId)

               if   siteInfo['venueMechanism'] == 'local':
                  self.jobs[instance] = VenueMechanismLocal(self.remoteMonitors,
                                                            self.hubUserName,self.hubUserId,
                                                            self.batchCommands,
                                                            self.isParametric,self.localJobId,instanceId,destination,
                                                            enteredCommand,gridsite,
                                                            stageInTarFile,
                                                            transferExecutable,executable,
                                                            self.stdinput,arguments,useEnvironment,environment,
                                                            self.isMultiCoreRequest,siteInfo,managerInfo,
                                                            nCpus,nNodes,ppn,wallTime,
                                                            self.quotaLimit,
                                                            self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                            timeHistoryLogs,self.pegasusVersion,self.pegasusHome)
               elif siteInfo['venueMechanism'] == 'ssh':
                  self.jobs[instance] = VenueMechanismSsh(self.remoteMonitors,
                                                          self.hubUserName,self.hubUserId,
                                                          self.batchCommands,
                                                          self.isParametric,self.localJobId,instanceId,destination,
                                                          enteredCommand,gridsite,
                                                          self.tunnelsInfo,
                                                          cloudInstanceId,
                                                          stageInTarFile,
                                                          transferExecutable,executable,
                                                          self.stdinput,arguments,useEnvironment,environment,
                                                          self.isMultiCoreRequest,siteInfo,managerInfo,
                                                          nCpus,nNodes,ppn,wallTime,
                                                          self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                          self.quotaLimit,
                                                          timeHistoryLogs,self.pegasusVersion,self.pegasusHome)

               exitCode,scriptFiles = self.jobs[instance].createScripts()
               if exitCode:
                  raise InstanceError

               if siteInfo['stageFiles']:
                  if len(scriptFiles) > 0:
                     tarCommand = buildAppendTarCommand(stageInTarFile,scriptFiles)
                     log("command = " + tarCommand)
                     tarExitStatus,tarStdOutput,tarStdError = self.executeCommand(tarCommand)
                     if tarExitStatus:
                        log(tarStdError)
                        log(tarStdOutput)
                        sys.stderr.write(tarStdError)
                        sys.stderr.flush()
                        exitCode = 1
                        raise InstanceError

                  niceCommand = "nice -n 19 gzip " + stageInTarFile
                  log("command = " + niceCommand)
                  gzipExitStatus,gzipStdOutput,gzipStdError = self.executeCommand(niceCommand)
                  if gzipExitStatus:
                     log(gzipStdError)
                     log(gzipStdOutput)
                     sys.stderr.write(gzipStdError)
                     sys.stderr.flush()
                     exitCode = 1
                     raise InstanceError
                  stageInTarFile += '.gz'
                  filesToRemove.append(stageInTarFile)

               self.jobs[instance].stageFilesToInstanceDirectory()

               self.waitForJobsInfo[instance] = {}
               self.waitForJobsInfo[instance]['recentJobStatus'] = '?'

            except InstanceError:
               instanceError = True
               if instance in self.jobs:
                  self.jobs[instance].jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobs[instance].jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobs[instance].jobStatistics[0]['exitCode']               = exitCode
               else:
                  self.jobStatistics[0]['jobSubmissionMechanism'] = siteInfo['venueMechanism']
                  self.jobStatistics[0]['venue']                  = siteInfo['venue']
                  self.jobStatistics[0]['exitCode']               = exitCode

         if instanceError:
            exitCode = 1
            raise InstancesError

         nInstances      = instance
         executeInstance = nInstances+1

         timeHistoryLogs = {}
         timeHistoryLogs['hubGridJobIdLogPath']      = self.hubGridJobIdLogPath
         timeHistoryLogs['hubGridJobHistoryLogPath'] = self.hubGridJobHistoryLogPath
         timeHistoryLogs['timestampTransferred']     = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.localJobId,"0")
         timeHistoryLogs['timestampStart']           = "%s.%s_%s" % (TIMESTAMPSTART,self.localJobId,"0")
         timeHistoryLogs['timestampFinish']          = "%s.%s_%s" % (TIMESTAMPFINISH,self.localJobId,"0")
         timeHistoryLogs['timeResults']              = "%s.%s_%s" % (TIMERESULTS,self.localJobId,"0")
         timeHistoryLogs['jobGridResource']          = "%s.%s_%s" % (JOBGRIDRESOURCE,self.localJobId,"0")
         timeHistoryLogs['jobGridHistory']           = "%s.%s_%s" % (JOBGRIDHISTORY,self.localJobId,"0")
         timeHistoryLogs['jobGridJobId']             = "%s.%s_%s" % (JOBGRIDJOBID,self.localJobId,"0")

         enteredCommand = self.commandParser.getEnteredCommand()

         if   siteInfo['venueMechanism'] == 'local':
            self.jobs[executeInstance] = VenueMechanismLocal(self.remoteMonitors,
                                                             self.hubUserName,self.hubUserId,
                                                             self.batchCommands,
                                                             self.isParametric,self.localJobId,"WF;%d" % (nInstances),destination,
                                                             enteredCommand,gridsite,
                                                             stageInTarFile,
                                                             transferExecutable,executable,
                                                             self.stdinput,arguments,useEnvironment,environment,
                                                             self.isMultiCoreRequest,siteInfo,managerInfo,
                                                             nCpus,nNodes,ppn,wallTime,
                                                             self.quotaLimit,
                                                             self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                             timeHistoryLogs,self.pegasusVersion,self.pegasusHome)
         elif siteInfo['venueMechanism'] == 'ssh':
            self.jobs[executeInstance] = VenueMechanismSsh(self.remoteMonitors,
                                                           self.hubUserName,self.hubUserId,
                                                           self.batchCommands,
                                                           self.isParametric,self.localJobId,"WF;%d" % (nInstances),destination,
                                                           enteredCommand,gridsite,
                                                           self.tunnelsInfo,
                                                           cloudInstanceId,
                                                           stageInTarFile,
                                                           transferExecutable,executable,
                                                           self.stdinput,arguments,useEnvironment,environment,
                                                           self.isMultiCoreRequest,siteInfo,managerInfo,
                                                           nCpus,nNodes,ppn,wallTime,
                                                           self.x509SubmitProxy,self.disableProbeCheck,self.disableStateCheck,
                                                           self.quotaLimit,
                                                           timeHistoryLogs,self.pegasusVersion,self.pegasusHome)

         exitCode,scriptFiles = self.jobs[executeInstance].createScripts()
         if exitCode:
            raise InstancesError

         self.waitForJobsInfo[executeInstance] = {}
         waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
         self.waitForJobsInfo[executeInstance]['isBatchJob']            = waitForJobInfo['isBatchJob']
         self.waitForJobsInfo[executeInstance]['siteMonitorDesignator'] = waitForJobInfo['siteMonitorDesignator']
         self.waitForJobsInfo[executeInstance]['remoteBatchSystem']     = self.jobs[executeInstance].remoteBatchSystem
         self.waitForJobsInfo[executeInstance]['instanceDirectory']     = self.jobs[executeInstance].instanceDirectory

         stdFile = "%s.stderr" % (self.localJobId)
         emptyFilesToRemove.append(stdFile)
         for batchQueue in 'pbs','lsf','ll','slurm','condor','factory':
            for outFile in 'stdout','stderr':
               batchQueueOutputFile = "%s_%s.%s" % (batchQueue,self.localJobId,outFile)
               emptyFilesToRemove.append(batchQueueOutputFile)

         jobsRunning = False
         if self.waitForJobsInfo[executeInstance]['isBatchJob']:
            if self.jobs[executeInstance].sendFiles():
               if self.jobs[executeInstance].executeJob():
                  waitForJobInfo = self.jobs[executeInstance].getWaitForJobInfo()
                  self.waitForJobsInfo[executeInstance]['remoteJobId']     = waitForJobInfo['remoteJobId']
                  self.waitForJobsInfo[executeInstance]['knownSite']       = waitForJobInfo['knownSite']
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = '?'
                  jobsRunning = True
               else:
                  self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'
            else:
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

         while jobsRunning and (not self.successfulInstance) and (not self.abortGlobal['abortAttempted']):
            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForWorkflowJobs(self.waitForJobsInfo,
                                                                                      nInstances,parameterCombinationsPath,
                                                                                      self.isClientTTY,self.abortGlobal)
#           if 'recentJobSite' in self.waitForJobsInfo[executeInstance]:
#              executionVenue = self.waitForJobsInfo[executeInstance]['recentJobSite']
#              self.jobs[executeInstance].updateVenue(executionVenue)
            for instance in completeRemoteJobIndexes:
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance
            if len(completeRemoteJobIndexes) == 0:
               jobsRunning = False

         if self.abortGlobal['abortAttempted']:
            if self.waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
               self.jobs[executeInstance].killScripts(self.abortGlobal['abortSignal'])
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'K'
            self.remoteMonitors['job'].waitForKilledBatchJobs(self.waitForJobsInfo)
            if 'recentJobSite' in self.waitForJobsInfo[executeInstance]:
               executionVenue = self.waitForJobsInfo[executeInstance]['recentJobSite']
               self.jobs[executeInstance].updateVenue(executionVenue)
            if self.waitForJobsInfo[executeInstance]['recentJobStatus'] == 'KD':
               self.jobs[executeInstance].cleanupFiles()
               self.waitForJobsInfo[executeInstance]['recentJobStatus'] = 'D'

         if self.successfulInstance:
            for instance in xrange(1,nInstances+1):
               self.jobs[instance].jobSubmitted = True

         for instance in xrange(1,nInstances+1):
            if 'recentJobSite' in self.waitForJobsInfo[instance]:
               executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
               self.jobs[instance].updateVenue(executionVenue)
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()

         for fileToRemove in filesToRemove:
            if(os.path.exists(fileToRemove)):
               os.remove(fileToRemove)

         for fileToRemove in emptyFilesToRemove:
            if(os.path.exists(fileToRemove)):
               if(os.path.getsize(fileToRemove) == 0):
                  os.remove(fileToRemove)

         if os.path.isdir(inputsPath):
            shutil.rmtree(inputsPath,True)

      except InstancesError:
         self.jobStatistics[0]['exitCode'] = exitCode

      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      return(exitCode)


   def buildJobInstances(self):
      if self.isParametric:
         exitCode = self.buildParametricJobInstances()
      else:
         exitCode = self.buildRedundantJobInstances()

      return(exitCode)


