# @package      hubzero-submit-client
# @file         LocalBatchAppScript.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class LocalBatchAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                instanceDirectory,
                localJobId,
                instanceId,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                submissionScriptsInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.instanceDirectory     = instanceDirectory
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.executable            = executable
      self.stdinput              = stdinput
      if isinstance(arguments,list):
         self.arguments          = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments   += "%s " % (argument.replace('\\ ',' ').replace(' ','\ '))
            else:
               self.arguments   += "%s " % (argument)
      else:
         self.arguments          = arguments
      self.useEnvironment        = useEnvironment
      self.environment           = environment
      self.isMultiCoreRequest    = False
      self.submissionScriptsInfo = submissionScriptsInfo
      self.timestampTransferred  = timeHistoryLogs['timestampTransferred']
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timeResults           = timeHistoryLogs['timeResults']
      if 'exitCode' in timeHistoryLogs:
         self.exitCode           = timeHistoryLogs['exitCode']
      else:
         self.exitCode           = ""


   def __buildAppSerialScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','serial')

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      substitutions["JOBID"]             = self.localJobId
      substitutions["RUNNAME"]           = self.runName
      substitutions["INSTANCEID"]        = self.instanceId
      substitutions["INSTANCEDIRECTORY"] = self.instanceDirectory
      substitutions["EXECUTABLE"]        = self.executable
      if self.stdinput == "":
         substitutions["STDIN"]          = '/dev/null'
      else:
         substitutions["STDIN"]          = self.stdinput
      substitutions["ARGUMENTS"]         = self.arguments
      substitutions["USEENVIRONMENT"]    = self.useEnvironment
      substitutions["ENVIRONMENT"]       = environmentExport
      substitutions["TS_TRANSFERRED"]    = self.timestampTransferred
      substitutions["TS_START"]          = self.timestampStart
      substitutions["TS_FINISH"]         = self.timestampFinish
      substitutions["TIME_RESULTS"]      = self.timeResults
      if self.exitCode:
         substitutions["EXITCODE"]       = "echo ${exitStatus} > %s\nexitStatus=0" % (self.exitCode)
      else:
         substitutions["EXITCODE"]       = ""
      substitutions["HUBUSERNAME"]       = self.hubUserName
      substitutions["HUBUSERID"]         = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         appScript = ""
      else:
         appScript = self.__buildAppSerialScript()
      appScriptExecutable = True

      return(appScriptName,appScript,appScriptExecutable)


