# @package      hubzero-submit-client
# @file         LocalBatchAppScript.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os.path
import re

class LocalBatchAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                instanceDirectory,
                localJobId,
                instanceId,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                timeHistoryLogs):
      self.hubUserName          = hubUserName
      self.hubUserId            = hubUserId
      self.runName              = runName
      self.instanceDirectory    = instanceDirectory
      self.localJobId           = localJobId
      self.instanceId           = instanceId
      self.executable           = executable
      self.stdinput             = stdinput
      if isinstance(arguments,list):
         self.arguments         = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments  += '"%s" ' % (argument)
            else:
               self.arguments  += "%s " % (argument)
      else:
         self.arguments         = arguments
      self.useEnvironment       = useEnvironment
      self.environment          = environment
      self.isMultiCoreRequest   = False
      self.timestampTransferred = timeHistoryLogs['timestampTransferred']
      self.timestampStart       = timeHistoryLogs['timestampStart']
      self.timestampFinish      = timeHistoryLogs['timestampFinish']
      self.timeResults          = timeHistoryLogs['timeResults']
      if 'exitCode' in timeHistoryLogs:
         self.exitCode          = timeHistoryLogs['exitCode']
      else:
         self.exitCode          = ""


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# LocalBatchAppScript:makeSerialTemplate
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

exitCode=0

cd INSTANCEDIRECTORY
export PATH=INSTANCEDIRECTORY:${PATH}

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS EXECUTABLE ARGUMENTS STDIN > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
exitCode=$?

date +"%s" > TS_FINISH

EXITCODE

exit ${exitCode}
"""


   def __buildAppSerialScript(self):
      # setup regex's for the template
      re_jobId             = re.compile("JOBID")
      re_runName           = re.compile("RUNNAME")
      re_instanceId        = re.compile("INSTANCEID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_exe               = re.compile("EXECUTABLE")
      re_stdinput          = re.compile("STDIN")
      re_arguments         = re.compile("ARGUMENTS")
      re_useEnvironment    = re.compile("USEENVIRONMENT")
      re_environment       = re.compile("ENVIRONMENT")
      re_tsTransferred     = re.compile("TS_TRANSFERRED")
      re_tsStart           = re.compile("TS_START")
      re_tsFinish          = re.compile("TS_FINISH")
      re_timeResults       = re.compile("TIME_RESULTS")
      re_exitCode          = re.compile("EXITCODE")
      re_hubUserName       = re.compile("HUBUSERNAME")
      re_hubUserId         = re.compile("HUBUSERID")

      template = self.__makeSerialTemplate()

      if self.stdinput == "":
         template = re_stdinput.sub(self.stdinput,template)
      else:
         template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_runName.sub(self.runName,template)
      template = re_instanceDirectory.sub(self.instanceDirectory,template)
      template = re_exe.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_useEnvironment.sub(self.useEnvironment,template)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"
      template = re_environment.sub(environmentExport,template)

      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      if self.exitCode:
         template = re_exitCode.sub("echo ${exitCode} > %s\nexitCode=0" % (self.exitCode),template)
      else:
         template = re_exitCode.sub("",template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)
   
      return(template)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         appScript = ""
      else:
         appScript = self.__buildAppSerialScript()

      return(appScriptName,appScript)


