# @package      hubzero-submit-client
# @file         SubmitClient.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

__version__ = '2.5.1'

import os
import sys
import re
import shutil
import select
import time
import signal
import resource
import subprocess
import math
import stat
import csv
from errno import EPIPE

from hubzero.submit.LogMessage             import logID as log, isLogMessageFileStdout, isLogMessageFileStderr, isLoggingOn
from hubzero.submit.CommandParser          import CommandParser
from hubzero.submit.ParameterTemplate      import ParameterTemplate
from hubzero.submit.LocalBatchAppScript    import LocalBatchAppScript
from hubzero.submit.JobScanner             import JobScanner
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.TarCommand             import buildCreate as buildCreateTarCommand
from hubzero.submit.UnboundConnection      import UnboundConnection
from hubzero.submit.ClientIdAuthAttributes import ClientIdAuthAttributes

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"
EXITCODE             = ".__exit_code"

class SubmitClient:
   def __init__(self,
                configFilePath):
      self.configData     = {}
      self.configFilePath = configFilePath

      self.clientIdAuthAttributes = ClientIdAuthAttributes()
      self.serverConnection       = None
      self.commandParser          = None
      self.localWorkflowPEGASUS   = None
      self.jobScanner             = None

      self.serverData                      = {}
      self.serverData['version']           = ""
      self.serverData['authz']             = 0
      self.serverData['jobId']             = 0
      self.serverData['event']             = ""
      self.serverData['operationMode']     = 0
      self.serverData['heartbeatInterval'] = 0
      self.serverData['pegasusRC']         = ""
      self.serverData['pegasusSites']      = ""

      self.clientData                        = {}
      self.clientData['version']             = __version__
      self.clientData['operationMode']       = 0
      self.clientData['localJobId']          = "00000000"
      self.clientData['userName']            = None
      self.clientData['userId']              = None
      self.clientData['detach']              = False
      self.clientData['attachId']            = 0L
      self.clientData['pegasusExists']       = False
      self.clientData['pegasusVersion']      = None
      self.clientData['pegasusHome']         = None
      self.clientData['pegasusSetupError']   = ""
      self.clientData['reportMetrics']       = False
      self.clientData['isClientTTY']         = True
      self.clientData['readyToExit']         = False
      self.clientData['selectTimeout']       = 5*60.
      self.clientData['transferFiles']       = []
      self.clientData['inputObjects']        = {}
      self.clientData['outputObjects']       = {}
      self.clientData['isParametric']        = False
      self.clientData['localExecution']      = False
      self.clientData['childPid']            = None
      self.clientData['childReadStdout']     = None
      self.clientData['childReadStderr']     = None
      self.clientData['localStartTime']      = time.time()
      self.clientData['localFinishTime']     = None
      self.clientData['childHasExited']      = False
      self.clientData['childKillPending']    = False
      self.clientData['exitCode']            = 2
      self.clientData['jobStates']           = {}
      self.clientData['jobScanPath']         = ""
      self.clientData['jobScannerStarted']   = False
      self.clientData['enteredCommand']      = ""
      self.clientData['startDate']           = None
      self.clientData['finishDate']          = None
      self.clientData['runName']             = ""
      self.clientData['jobPath']             = os.getcwd()
      self.clientData['workDirectory']       = os.getcwd()
      self.clientData['workDirectoryInode']  = os.lstat(self.clientData['workDirectory']).st_ino
      self.clientData['importObjects']       = {}
      self.clientData['exportObjects']       = {}
      self.clientData['closePendingObjects'] = []
      self.clientData['filesToRemove']       = []
      self.clientData['emptyFilesToRemove']  = []

      signal.signal(signal.SIGINT,self.handleTerminationSignal)
      signal.signal(signal.SIGHUP,self.handleTerminationSignal)
      signal.signal(signal.SIGQUIT,self.handleDetachSignal)
      signal.signal(signal.SIGABRT,self.handleTerminationSignal)
      signal.signal(signal.SIGTERM,self.handleTerminationSignal)
      signal.signal(signal.SIGCHLD,self.handleLocalCompleteSignal)


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stdout: %s" % (message))


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stderr: %s" % (message))


   def sendUserErrorNotification(self,
                                 message):
      if   isLogMessageFileStdout():
         self.__writeToStdout(message + '\n')
      elif isLogMessageFileStderr():
         self.__writeToStderr(message + '\n')
      elif isLoggingOn():
         log(message)
      else:
         self.__writeToStderr(message + '\n')


   def killChild(self):
      if self.clientData['childPid'] and not self.clientData['childKillPending']:
         self.clientData['childKillPending'] = True
         try:
            os.kill(self.clientData['childPid'],signal.SIGINT)
            for retry in range(0,12):
               time.sleep(10)
               os.kill(self.clientData['childPid'],0)
               if not self.clientData['childKillPending']:
                  break
            if self.clientData['childKillPending']:
               os.kill(self.clientData['childPid'],signal.SIGTERM)
               for retry in range(0,6):
                  time.sleep(10)
                  os.kill(self.clientData['childPid'],0)
                  if not self.clientData['childKillPending']:
                     break
            if self.clientData['childKillPending']:
               os.kill(self.clientData['childPid'],signal.SIGKILL)
         except:
            pass


   def handleTerminationSignal(self,
                               signalNumber,
                               frame):
      if not self.clientData['readyToExit']:
         if self.clientData['localExecution']:
            if self.clientData['childPid']:
               self.killChild()
            else:
               exitCode = (1 << 7) | signalNumber
               self.clientData['exitCode'] = exitCode
         else:
            if self.serverConnection and self.serverConnection.isConnected():
               self.serverConnection.postMessage('clientSignal %d\n' % (signalNumber))
            else:
               self.clientData['readyToExit'] = True
               exitCode = (1 << 7) | signalNumber
               self.clientData['exitCode'] = exitCode
               self.exit(exitCode)


   def handleLocalCompleteSignal(self,
                                 signalNumber,
                                 frame):
      self.clientData['childKillPending'] = False


   def handleDetachSignal(self,
                          signalNumber,
                          frame):
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if not self.clientData['localExecution']:
            self.serverConnection.postMessage('detachSignal\n')


   def setupPegasus(self):
      pegasusVersion    = None
      pegasusHome       = None
      pegasusPythonPath = None
      
      try:
         from Pegasus.DAX3 import ADAG

         pegasusCommand = ['pegasus-version']
         child = subprocess.Popen(pegasusCommand,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         pegasusStdOutput,pegasusStdError = child.communicate()
         pegasusExitStatus = child.returncode
         if pegasusExitStatus == 0:
            pegasusVersion = pegasusStdOutput.strip()

            pegasusCommand = ['pegasus-config','--bin']
            child = subprocess.Popen(pegasusCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            pegasusStdOutput,pegasusStdError = child.communicate()
            pegasusExitStatus = child.returncode
            if pegasusExitStatus == 0:
               pegasusHome = os.path.dirname(pegasusStdOutput.strip())

               pegasusCommand = ['pegasus-config','--python']
               child = subprocess.Popen(pegasusCommand,
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        close_fds=True)
               pegasusStdOutput,pegasusStdError = child.communicate()
               pegasusExitStatus = child.returncode
               if pegasusExitStatus == 0:
                  pegasusPythonPath = pegasusStdOutput.strip()
      except:
         pass
      
      if not pegasusVersion or not pegasusHome or not pegasusPythonPath:
         pegasusCommand = ". %s\n" % (self.configData['useSetup']) + \
                          "use -e -r pegasus-%s\n" % (self.configData['pegasusVersion']) + \
                          "pegasus-version\n" + \
                          "pegasus-config --bin\n" + \
                          "pegasus-config --python"
         child = subprocess.Popen(pegasusCommand,shell=True,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         pegasusStdOutput,pegasusStdError = child.communicate()
         pegasusExitStatus = child.returncode
         if pegasusExitStatus == 0:
            try:
               pegasusVersion,pegasusBin,pegasusPythonPath = pegasusStdOutput.strip().split()
               pegasusHome = os.path.dirname(pegasusBin)
               if not pegasusPythonPath in sys.path:
                  sys.path.insert(0,pegasusPythonPath)
                  from Pegasus.DAX3 import ADAG
            except:
               pegasusVersion    = ""
               pegasusHome       = ""
               pegasusPythonPath = ""
         else:
            self.clientData['pegasusSetupError'] = "Pegasus version determination failed.\n%s\n" % (pegasusStdError)
      
      if pegasusVersion and pegasusHome and pegasusPythonPath:
         self.clientData['pegasusExists'] = True
      self.clientData['pegasusVersion']   = pegasusVersion
      self.clientData['pegasusHome']      = pegasusHome


   def scheduleHeartbeat(self):
      signal.signal(signal.SIGALRM,self.heartbeat)
      signal.alarm(self.serverData['heartbeatInterval'])


   def heartbeat(self,
                 signalNumber,
                 frame):
      log("Client heartbeat")
      self.connect()
      self.serverConnection.postMessage('pwd %s %ld\n' % (self.clientData['workDirectory'],
                                                                self.clientData['workDirectoryInode']))
      self.serverConnection.postMessage('jobId %d\n' % (self.serverData['jobId']))
      self.signon()
      self.serverConnection.postMessage('clientHeartbeat\n')

      self.scheduleHeartbeat()


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inClientSection = False

      fpConfig = open(self.configFilePath,'r')
      if fpConfig:
         eof = False
         while not eof:
            record = fpConfig.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   sectionPattern.match(record):
                  sectionName = sectionPattern.match(record).group(2)
                  inClientSection = (sectionName == 'client')
                  if inClientSection:
                     self.configData = {'listenURIs':[],
                                        'maximumConnectionPasses':15,
                                        'submitSSLCA':"/etc/submit/submit_server_ca.crt",
                                        'pegasusVersion':"4.1.0",
                                        'useSetup':"/etc/environ.sh"
                                       }
               elif inClientSection:
                  if keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.configData:
                        if   isinstance(self.configData[key],list):
                           self.configData[key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.configData[key],bool):
                           self.configData[key] = bool(value.lower() == 'true')
                        elif isinstance(self.configData[key],float):
                           self.configData[key] = float(value)
                        elif isinstance(self.configData[key],int):
                           self.configData[key] = int(value)
                        else:
                           self.configData[key] = value
                     else:
                        log("Undefined key = value pair %s = %s" % (key,value))
            else:
               eof = True
         fpConfig.close()

      configured = True
      if len(self.configData['listenURIs']) == 0:
         log("listenURIs missing from %s" % (self.configFilePath))
         configured = False

      self.setupPegasus()

      return(configured)


   def exit(self,
            exitCode=0):
      if self.clientData['readyToExit']:
         if not self.jobScanner:
            sys.exit(self.clientData['exitCode'])
      else:
         if self.serverConnection and self.serverConnection.isConnected():
            self.serverConnection.postMessage('clientExit %d\n' % (exitCode))
            self.clientData['readyToExit'] = True
         else:
            self.clientData['exitCode'] = exitCode
            sys.exit(exitCode)


   def getUserAttributes(self):
      return(self.clientIdAuthAttributes.getUserAttributes())


   def haveIdAuthAttributes(self):
      return(self.clientIdAuthAttributes.haveIdAuthAttributes())


   def connect(self):
      isConnected = False
      if self.haveIdAuthAttributes():
         self.serverConnection = UnboundConnection(UnboundConnection.TLSREQUIREMENTNONE,
                                                   listenURIs=self.configData['listenURIs'],
                                                   maximumConnectionPasses=self.configData['maximumConnectionPasses'],
                                                   submitSSLCA=self.configData['submitSSLCA'])
         isConnected = self.serverConnection.isConnected()

      return(isConnected)


   def sendContext(self):
      self.serverConnection.postMessage('clientVersion %s\n' % (self.clientData['version']))
      for arg in sys.argv[0:]:
         self.serverConnection.postMessageBySize('arg',arg)

      blackListEnvironmentVariables = ['DISPLAY' 'EDITOR','HOME','LOGNAME','LS_COLORS','MAIL', \
                                       'OLDPWD','RESULTSDIR','SESSIONDIR','SHELL','SHLVL', \
                                       'SVN_EDITOR','TERM','TIMEOUT','USER','VISUAL','WINDOWID', \
                                       'XTERM_LOCALE','XTERM_SHELL','XTERM_VERSION','_']
      for environmentVar in os.environ:
         if not environmentVar in blackListEnvironmentVariables:
            self.serverConnection.postMessagesBySize('var',[environmentVar,os.environ[environmentVar]])

      umask = os.umask(0)
      os.umask(umask)
      self.serverConnection.postMessage('umask 0%o\n' % (umask))
      self.serverConnection.postMessage('pwd %s %ld\n' % (self.clientData['workDirectory'],
                                                                self.clientData['workDirectoryInode']))
      self.clientData['isClientTTY'] = sys.stdin.isatty() and sys.stdout.isatty()
      self.serverConnection.postMessage('isClientTTY %d\n' % (self.clientData['isClientTTY']))
      if self.clientData['pegasusVersion']:
         self.serverConnection.postMessage('pegasusVersion %s\n' % (self.clientData['pegasusVersion']))
      else:
         self.serverConnection.postMessage('pegasusVersion %s\n' % (self.configData['pegasusVersion']))


   def signon(self):
      signonAttributes = self.clientIdAuthAttributes.getSignonAttributes()
      self.serverConnection.postMessage('userName %s\n' % (signonAttributes['userName']))
      if 'password' in signonAttributes:
         self.serverConnection.postMessage('password %s\n' % (signonAttributes['password']))
      if 'token' in signonAttributes:
         self.serverConnection.postMessage('token %s\n' % (signonAttributes['token']))
      self.serverConnection.postMessage('signon\n')

      self.clientData['userName'] = signonAttributes['userName']
      if 'userId' in signonAttributes:
         self.clientData['userId'] = signonAttributes['userId']


   def connectToServer(self):
      connectedToServer = False
      if self.haveIdAuthAttributes():
         if self.connect():
            self.sendContext()
            self.signon()
            connectedToServer = True
         else:
            self.sendUserErrorNotification("Connection to submit server failed.")
            self.exit(1)
      else:
         self.sendUserErrorNotification("Could not acquire proper authentication information.")
         sys.exit(1)

      return(connectedToServer)


   def getInputObjects(self):
      serverReader = self.serverConnection.getInputObject()
      childProcessInputObjects = []
      if self.clientData['childReadStdout']:
         childProcessInputObjects.append(self.clientData['childReadStdout'])
      if self.clientData['childReadStderr']:
         childProcessInputObjects.append(self.clientData['childReadStderr'])

      return(serverReader,
             self.clientData['inputObjects'].keys(),
             self.clientData['exportObjects'].keys(),
             childProcessInputObjects)


   def getOutputObjects(self):
      serverWriter = self.serverConnection.getOutputObject()

      return(serverWriter,
             self.clientData['importObjects'].keys())


   def readFile(self,
                fileObject):
      if   fileObject in self.clientData['inputObjects']:
         inputFile = self.clientData['inputObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.serverConnection.postMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.serverConnection.postMessagesBySize('read %s' % (inputFile),[text])
         except:
            self.exit(1)
      elif fileObject in self.clientData['exportObjects']:
         inputFile = self.clientData['exportObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.serverConnection.postMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.serverConnection.postMessagesBySize('read %s' % (inputFile),[text])
         except:
            self.exit(1)


   def writeFile(self,
                 fileObject):
      if fileObject in self.clientData['importObjects']:
         writeBuffer = self.clientData['importObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.clientData['importObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)


   def closeFile(self,
                 fileObject):
      if fileObject in self.clientData['inputObjects']:
         del self.clientData['inputObjects'][fileObject]

      if fileObject in self.clientData['exportObjects']:
         exportFile = self.clientData['exportObjects'][fileObject]
         exportPath = os.path.join(os.sep,"tmp",exportFile)
         if os.path.exists(exportPath):
            os.remove(exportPath)
         del self.clientData['exportObjects'][fileObject]

      if fileObject in self.clientData['outputObjects']:
         del self.clientData['outputObjects'][fileObject]

      transferTarFile = ""
      if fileObject in self.clientData['importObjects']:
         transferTarFile = self.clientData['importObjects'][fileObject]['path']
         del self.clientData['importObjects'][fileObject]
         if len(self.clientData['importObjects']) == 0:
            self.serverConnection.postMessage('importFilesComplete\n')

      if fileObject == self.clientData['childReadStdout']:
         self.clientData['childReadStdout'] = None
      if fileObject == self.clientData['childReadStderr']:
         self.clientData['childReadStderr'] = None

      try:
         fileObject.close()
         if transferTarFile:
            tarCommand = ['tar','xvf',os.path.join(self.clientData['workDirectory'],transferTarFile)]
            log("command: %s" % (tarCommand))
            child = subprocess.Popen(tarCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            tarStdOutput,tarStdError = child.communicate()
            tarExitStatus = child.returncode
            if tarExitStatus == 0:
               os.remove(os.path.join(self.clientData['workDirectory'],transferTarFile))
            else:
               log("Failed to extract transfer tarfile %s" % (transferTarFile))
               if tarStdOutput:
                  log(tarStdOutput)
               if tarStdError:
                  log(tarStdError)
      except:
         pass


   def checkClosePendingObjects(self):
      markedForDeletion = []
      for fileObject in self.clientData['closePendingObjects']:
         if fileObject in self.clientData['importObjects']:
            if self.clientData['importObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
      for fileObject in markedForDeletion:
         self.clientData['closePendingObjects'].remove(fileObject)
      del markedForDeletion


   def showHelpUsage(self,
                     operationMode):
      if operationMode & self.commandParser.OPERATIONMODEHELPUSAGE:
         if operationMode & (self.commandParser.OPERATIONMODEHELPTOOLS | \
                             self.commandParser.OPERATIONMODEHELPVENUES | \
                             self.commandParser.OPERATIONMODEHELPMANAGERS | \
                             self.commandParser.OPERATIONMODEVERSIONSERVER | \
                             self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR):
            showHelp = False
         else:
            showHelp = True
      else:
         showHelp = False

      return(showHelp)


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      parseCommandError = False
      contactServer = False
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.clientData['operationMode'] = self.commandParser.getOperationMode()

      if self.commandParser.validateArguments():
         if self.clientData['operationMode'] & self.commandParser.OPERATIONMODEVERSIONCLIENT:
            self.__writeToStdout("Submit client version: %s\n" % (self.clientData['version']))

         if   self.showHelpUsage(self.clientData['operationMode']):
            self.commandParser.showUsage()
         elif self.clientData['operationMode'] & (self.commandParser.OPERATIONMODERUNSTATUS | \
                                                  self.commandParser.OPERATIONMODERUNKILL | \
                                                  self.commandParser.OPERATIONMODERUNSERVER | \
                                                  self.commandParser.OPERATIONMODERUNDISTRIBUTOR):
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNPROXY:
            self.clientData['attachId'] = self.commandParser.getOption('attachId')
            contactServer = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
            exitCode = self.commandParser.setSweepParameters()
            if exitCode == 0:
               exitCode = self.commandParser.setCSVDataParameters()
               if exitCode == 0:
                  parameterCombinationCount = self.commandParser.getParameterCombinationCount()
                  if parameterCombinationCount > 0:
                     self.clientData['isParametric'] = True

                  localExecution = self.commandParser.getOption('localExecution')
                  enteredExecutable = self.commandParser.getEnteredExecutable()
                  if   self.clientData['isParametric'] and not self.clientData['pegasusExists'] and localExecution:
                     message = "Pegasus is required to execute a parametric sweep.\n"
                     self.__writeToStderr(message)
                     if self.clientData['pegasusSetupError']:
                        self.__writeToStderr(self.clientData['pegasusSetupError'])
                     parseCommandError = True
                  elif self.clientData['isParametric'] and '@:' in enteredExecutable:
                     enteredExecutable = enteredExecutable.replace('@:','')
                     message = "Parameter substitution is not allowed within the command file: %s\n" % (enteredExecutable)
                     self.__writeToStderr(message)
                     parseCommandError = True
                  elif self.clientData['isParametric'] and '@@' in enteredExecutable:
                     message = "Parameter substitution is not allowed in the command name: %s\n" % (enteredExecutable)
                     self.__writeToStderr(message)
                     parseCommandError = True
                  elif self.clientData['isParametric'] and \
                       self.commandParser.getParametricArgumentCount() == 0 and \
                       self.commandParser.getParametricInputCount() == 0:
                     message = "Parameters supplied for %d jobs, " % (parameterCombinationCount) + \
                               "but there are no template files or\ncommand substitutions that would make the jobs unique.  " + \
                               "You should prefix any template files\nwith '@:' or " + \
                               "embed parameter names (in the form '@@name') into command line arguments.\n"
                     self.__writeToStderr(message)
                     parseCommandError = True
                  else:
                     self.clientData['localExecution'] = localExecution
                     self.clientData['reportMetrics']  = self.commandParser.getOption('metrics')
                     self.clientData['detach']         = self.commandParser.getOption('detach')
                     if self.commandParser.getOption('runName'):
                        self.clientData['runName']     = self.commandParser.getOption('runName')
                     contactServer = True

                     if self.clientData['isParametric'] and self.clientData['localExecution']:
                        runDirectory = os.path.join(os.getcwd(),self.clientData['runName'])
                        if os.path.exists(runDirectory):
                           message = "Run directory %s exists\n" % (runDirectory)
                           self.__writeToStderr(message)
                           parseCommandError = True
                           contactServer     = False
               else:
                  parseCommandError = True
            else:
               parseCommandError = True
         elif self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNNONE:
            if not self.clientData['operationMode'] & self.commandParser.OPERATIONMODEVERSIONCLIENT:
               message = "No command given to execute\n"
               self.__writeToStderr(message)
               parseCommandError = True
      else:
         parseCommandError = True

      return(parseCommandError,contactServer)


   def reportDebugOutput(self):
      return(self.commandParser.getOption('debug'))


   def __updatePegasusWorkflowStatus(self):
      jobStatusReportOrders = {}
      jobStatusReportOrders['waiting']    = 5
      jobStatusReportOrders['aborted']    = 1
      jobStatusReportOrders['setup']      = 6
      jobStatusReportOrders['setting up'] = 6
      jobStatusReportOrders['failed']     = 3
      jobStatusReportOrders['executing']  = 4
      jobStatusReportOrders['finished']   = 2

      parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
      parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
      if '.' in parameterCombinationsBase:
         parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
      tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
      tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
      copyTmpFile = False

      if os.path.exists(self.clientData['jobScanPath']):
         fpCSVIn = open(self.clientData['jobScanPath'],'rb')
         if fpCSVIn:
            csvReader = csv.reader(fpCSVIn)
            fpCSVOut = open(tmpParameterCombinationsPath,'wb')
            if fpCSVOut:
               csvWriter = csv.writer(fpCSVOut)
               csvWriter.writerow(('# command: ' + self.clientData['enteredCommand'],))
               csvWriter.writerow(('# started: ' + self.clientData['startDate'],))
               if self.clientData['finishDate']:
                  csvWriter.writerow(('# finished: ' + self.clientData['finishDate'],))
               nCompleted = 0
               for instance in self.clientData['jobStates']:
                  jobStatusState       = self.clientData['jobStates'][instance]
                  jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
                  if jobStatusState in ['finished','failed','aborted']:
                     nCompleted += 1
               csvWriter.writerow(('# completed: %d/%d jobs' % (nCompleted,len(self.clientData['jobStates'])),))
               parameterNames = []
               while len(parameterNames) <= 1:
                  parameterNames = csvReader.next()
               csvWriter.writerow(parameterNames)
               parameterCombinations = {}
               for parameterCombination in csvReader:
                  instance = int(parameterCombination[0])
                  if instance in self.clientData['jobStates']:
                     jobStatusState       = self.clientData['jobStates'][instance]
                     jobStatusReportOrder = jobStatusReportOrders[jobStatusState]
                     parameterCombination[1] = jobStatusState
                     if not jobStatusReportOrder in parameterCombinations:
                        parameterCombinations[jobStatusReportOrder] = []
                     parameterCombinations[jobStatusReportOrder].append(parameterCombination)
               jobStatusReportOrders = parameterCombinations.keys()
               jobStatusReportOrders.sort()
               for jobStatusReportOrder in jobStatusReportOrders:
                  for parameterCombination in parameterCombinations[jobStatusReportOrder]:
                     csvWriter.writerow(parameterCombination)
               fpCSVOut.close()
               copyTmpFile = True
            fpCSVIn.close()

      if copyTmpFile:
         os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])


   @staticmethod
   def __which(program):
      def isExe(fpath):
         return(os.path.isfile(fpath) and os.access(fpath,os.X_OK))

      fpath,fname = os.path.split(program)
      if fpath:
         if isExe(program):
            return(program)
      else:
         for path in os.environ["PATH"].split(os.pathsep):
            exePath = os.path.join(path,program)
            if isExe(exePath):
               return(exePath)

      return(None)


   def __getStdInputFile(self):
      stdInputFile = '/dev/null'
      if not sys.stdin.isatty():
         toCheck = []
         stdinFd = sys.stdin.fileno()
         toCheck.append(stdinFd)
         try:
            ready = select.select(toCheck,[],[],0.) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if stdinFd in ready[0]:
            content = sys.stdin.read()
            if content != "":
               stdInputFile = os.path.join(self.clientData['jobPath'],".__%s.stdin" % (self.clientData['localJobId']))
               stdinFile = open(stdInputFile,'w')
               stdinFile.write(content)
               stdinFile.close()
               try:
                  si = open('/dev/tty')
                  os.dup2(si.fileno(),sys.stdin.fileno())
               except:
                  pass
            del content

      return(stdInputFile)


   def startLocalParametric(self):
      from LocalWorkflowPEGASUS import LocalWorkflowPEGASUS
      exitCode = 0

      useEnvironment = ""
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = filter(reChoice.search,os.environ.keys())
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         useEnvironment = ". %s\n" % (self.configData['useSetup'])
         if "ENVIRON_CONFIG_DIRS" in os.environ:
            useEnvironment += "export ENVIRON_CONFIG_DIRS=\"" + os.environ['ENVIRON_CONFIG_DIRS'] + "\"\n"
         for environmentVar in environmentVars:
            useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

      environment = ""
      if self.commandParser.getOption('environment'):
         for environmentVariableValue in self.commandParser.getOption('environment'):
            environmentVariable = ""
            value               = ""
            nParts = len(environmentVariableValue.split('='))
            if   nParts == 1:
               environmentVariable = environmentVariableValue.strip()
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
            elif nParts == 2:
               environmentVariable,value = environmentVariableValue.split('=')
               environmentVariable = environmentVariable.strip()
               value               = value.strip()
               if value == "":
                  if environmentVariable in os.environ:
                     value = os.environ[environmentVariable]
            if environmentVariable == "" or value == "":
               log("Invalid environment variable %s specified." % (environmentVariableValue))
               if not exitCode:
                  exitCode = 1
            else:
               environment += environmentVariable + "=" + value + " "
         environment = environment.strip()

      clientInputFiles = []
      if self.commandParser.getOption('inputFiles'):
         for inputFile in self.commandParser.getOption('inputFiles'):
            clientInputFiles.append(inputFile)

      self.clientData['enteredCommand'] = self.commandParser.getEnteredCommand()
      enteredCommandArguments = self.commandParser.getEnteredCommandArguments()

      self.clientData['jobPath'] = os.path.join(os.getcwd(),self.clientData['runName'])
      if not os.path.isdir(self.clientData['jobPath']):
         os.mkdir(self.clientData['jobPath'])
      self.clientData['jobScanPath'] = os.path.join(self.clientData['jobPath'],'parameterCombinations.csv')
      self.commandParser.writeParameterCombinations(self.clientData['jobScanPath'])

      self.stdinput = self.__getStdInputFile()

      inputError = False
      executableExistsError = False
      nInstanceIdDigits = max(2,int(math.log10(self.commandParser.getParameterCombinationCount())+1))
      instance = 0
      for substitutions in self.commandParser.getNextParameterCombinationFromCSV(self.clientData['jobScanPath']):
         instance += 1
         instanceId = str(instance).zfill(nInstanceIdDigits)
         self.clientData['jobStates'][instance] = 'setup'

         instanceDirectory = os.path.join(self.clientData['jobPath'],instanceId)
         if not os.path.isdir(instanceDirectory):
            os.makedirs(instanceDirectory)
         instanceInputsPath = instanceDirectory

         exitCode = 0

         enteredCommand = self.commandParser.getEnteredCommand()
         template = ParameterTemplate(enteredCommand)
         try:
            enteredCommand = template.substitute(substitutions)
         except KeyError,err:
            self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
            inputError = True

         inputFiles = []
         for inputFile in clientInputFiles:
            template = ParameterTemplate(inputFile)
            try:
               inputFile = template.substitute(substitutions)
            except KeyError,err:
               self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
               inputError = True
            inputFiles.append(inputFile)

         if len(enteredCommandArguments) > 1:
            for arg in enteredCommandArguments[1:]:
               if not arg.startswith('-'):
                  template = ParameterTemplate(arg)
                  try:
                     arg = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     inputError = True
                  if os.path.isfile(arg.replace('@:','')):
                     if not arg in inputFiles:
                        inputFiles.append(arg)

         substitutedInputFiles = []
         for inputFile in inputFiles:
            if '@:' in inputFile:
               inputFile = inputFile.replace('@:','')
               fpInputFile = open(inputFile,'r')
               if fpInputFile:
                  inputText = fpInputFile.readlines()
                  fpInputFile.close()
                  template = ParameterTemplate(''.join(inputText))
                  try:
                     inputText = template.substitute(substitutions)
                  except KeyError,err:
                     self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                     inputError = True
                  inputName = os.path.basename(inputFile)
                  inputPath = os.path.join(instanceInputsPath,inputName)
                  fpInputPath = open(inputPath,'w')
                  if fpInputPath:
                     fpInputPath.writelines(inputText)
                     fpInputPath.close()
                     substitutedInputFiles.append(inputPath)
            else:
               substitutedInputFiles.append(inputFile)
         del inputFiles
         inputFiles = substitutedInputFiles

         remoteArgs = []

         executable = self.__which(enteredCommandArguments[0])
         if executable:
            executable = os.path.realpath(executable)
            remoteArgs.append(executable)
         else:
            if not executableExistsError:
               self.__writeToStderr("Specified command %s is not in your PATH\n" % (enteredCommandArguments[0]))
            inputError = True
            # allow completion of input checks
            remoteArgs.append(enteredCommandArguments[0])
            executable = enteredCommandArguments[0]
            executableExistsError = True

         for arg in enteredCommandArguments[1:]:
            if arg.startswith('-'):
               remoteArgs.append(arg)
            else:
               template = ParameterTemplate(arg)
               try:
                  arg = template.substitute(substitutions)
               except KeyError,err:
                  self.__writeToStderr("Pattern substitution failed for @@%s\n" % (err[0]))
                  inputError = True
               if '@:' in arg:
                  if os.path.isfile(arg.replace('@:','')):
                     inputName = os.path.basename(arg.replace('@:',''))
                     remoteArgs.append(inputName)
               else:
                  if os.path.isfile(arg):
                     rarg = os.path.abspath(arg)
                     remoteArgs.append(rarg)
                  else:
                     remoteArgs.append(arg)

         remoteCommand = " ".join(remoteArgs)
         arguments     = remoteArgs[1:]

         timeHistoryLogs = {}
         timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
         timeHistoryLogs['exitCode']             = "%s.%s_%s" % (EXITCODE,self.clientData['localJobId'],instanceId)

         localBatchAppScript = LocalBatchAppScript(self.clientData['userName'],self.clientData['userId'],
                                                   self.clientData['runName'],instanceDirectory,
                                                   self.clientData['localJobId'],instanceId,
                                                   executable,self.stdinput,arguments,useEnvironment,environment,
                                                   timeHistoryLogs)
         appScriptName,appScript = localBatchAppScript.buildAppScript()
         appScriptPath = os.path.join(self.clientData['jobPath'],instanceId,appScriptName)

         fpAppScript = open(appScriptPath,'w')
         if fpAppScript:
            fpAppScript.write(appScript)
            fpAppScript.close()
            os.chmod(appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            self.clientData['filesToRemove'].append(appScriptPath)
         else:
            log("could not open %s for writing" % (appScriptPath))
            exitCode = 1
         del localBatchAppScript

         stdFile = os.path.join(instanceDirectory,"%s_%s.stderr" % (self.clientData['runName'],instanceId))
         self.clientData['emptyFilesToRemove'].append(stdFile)

      nInstances      = instance
      executeInstance = nInstances+1

      instance = 0
      instanceId = str(instance).zfill(nInstanceIdDigits)
      timeHistoryLogs = {}
      timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
      timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)

      pegasusTemplates = {}
      if self.serverData['pegasusRC'] != "":
         pegasusTemplates['rc'] = self.serverData['pegasusRC']
      if self.serverData['pegasusSites'] != "":
         pegasusTemplates['sites'] = self.serverData['pegasusSites']

      self.localWorkflowPEGASUS = LocalWorkflowPEGASUS(self.clientData['userName'],self.clientData['userId'],
                                                       self.clientData['runName'],
                                                       self.clientData['jobPath'],self.clientData['jobPath'],
                                                       self.configData['useSetup'],
                                                       self.clientData['pegasusVersion'],self.clientData['pegasusHome'],
                                                       self.clientData['localJobId'],pegasusTemplates,timeHistoryLogs)
      batchScriptName,batchScript = self.localWorkflowPEGASUS.buildWorkflowScript()
      batchScriptPath = os.path.join(self.clientData['jobPath'],batchScriptName)

      fpBatchScript = open(batchScriptPath,'w')
      if fpBatchScript:
         fpBatchScript.write(batchScript)
         fpBatchScript.close()
         os.chmod(batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         self.clientData['filesToRemove'].append(batchScriptPath)
      else:
         log("could not open %s for writing" % (batchScriptPath))
         exitCode = 1

      enteredCommand = self.commandParser.getEnteredCommand()

      stdFile = os.path.join(self.clientData['jobPath'],"%s.stdout" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)
      stdFile = os.path.join(self.clientData['jobPath'],"%s.stderr" % (self.clientData['runName']))
      self.clientData['emptyFilesToRemove'].append(stdFile)

      self.clientData['startDate']      = time.strftime("%a %b %e %X %Z %Y")
      self.clientData['localStartTime'] = time.time()
      self.__updatePegasusWorkflowStatus()

      if inputError:
         self.exit(1)
      else:
         self.clientData['childHasExited'] = False
         child = subprocess.Popen(batchScriptPath,shell=False,bufsize=1,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr

         self.serverConnection.postMessage('localWait\n')

         if not sys.stdin in self.clientData['inputObjects']:
            self.clientData['inputObjects'][sys.stdin] = '#STDIN#'

         self.clientData['isClientTTY'] = sys.stdin.isatty() and sys.stdout.isatty()
         if self.clientData['isClientTTY']:
            self.jobScanner = JobScanner(self.clientData['jobScanPath'])
            self.clientData['jobScannerStarted'] = True
            if not self.jobScanner.start():
               self.jobScanner.finish()
               self.jobScanner = None


   def startLocalExecute(self):
      self.clientData['localStartTime'] = time.time()
      commandArguments = self.commandParser.getEnteredCommandArguments()
      os.environ['SUBMIT_JOB'] = "%s" % (self.serverData['jobId'])
      try:
         self.clientData['childHasExited'] = False
         child = subprocess.Popen(commandArguments,shell=False,bufsize=1,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  env=os.environ,
                                  close_fds=True)
         self.clientData['childPid']        = child.pid
         self.clientData['childReadStdout'] = child.stdout
         self.clientData['childReadStderr'] = child.stderr
         self.serverConnection.postMessage('localWait\n')
         if self.serverData['heartbeatInterval'] > 0:
            self.scheduleHeartbeat()
      except OSError,err:
         self.__writeToStderr("Failed to invoke %s: %s\n" % (commandArguments[0],err.args[1]))
         self.exit(err.args[0])


   def startLocal(self):
      if self.clientData['isParametric']:
         self.startLocalParametric()
      else:
         self.startLocalExecute()


   def finishLocalParametric(self):
      if os.path.isfile(self.stdinput):
         os.remove(self.stdinput)

      parameterCombinationCount = self.commandParser.getParameterCombinationCount()
      if parameterCombinationCount > 0:
         nInstanceIdDigits = max(2,int(math.log10(parameterCombinationCount)+1))
         for instance in xrange(1,parameterCombinationCount+1):
            instanceId = str(instance).zfill(nInstanceIdDigits)

            jobStatistic = JobStatistic(1)
            timeFile = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobStartedTime',timePath)
            timeFile = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
            timePath = os.path.join(self.clientData['jobPath'],instanceId,timeFile)
            jobStatistic.recordTime('jobFinshedTime',timePath)
            timerFile = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
            timerPath = os.path.join(self.clientData['jobPath'],instanceId,timerFile)
            jobStatistic.recordTimer(timerPath)
            jobStatistic.setWaitingTime()
            jobStatistic.setElapsedRunTime()

            exitCodeFile = "%s.%s_%s" % (EXITCODE,self.clientData['localJobId'],instanceId)
            exitCodePath = os.path.join(self.clientData['jobPath'],instanceId,exitCodeFile)
            if os.path.exists(exitCodePath):
               os.remove(exitCodePath)

            exitCode       = jobStatistic['exitCode']
            cpuTime        = jobStatistic['userTime']+jobStatistic['sysTime']
            elapsedRunTime = jobStatistic['elapsedRunTime']

            workDirectoryName = os.path.join(self.clientData['jobPath'],'work')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            workDirectoryName = os.path.join(self.clientData['jobPath'],'scratch')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            for ftype in '.dax','_sites.xml','.pegasusrc','_tc.txt':
               pegasusFile = os.path.join(self.clientData['jobPath'],"%s_%s%s" % (self.clientData['localJobId'],instanceId,ftype))
               if os.path.isfile(pegasusFile):
                  os.remove(pegasusFile)
               else:
                  pegasusFile = os.path.join(self.clientData['jobPath'],"%s%s" % (self.clientData['localJobId'],ftype))
                  if os.path.isfile(pegasusFile):
                     os.remove(pegasusFile)

            self.serverConnection.postMessage('localMetrics %d %d %f %f\n' % (instance,exitCode,cpuTime,elapsedRunTime))
            del jobStatistic

         instance = 0
         instanceId = str(instance).zfill(nInstanceIdDigits)
         jobStatistic = JobStatistic(1)
         timeFile = "%s.%s_%s" % (TIMESTAMPSTART,self.clientData['localJobId'],instanceId)
         timePath = os.path.join(self.clientData['jobPath'],timeFile)
         jobStatistic.recordTime('jobStartedTime',timePath)
         timeFile = "%s.%s_%s" % (TIMESTAMPFINISH,self.clientData['localJobId'],instanceId)
         timePath = os.path.join(self.clientData['jobPath'],timeFile)
         jobStatistic.recordTime('jobFinshedTime',timePath)
         timerFile = "%s.%s_%s" % (TIMERESULTS,self.clientData['localJobId'],instanceId)
         timerPath = os.path.join(self.clientData['jobPath'],timerFile)
         jobStatistic.recordTimer(timerPath)
         jobStatistic.setWaitingTime()
         jobStatistic.setElapsedRunTime()

         exitCode       = jobStatistic['exitCode']
         cpuTime        = jobStatistic['userTime']+jobStatistic['sysTime']
         elapsedRunTime = jobStatistic['elapsedRunTime']

         self.serverConnection.postMessage('localMetrics %d %d %f %f\n' % (instance,exitCode,cpuTime,elapsedRunTime))
         del jobStatistic

         if not self.clientData['isClientTTY']:
            self.__writeToStdout("Simulations complete. Results are stored in directory %s\n" % (self.clientData['jobPath']))


   def finishLocal(self):
      if self.clientData['isParametric']:
         self.finishLocalParametric()

      for fileToRemove in self.clientData['filesToRemove']:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.clientData['emptyFilesToRemove']:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)


   def scanLocalOutputFile(self,
                           outputFileObject):
      outChunk = os.read(outputFileObject.fileno(),1024)
      if len(outChunk) > 0:
         if self.clientData['isParametric']:
            update = False
            jobStartPattern    = re.compile('.* Executing JOB ([^ ]*) *-n *([^ ]*) *-N *([^ ]*).*/([^ ]*)')
            jobCompletePattern = re.compile('.* JOB *([0-9]+)_([0-9]+)\.sh_[A-Za-z0-9]*_[0-9]*')
            for record in outChunk.split('\n'):
               matchObj = jobStartPattern.match(record)
               if matchObj:
                  if matchObj.group(2) == matchObj.group(4):
                     instance = int(matchObj.group(3).split('_')[-1])
                     self.clientData['jobStates'][instance] = 'executing'
                     update = True
               else:
                  matchObj = jobCompletePattern.match(record)
                  if matchObj:
                     instance = int(matchObj.group(2))
                     instanceExitCode = 0
                     exitCodeFile = "%s.%s_%s" % (EXITCODE,matchObj.group(1),matchObj.group(2))
                     exitCodePath = os.path.join(self.clientData['jobPath'],matchObj.group(2),exitCodeFile)
                     if os.path.exists(exitCodePath):
                        fpExitCode = open(exitCodePath,'r')
                        if fpExitCode:
                           instanceExitCode = int(fpExitCode.readline())
                           fpExitCode.close()
                     if instanceExitCode:
                        self.clientData['jobStates'][instance] = 'failed'
                     else:
                        self.clientData['jobStates'][instance] = 'finished'
                     update = True
            if update:
               self.__updatePegasusWorkflowStatus()
         else:
            if outputFileObject == self.clientData['childReadStdout']:
               self.__writeToStdout(outChunk)
            if outputFileObject == self.clientData['childReadStderr']:
               self.__writeToStderr(outChunk)
      else:
         self.closeFile(outputFileObject)


   def mapInputFiles(self):
      if self.clientData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         inputFiles = []
         enteredCommandArguments = self.commandParser.getEnteredCommandArguments()
         if self.clientData['isParametric']:
            for substitutions in self.commandParser.getNextParameterCombination():
               if self.commandParser.getOption('inputFiles'):
                  for inputFile in self.commandParser.getOption('inputFiles'):
                     template = ParameterTemplate(inputFile)
                     actualInputFile = template.substitute(substitutions).replace('@:','')
                     if not actualInputFile in inputFiles:
                        inputFiles.append(actualInputFile)

               if len(enteredCommandArguments) > 0:
                  for arg in enteredCommandArguments:
                     template = ParameterTemplate(arg)
                     arg = template.substitute(substitutions)
                     arglets = re.split('( |=)',arg)
                     for arglet in arglets:
                        if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                           actualInputFile = arglet.replace('@:','')
                           if os.path.isfile(actualInputFile):
                              if not actualInputFile in inputFiles:
                                 inputFiles.append(actualInputFile)
         else:
            if self.commandParser.getOption('inputFiles'):
               for inputFile in self.commandParser.getOption('inputFiles'):
                  inputFiles.append(inputFile)

            if len(enteredCommandArguments) > 0:
               for arg in enteredCommandArguments:
                  arglets = re.split('( |=)',arg)
                  for arglet in arglets:
                     if not arglet.startswith('-') and arglet != ' ' and arglet != '=':
                        if os.path.isfile(arglet) or os.path.isdir(arglet):
                           if not arglet in inputFiles:
                              inputFiles.append(arglet)

         for inputFile in inputFiles:
            try:
               inode = os.lstat(inputFile).st_ino
               self.serverConnection.postMessageBySize('inputFile %ld' % (inode),os.path.abspath(inputFile))
            except OSError,err:
               self.__writeToStderr("%s: %s\n" % (inputFile,err.args[1]))
               self.exit(err.args[0])

      self.serverConnection.postMessage('inputFileInodesSent\n')


   def wait(self):
      if self.serverConnection.isConnected():
         self.serverConnection.postMessage('exit\n')


   def processServerRequests(self):
      message = self.serverConnection.pullMessage(0)
      while message:
         args = message.split()
         log("server request = %s" % (args[0]))

         if   args[0] == 'serverVersion':
            self.serverData['version'] = args[1]
         elif args[0] == 'message':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  log(messageText)
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'authz':
            self.serverData['authz'] = int(args[1])
            if self.serverData['authz'] == 0:
               log("Session authentication failed")
               self.clientIdAuthAttributes.clearSessionToken()
               self.getUserAttributes()
               if self.haveIdAuthAttributes():
                  self.signon()
            else:
               if self.clientData['attachId'] == 0L:
                  if self.clientData['localExecution']:
                     self.serverConnection.postMessage('startLocal\n')
                  else:
                     self.serverConnection.postMessage('setupRemote\n')
               else:
                  log("Do something for Proxy")
         elif args[0] == 'writeStdout':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  self.__writeToStdout(messageText)
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'writeStderr':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  self.__writeToStderr(messageText)
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'write':
            serverOutputFile  = args[1]
            textLength = int(args[2])
            if textLength > 0:
               text = self.serverConnection.pullMessage(textLength)
               if len(text) > 0:
                  found = False
                  if not found:
                     for importObject in self.clientData['importObjects']:
                        if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                           self.clientData['importObjects'][importObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     log("File object not found to match %s" % (serverOutputFile))
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'close':
            serverOutputFile  = args[1]
            found = False
            for importObject in self.clientData['importObjects']:
               if self.clientData['importObjects'][importObject]['path'] == serverOutputFile:
                  self.clientData['closePendingObjects'].append(importObject)
                  found = True
                  break
            if not found:
               log("File object not found to match %s" % (serverOutputFile))
         elif args[0] == 'jobId':
            self.serverData['jobId'] = int(args[1])
            if self.serverData['jobId'] == 0:
               self.exit(1)
            else:
               self.clientData['localJobId'] = "%08d" % (self.serverData['jobId'])
               if not self.clientData['runName']:
                  self.clientData['runName'] = self.clientData['localJobId']
            if self.clientData['reportMetrics']:
               self.__writeToStderr("=SUBMIT-METRICS=> job=%s\n" % (self.clientData['localJobId']))
         elif args[0] == 'runName':
            self.clientData['runName'] = args[1]
            self.serverData['jobId'] = int(args[2])
            self.clientData['localJobId'] = "%08d" % (self.serverData['jobId'])
         elif args[0] == 'event':
            self.serverData['event'] = args[1]
         elif args[0] == 'operationMode':
            self.serverData['operationMode'] = int(args[1])
         elif args[0] == 'heartbeatInterval':
            self.serverData['heartbeatInterval'] = int(args[1])
         elif args[0] == 'pegasusRC':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  self.serverData['pegasusRC'] = messageText
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'pegasusSites':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  self.serverData['pegasusSites'] = messageText
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'jobScanUpdate':
            messageLength = int(args[1])
            if messageLength > 0:
               messageText = self.serverConnection.pullMessage(messageLength)
               if len(messageText) > 0:
                  if self.clientData['jobScanPath'] == "":
                     jobScanPath = os.path.join(self.clientData['workDirectory'],self.clientData['runName'])
                     if not os.path.exists(jobScanPath):
                        os.makedirs(jobScanPath)
                     jobScanPath = os.path.join(jobScanPath,'parameterCombinations.csv')
                     self.clientData['jobScanPath'] = jobScanPath
                  try:
                     parameterCombinationsDir  = os.path.dirname(self.clientData['jobScanPath'])
                     parameterCombinationsBase = os.path.basename(self.clientData['jobScanPath'])
                     if '.' in parameterCombinationsBase:
                        parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
                     tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
                     tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
                     fpJobScan = open(tmpParameterCombinationsPath,'w')
                     if fpJobScan:
                        fpJobScan.write(messageText)
                        fpJobScan.close()
                        os.rename(tmpParameterCombinationsPath,self.clientData['jobScanPath'])
                  except:
                     pass
                  if os.path.exists(self.clientData['jobScanPath']):
                     if not self.jobScanner and not self.clientData['jobScannerStarted']:
                        self.jobScanner = JobScanner(self.clientData['jobScanPath'])
                        self.clientData['jobScannerStarted'] = True
                        if not self.jobScanner.start():
                           self.jobScanner.finish()
                           self.jobScanner = None
               else:
                  self.serverConnection.pushMessage(message + '\n')
                  break
         elif args[0] == 'serverReadyForIO':
            if self.clientData['localExecution']:
               self.startLocal()
            else:
               self.clientData['inputObjects'][sys.stdin] = '#STDIN#'
               if self.clientData['detach']:
                  self.serverConnection.postMessage('detachSignal\n')
         elif args[0] == 'readyToDetach':
            self.serverConnection.postMessage('detach\n')
            if self.jobScanner:
               self.jobScanner.finish()
               self.jobScanner = None
            self.clientData['readyToExit'] = True
            self.clientData['exitCode']    = 0
            notificationMessage = "Detaching from run %d.\n" % (self.serverData['jobId']) + \
                                  "    Check run status with the command: submit --status %d\n" % (self.serverData['jobId']) + \
                                  "       Terminate run with the command: submit --kill %d\n" % (self.serverData['jobId']) + \
                                  "   Re-connect to run with the command: submit --attach %d\n" % (self.serverData['jobId'])
            self.__writeToStdout(notificationMessage)
         elif args[0] == 'attached':
            notificationMessage = "Attached to run %d.\n" % (self.clientData['attachId'])
            self.__writeToStdout(notificationMessage)
            self.serverConnection.postMessage('clientVersion %s\n' % (self.clientData['version']))
         elif args[0] == 'serverReadyForInputMapping':
            self.mapInputFiles()
         elif args[0] == 'addExportFile':
            clientInputFile = args[1]
            self.clientData['transferFiles'].append(clientInputFile)
         elif args[0] == 'exportFiles':
            transferTarFile = args[1]
            transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
            tarCommand = buildCreateTarCommand(transferTarPath,self.clientData['transferFiles'],useAbsolutePaths=True)
            log("command = " + str(tarCommand))
            try:
               child = subprocess.Popen(tarCommand,
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        close_fds=True)
               tarStdOutput,tarStdError = child.communicate()
               tarExitStatus = child.returncode
               if tarExitStatus != 0:
                  log("Failed to write transfer tarfile %s" % (transferTarPath))
                  if tarStdOutput:
                     log(tarStdOutput)
                  if tarStdError:
                     log(tarStdError)
                  exitCode = 1
               else:
                  fp = open(transferTarPath,'r')
                  if fp:
                     self.clientData['exportObjects'][fp] = transferTarFile
                  else:
                     log("input file open failed: %s" % (transferTarPath))
            except OSError,err:
               log("Failed to to create transfer tarfile %s" % (transferTarPath))
               log(err.args[1])
         elif args[0] == 'exportFilesComplete':
            self.serverConnection.postMessage('startRemote\n')
         elif args[0] == 'importFile':
            transferTarFile = args[1]
            if transferTarFile == "None":
               self.serverConnection.postMessage('importFilesComplete\n')
            else:
               transferTarPath = os.path.join(self.clientData['workDirectory'],transferTarFile)
               try:
                  fp = open(transferTarPath,'w')
                  if fp:
                     self.clientData['importObjects'][fp]           = {}
                     self.clientData['importObjects'][fp]['path']   = transferTarFile
                     self.clientData['importObjects'][fp]['buffer'] = ""
                  else:
                     log("input file open failed: %s" % (transferTarPath))
               except OSError,err:
                  log("Failed to to create transfer tarfile %s" % (transferTarPath))
                  log(err.args[1])
         elif args[0] == 'serverExit':
            exitCode = int(args[1])
            self.clientData['readyToExit'] = True
            self.clientData['exitCode']    = exitCode
            self.exit(exitCode)
         elif args[0] == 'exit':
            exitCode = int(args[1])
            self.exit(exitCode)
         elif args[0] == 'wait':
            self.wait()
         else:
            log("Discarded server message: %s" % (message))

         message = self.serverConnection.pullMessage(0)


   def run(self):
      while True:
         try:
            serverReader,inputObjects,exportObjects,childProcessInputObjects = self.getInputObjects()
            serverWriter,importObjects                                       = self.getOutputObjects()
            if self.clientData['childHasExited'] or self.clientData['readyToExit']:
               selectTimeout = 0.1
            elif self.jobScanner:
               selectTimeout = 10.
            else:
               selectTimeout = self.clientData['selectTimeout']
            readyReaders,readyWriters,readyExceptions = select.select(serverReader+inputObjects+ \
                                                                                   exportObjects+childProcessInputObjects,
                                                                      serverWriter+importObjects,[],
                                                                      selectTimeout)
         except select.error,err:
            readyReaders = []
            readyWriters = []
#
# If the timeout has occurred (nothing to read/write) send a keepalive.
#
         if readyReaders == [] and readyWriters == []:
            self.serverConnection.postMessage('null\n')
            if   self.jobScanner:
               wantToQuit,jobsFinished = self.jobScanner.processInput()
               if wantToQuit:
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],self.clientData['runName'])
                  self.__writeToStdout("Simulations complete. Results are being stored in directory %s\n" % (resultsPath))
            elif self.clientData['readyToExit']:
               self.exit()

         for readyReader in readyReaders:
            if   readyReader in serverReader:
               self.serverConnection.receiveMessage()
            elif readyReader in childProcessInputObjects:
               self.scanLocalOutputFile(readyReader)
            elif self.jobScanner and readyReader == sys.stdin:
               wantToQuit,jobsFinished = self.jobScanner.processInput()
               if wantToQuit:
                  self.jobScanner.finish()
                  self.jobScanner = None
                  if self.clientData['localExecution']:
                     resultsPath = self.clientData['jobPath']
                  else:
                     resultsPath = os.path.join(self.clientData['jobPath'],self.clientData['runName'])
                  self.__writeToStdout("Simulations complete. Results are being stored in directory %s\n" % (resultsPath))
                  if not jobsFinished:
                     os.kill(os.getpid(),signal.SIGINT)
            else:
               self.readFile(readyReader)

         self.processServerRequests()

         for readyWriter in readyWriters:
            if   readyWriter in serverWriter:
               self.serverConnection.sendMessage()
            elif readyWriter in importObjects:
               self.writeFile(readyWriter)
         self.checkClosePendingObjects()

         if self.clientData['childPid']:
            pid = 0
            try:
               (pid,exitCode) = os.waitpid(self.clientData['childPid'],os.WNOHANG)
               if exitCode != 0:
                  if   os.WIFSIGNALED(exitCode):
                     exitCode = (1 << 7) | os.WTERMSIG(exitCode)
                  else:
                     if os.WIFEXITED(exitCode):
                        exitCode = os.WEXITSTATUS(exitCode)
            except:
               try:
                  os.kill(self.clientData['childPid'],0)
               except:
                  pid = self.clientData['childPid']
                  exitCode = 1

            if pid != 0:
               self.clientData['childPid']       = None
               self.clientData['exitCode']       = exitCode
               self.clientData['childHasExited'] = True

               rusage = resource.getrusage(resource.RUSAGE_CHILDREN)
               self.clientData['localFinishTime'] = time.time()
               realTime = self.clientData['localFinishTime']-self.clientData['localStartTime']
               cpuTime  = rusage[0] + rusage[1]
               self.clientData['finishDate'] = time.strftime("%a %b %e %X %Z %Y")
               if self.clientData['isParametric']:
                  self.__updatePegasusWorkflowStatus()

               if self.clientData['reportMetrics']:
                  message =  "=SUBMIT-METRICS=>"
                  message += " job=%d" % (self.serverData['jobId'])
                  message += " venue=local"
                  message += " status=%d" % (self.clientData['exitCode'])
                  message += " cputime=%f" % (cpuTime)
                  message += " realtime=%f" % (realTime)
                  self.__writeToStderr("%s\n" % (message))

               self.connect()
               self.serverConnection.postMessage('pwd %s %ld\n' % (self.clientData['workDirectory'],
                                                                         self.clientData['workDirectoryInode']))
               self.serverConnection.postMessage('jobId %d\n' % (self.serverData['jobId']))
               self.serverConnection.postMessage('operationMode %d\n' % (self.serverData['operationMode']))
               self.signon()

               try:
                  submitApplicationRevision = os.environ["SUBMIT_APPLICATION_REVISION"]
               except:
                  submitApplicationRevision = ""
               if submitApplicationRevision != "":
                  self.serverConnection.postMessage('event %s\n' % (submitApplicationRevision))
               else:
                  self.serverConnection.postMessage('event %s\n' % (self.serverData['event']))

               self.finishLocal()

               self.serverConnection.postMessage('localExit %d %f %f\n' % (exitCode,cpuTime,realTime))
               self.clientData['readyToExit'] = True


