# @package      hubzero-python
# @file         group.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import ldap.modlist as modlist
import datetime
import hubzero.config.passwords
import hubzero.utilities.misc
import hubzero.config.ldapconfig
import hubzero.config.webconfig
import hubzero.data.db
import ldap
import os
import subprocess


def groupCleanup(db, groupname):
	"""
	We don't use foriegn keys, so messy cleanup here when you delete a group
	"""

	sql = 'SELECT gidNumber FROM `jos_xgroups` WHERE `cn` = %s'
	data = (groupname)
	gidNumber = db.query_selectscalar(sql, data)

	# don't bother if the group doesn't exist
	if not gidNumber:
		return

	sql = "DELETE FROM `jos_xgroups_applicants` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_events` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_inviteemails` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_invitees` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_managers` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_memberoption` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_members` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_reasons` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_roles` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_tracperm` WHERE `group_id`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)


def groupExists(groupname):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()

	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	if _groupExists(db, groupname):
		return True
	else:
		return False

def groupNumberLookup(groupName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()

	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	return _groupExists(db, groupName)


def _groupExists(db, groupname):
	sql = 'SELECT gidNumber FROM `jos_xgroups` WHERE `cn` = %s'
	data = (groupname)
	gidNumber = db.query_selectscalar(sql, data)

	# will be None if none exists
	return gidNumber


def _del_jos_xgroups(db, name):
	sql = "DELETE FROM jos_xgroups WHERE `CN`=%s "
	data = (name)
	db.query_rowcount(sql, data)


def _del_jos_xgroups_members(db, gidNumber, uidNumber):
	sql = "DELETE FROM jos_xgroups_members WHERE `gidNumber`=%s AND `uidNumber`=%s"
	data = (gidNumber, uidNumber)
	db.query_rowcount(sql, data)


def del_jos_xgroups_members(gidNumber, uidNumber):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	_del_jos_xgroups_members(db, gidNumber, uidNumber)


def add_ldap_group(cn, groupDescription, gidNumber=0, members = {}):
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	if not gidNumber:
		gidNumber = _getnextLDAPGIDNumber()

	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + cn + ",ou=groups," + hubLDAPBaseDN
	attrs = {}
	attrs['objectclass'] = ['posixGroup', 'top']
	attrs['cn'] = cn
	attrs['gidNumber'] = str(gidNumber)
	attrs['description'] = groupDescription
	attrs['memberuid'] = members

	ldif = ldap.modlist.addModlist(attrs)

	try:
		l.add_s(dn, ldif)
	except ldap.ALREADY_EXISTS:
		print "Group " + cn + " already exists"
		return 0

	# to avoid delays in cache
	#os.system("/etc/init.d/nscd restart")
	#os.system("/etc/init.d/sssd restart")

	return gidNumber


def delete_ldap_group(groupName):
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + groupName + ",ou=groups," + hubLDAPBaseDN
	print "delete: " + dn	
	try:
		l.delete_s(dn)
	except ldap.NO_SUCH_OBJECT:
		print "group not found"

	
def _insert_jos_xgroups(db, name, description='', grouptype=0, gidNumber=0):

	if not gidNumber:
		sql =  "INSERT INTO jos_xgroups ( `cn`, `description`, `published`, `type`, `public_desc`, `private_desc`, `restrict_msg`, "
		sql += "`join_policy`, `discussion_email_autosubscribe`, `logo`,  `plugins`, `created`, `created_by`, `params`) " 
		sql += "VALUES (" + '%s,'*13 + "%s );" 
		data = (name, description, 0, int(grouptype), '', '', '', 1, 0, '', None, datetime.datetime.now(), None, None)
	else:
		sql =  "INSERT INTO jos_xgroups ( `gidNumber`, `cn`, `description`, `published`, `type`, `public_desc`, `private_desc`, `restrict_msg`, "
		sql += "`join_policy`, `discussion_email_autosubscribe`, `logo`, `plugins`, `created`, `created_by`, `params`) " 
		sql += "VALUES (" + '%s,'*14 + "%s );" 
		data = (int(gidNumber), name, description, 0, int(grouptype), '', '', '', 1, 0, '', None, datetime.datetime.now(), None, None)

	gidNumber = db.query_lastrowid(sql, data)

	return gidNumber


def _insert_jos_xgroups_members(db, gidNumber, uidNumber):
	sql =  "INSERT INTO jos_xgroups_members(gidNumber, uidNumber) "
	sql += "VALUES ( %s, %s );" 

	data = (gidNumber, uidNumber)
	gidNumber = db.query_rowcount(sql, data)


def insert_jos_xgroups_members(userName, groupName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	# look up uidNumber from username
	uidNumber = hubzero.utilities.user.userLookup(userName)

	# look up gidNumber from groupname
	gidNumber = groupNumberLookup(groupName)

	# enroll user in db
	_insert_jos_xgroups_members(db, gidNumber, uidNumber)


def delhubgroup(groupName):
	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()

	db =  hubzero.data.db.MySQLConnection("localhost", dbUserName, dbName, dbPW)
	delete_ldap_group(groupName)
	_del_jos_xgroups(db, groupName)
	groupCleanup(db, groupName)


def addhubgroup(groupName, description='', gidNumber=0, type=0):

	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	if not groupExists(groupName):
		gidNumber = _insert_jos_xgroups(db, groupName, description, gidNumber, type)
		add_ldap_group(groupName, description, gidNumber)
	else:
		return -1

	return gidNumber	


def addhubgroupdbonly(groupName, description='', gid=0, type=0):

	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	gidNumber = _insert_jos_xgroups(db, groupName, description, type, gid)
	return gidNumber	


def _ldapGroupExists(groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	resultid = l.search(hubLDAPBaseDN, ldap.SCOPE_SUBTREE, '(&(objectclass=posixGroup)(cn=' + groupName + '))', None)
	result_type, result_data = l.result(resultid, 0)
	if result_data == []:
		return False
	else:
		return True


def _getnextLDAPGIDNumber(start = 1000, end = 99999999):
	# query ldap directly for next gidNumber
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	siteSuffix = hubzero.config.ldapconfig.getLDAPSiteSuffix()
	
	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	nextGIDNumber = start

	# get all groups from ldap
	try:
		r = l.search_s(
			"ou=groups," + siteSuffix, 
			ldap.SCOPE_SUBTREE, 
			'(objectClass=posixGroup)',   
			['cn', 'gidNumber'])

	except ldap.NO_SUCH_OBJECT:
		return low

	# loop through all the gidNumbers
	for dn,entry in r:
		gidNumber = int(entry['gidNumber'][0])

		if gidNumber > nextGIDNumber:
			nextGIDNumber = gidNumber

		if gidNumber > end:
			raise Exception('No open gidNumbers available')

	return nextGIDNumber+1


def _getnextLdapGID():

	# pipe a `getent group` call into a awk used to analyze for next groupid
	# less desirable than _getnextLDAPGIDNumber due to caching issues causing
	# the getent group call to not reflect new groups just added

	p1 = subprocess.Popen(['getent', 'group'], stdout=subprocess.PIPE)
	s = '{uid[$3]=1}END{for(x=1000; x<=65000; x++) {if(uid[x] != ""){}else{print x; exit;}}}'
	p2 = subprocess.Popen(['awk', '-F:', s], stdin=p1.stdout, stdout=subprocess.PIPE)
	n = p2.communicate()
	return n[0].strip()


def addusertogroup_ldap(userName, groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	
	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")	
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	# add the data to ldap
	mods = [(ldap.MOD_ADD, 'memberUid', userName)]

	try:
		l.modify_s('cn=' + groupName + ',ou=groups,' + hubLDAPBaseDN, mods)
	except ldap.TYPE_OR_VALUE_EXISTS:
		pass


def deleteuserfromgroup_ldap(userName, groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	
	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")	
	l = ldap.initialize(ldapServer)
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	# add the data to ldap
	mods = [(ldap.MOD_DELETE, 'memberUid', userName)]

	l.modify_s('cn=' + groupName + ',ou=groups,' + hubLDAPBaseDN, mods)


def addUserToGroup(userName, groupName):
	addusertogroup_ldap(userName, groupName)
	insert_jos_xgroups_members(userName, groupName)


def deleteuserfromgroup(userName, groupName):
	uidNumber = hubzero.utilities.user.userLookup(userName)
	gidNumber = groupNumberLookup(groupName)

	del_jos_xgroups_members(gidNumber, uidNumber)
	deleteuserfromgroup_ldap(userName, groupName)


def getUserGroups(userName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	# get all user group memberships
	sql = "SELECT `p`.`username`, `g`.`cn` FROM jos_xgroups_members m JOIN jos_xprofiles p on (p.uidNumber = m.uidNumber) JOIN jos_xgroups g on (g.gidNumber = m.gidNumber) WHERE `p`.`username` = %s "

	result = db.query(sql, (userName))

	userGroupMembershipList = []

	if len(result) > 0:
		for userGroupMembership in result:
			userGroupMembershipList.append(userGroupMembership.cn)

	return userGroupMembershipList


def userInGroup(userName, groupName):
	groups = getUserGroups(userName)

	if groupName in groups:
		return True

	return False


def syncgroup(group, hubname = "", purgeorphans = True, addgroupmemberships = True, verbose = False, debug = False):
	_syncgroups(hubname,purgeorphans,addgroupmemberships,group,verbose,debug)


def syncgroups(hubname = "", purgeorphans = True, addgroupmemberships = True, verbose = False, debug = False):
    _syncgroups(hubname,purgeorphans,addgroupmemberships,'*',verbose,debug)


def _syncgroups(hubname = "", purgeorphans = True, addgroupmemberships = True, group = '*', verbose = False, debug = False):
	
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	siteSuffix = hubzero.config.ldapconfig.getLDAPSiteSuffix()
	ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
	
	if purgeorphans:

		# store of all gids from the DB and store them off for easy lookup
		if group == '*':
		    sql = "SELECT `cn` from jos_xgroups"
		else:
		    sql = "SELECT `cn` from jos_xgroups WHERE cn='" + group + "'"

		groupsFromDB = {}
		for dgroup in db.query(sql, None):
			groupsFromDB[dgroup.cn] = 1

		l = ldap.initialize(ldapServer)
		l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

		# get all groups from ldap
		r = None
		try:
			if group == '*':
				r = l.search_s("ou=groups," + siteSuffix, ldap.SCOPE_SUBTREE, '(objectClass=posixGroup)',   ['cn'])
			else:
				r = l.search_s("cn="+group+",ou=groups," + siteSuffix, ldap.SCOPE_SUBTREE, '(objectClass=posixGroup)',   ['cn']) 

		except ldap.NO_SUCH_OBJECT:
			if debug:
				print "No groups found"
	
		# delete any group in ldap that isn't in the database
		for dn,entry in r:
			cn = str(entry['cn'][0])
			if cn not in groupsFromDB:
				if verbose:
					print "deleting group '" + cn + "' from ldap"
				hubzero.utilities.group.delete_ldap_group(cn)	
	

	# grab complete informaton on all groups from db 
	if group == '*':
	    sql = "SELECT `cn`, `description`, `gidNumber` from jos_xgroups"
	else:
	    sql = "SELECT `cn`, `description`, `gidNumber` from jos_xgroups WHERE cn='" + group + "'"

	for dgroup in db.query(sql, None):
	
		if not dgroup.description: dgroup.description = ""
 
		# get the groups membership if asked
		groupMembers = []
		if addgroupmemberships:
			# get all group memberships for this group 
			sql = """SELECT `p`.`username`, `g`.`cn`, `p`.`uidNumber`
		    FROM jos_xgroups_members m 
		    JOIN jos_xprofiles p on (p.uidNumber = m.uidNumber) 
		    JOIN jos_xgroups g on (g.gidNumber = m.gidNumber)
		    WHERE g.cn = %s"""
			data = (dgroup.cn)
			
			for userGroupMembership in db.query(sql, data):
				groupMembers.append(str(userGroupMembership.username))

		try:
			print "syncing group '" + dgroup.cn + "' to ldap"
			add_ldap_group(dgroup.cn, dgroup.description, dgroup.gidNumber, groupMembers)
		except ldap.ALREADY_EXISTS:
			delete_ldap_group(dgroup.cn)
			add_ldap_group(dgroup.cn, dgroup.description, dgroup.gidNumber, groupMembers)

		if groupMembers and debug:
			print "  adding members:" + str(groupMembers)


def _delete_ldap_users(args): 
	
	# by only deleting users in db, any users that exist only in ldap will be preserved
	if not args.deleteorphans:
		print "Orphaned users in ldap without a corresponding group in the database will be PRESERVED"
		dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
		dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
		dbName = hubzero.config.webconfig.getWebConfigDBName()
		db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	
		# grab all users from db 
		sql = "SELECT `username`, `name`, `userPassword`, `uidNumber`, `homeDirectory`, `loginShell`, `gid`, gidNumber from jos_xprofiles"
		
		for user in db.query(sql, None):
			print "deleting " + user.username
			try:
				hubzero.utilities.user.delete_ldap_user(user.username)
			except ldap.NO_SUCH_OBJECT:
				print " user not found"

	else:
		# grab all users from ldap
		hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
		hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
		hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
		siteSuffix = hubzero.config.ldapconfig.getLDAPSiteSuffix()
		
		ldapServer = hubzero.config.webconfig.getComponentParam("com_system", "ldap_primary")
		l = ldap.initialize(ldapServer)
		l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

		try:
			r = l.search_s("ou=users," + siteSuffix, ldap.SCOPE_SUBTREE, '(objectClass=posixAccount)', ['uid'])

			for dn,entry in r:
				uid = str(entry['uid'][0])
				hubzero.utilities.user.delete_ldap_user(uid)

		except ldap.NO_SUCH_OBJECT:
			print "No users in ou=users"
	
