# @package      hubzero-python
# @file         webconfig.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#


import ConfigParser
import hubzero.config.hubzerositeconfig
import hubzero.data.db
import re
import os
import json


def getDefaultSite():
	return hubzero.config.hubzerositeconfig.getHubzeroConfigOption("DEFAULT", "site")
	

def getWebConfigOption(optionName, hubName = ""):
	
	if not hubName: hubName = getDefaultSite()

	wwwDocRoot = hubzero.config.hubzerositeconfig.getHubzeroConfigOption(hubName, "DocumentRoot")	
	
	# read config values from file
	file = open(wwwDocRoot + "/configuration.php")
	configValues = file.read()
	file.close()
	
	# hubconfig might still be around, if so, dump this file into our search string... we're gonna assume no variable
	if os.path.exists(wwwDocRoot + "/hubconfiguration.php"):
		file = open(wwwDocRoot + "/hubconfiguration.php")
		hubconfigFileContents = file.read()
		file.close()
		configValues += hubconfigFileContents
	
	# grep for right line
	regEx = "(?:var|public)\s*\$" + optionName + "\s*=\s*'(.*)';"
	m = re.search(regEx, configValues, re.MULTILINE | re.IGNORECASE)

	# return whatever we find, if not found, this will be none type
	if m:
		return m.group(1)
	else:
		return ""
		    

def getWebConfigDBPassword(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName:
		return getWebConfigOption("password")
	else:
		return getWebConfigOption("password", hubName)


def getWebConfigDBUsername(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName: 
		return getWebConfigOption("user")
	else:
		return getWebConfigOption("user",  hubName)


def getWebConfigDBName(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName: 
		return getWebConfigOption("db")
	else:
		return getWebConfigOption("db", hubName)


def _writeParamField(componentOption, NewParamValueDictionary, hubName):
	"""
	Take a dictionary, transform it to properly formatted params string for Joomla, and 
	update the params field of the jos_components table for the specified component
	"""
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	if hubzero.utilities.misc.JOOMLA_25:
		parmString = json.dumps(NewParamValueDictionary)
		sql = "UPDATE jos_extensions set params = %s WHERE type='component' AND `element` = %s"
	else:
		parmString = ""
		for parm in NewParamValueDictionary:
			parmString += parm + "=" + NewParamValueDictionary[parm] + "\n"
		sql = "UPDATE jos_components set params = %s WHERE `option` = %s"


	data = (parmString, componentOption)
	db.query_rowcount(sql, data)


def _writePluginParamField(folder, element, NewParamValueDictionary, hubName):
	"""
	Take a dictionary, transform it to properly formatted params string for Joomla, and 
	update the params field of the jos_extensions table for the specified plugin
	"""
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	if hubzero.utilities.misc.JOOMLA_25:
		parmString = json.dumps(NewParamValueDictionary)
		sql = "UPDATE `jos_extensions` SET `params`=%s WHERE `type`='plugin' AND `folder`=%s AND `element`=%s"
	else:
		parmString = ""
		for parm in NewParamValueDictionary:
			parmString += parm + "=" + NewParamValueDictionary[parm] + "\n"
		sql = "UPDATE `jos_plugins` SET `params`=%s WHERE `folder`=%s AND `element`=%s"

	data = (parmString, folder, element)
	db.query_rowcount(sql, data)


def _buildDictionary(componentOption, hubName):
	"""
	Put the parameter string of the database into a dictionary
	String follows the format:
	
	param1=value1
	param2=value2
	param3=value3
	
	"""
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	
	# look up parm string and parse it on newline, for each line, parse it further based on *first* '='
	if hubzero.utilities.misc.JOOMLA_25:
		sql = "SELECT `params` from jos_extensions WHERE type='component' AND `element`=%s"
	else:
		sql = "SELECT `params` from jos_components WHERE `option`=%s"

	data = (componentOption)
	paramsString = db.query_selectscalar(sql, data)
	
	if hubzero.utilities.misc.JOOMLA_25 and paramsString.startswith('{'):
		try:
			parmsDict = json.loads(paramsString)
			return parmsDict
		except:
			pass

	parmsDict = {}
	parms = paramsString.split("\n")
	for parmline in parms:
		if parmline: # in case of any blank lines in the params field
			n,v = parmline.split("=", 1)
			parmsDict[n] = v

	return parmsDict


def _buildPluginDictionary(folder, element, hubName):
	"""
	Put the parameter string of the database into a dictionary
	String follows the format:
	
	param1=value1
	param2=value2
	param3=value3
	
	"""
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	
	# look up parm string and parse it on newline, for each line, parse it further based on *first* '='
	if hubzero.utilities.misc.JOOMLA_25:
		sql = "SELECT `params` from jos_extensions WHERE type='plugin' AND folder=%s AND element=%s"
	else:
		sql = "SELECT `params` from jos_plugins WHERE `folder`=%s AND `element`=%s"

	data = (folder,element)
	paramsString = db.query_selectscalar(sql, data)

	if hubzero.utilities.misc.JOOMLA_25 and paramsString.startswith('{'):
		try:
			parmsDict = json.loads(paramsString)
			return parmsDict
		except:
			pass

	parmsDict = {}
	parms = paramsString.split("\n")
	for parmline in parms:
		if parmline: # in case of any blank lines in the params field
			n,v = parmline.split("=", 1)
			parmsDict[n] = v

	return parmsDict
	

def getComponentParam(componentOption, parameterName, hubName = ""):
	"""
	Get individual param value for a component and return to caller
	"""
	paramDict = _buildDictionary(componentOption, hubName)
	
	if paramDict.has_key(parameterName):
		return paramDict[parameterName]
	else:
		return None	


def getPluginParam(folder,element, parameterName, hubName = ""):
	"""
	Get individual param value for a component and return to caller
	"""
	paramDict = _buildPluginDictionary(folder,element, hubName)
	
	if paramDict.has_key(parameterName):
		return paramDict[parameterName]
	else:
		return None	


def addComponentParam(componentOption, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
	"""
	Add value to the param string in the jos_components table for the specified component.
	"""
	paramDict = _buildDictionary(componentOption, hubName)

	if allowOverwrite:
		paramDict[parameterName] = parameterValue
	else:
		if paramDict.has_key(parameterName):
			raise Exception("ERROR: addCompoentParam called for existing parameter with allowOverwrite = false")
		else:
			paramDict[parameterName] = parameterValue

	_writeParamField(componentOption, paramDict, hubName)	


def addPluginParam(folder, element, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
	"""
	Add value to the param string in the jos_components table for the specified component.
	"""
	paramDict = _buildPluginDictionary(folder,element,hubName)

	if allowOverwrite:
		paramDict[parameterName] = parameterValue
	else:
		if paramDict.has_key(parameterName):
			raise Exception("ERROR: addPluginParam called for existing parameter with allowOverwrite = false")
		else:
			paramDict[parameterName] = parameterValue

	_writePluginParamField(folder,element, paramDict, hubName)	

	
def delComponentParam(componentOption, parameterName, hubName = ""):
	"""
	Delete value from the param string in the in the jos_extensions table for the specified component
	"""
	paramDict = _buildDictionary(componentOption, hubName)
	del paramDict[parameterName]
	_writeParamField(componentOption, paramDict, hubName)


def delPluginParam(folder, element, parameterName, hubName = ""):
	"""
	Delete value from the param string in the in the jos_extensions table for the specified plugin
	"""
	paramDict = _buildPluginDictionary(folder, element, hubName)
	del paramDict[parameterName]
	_writePluginParamField(folder, element, paramDict, hubName)
	
	
def setPluginEnabled(pluginFolder, pluginElement, enable, hubName = ""):
	"""
	Enable or disable a plugin by setting the published field for the plugin in the jos_plugins table
	"""
	
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
		
	if hubzero.utilities.misc.JOOMLA_25:
		sql = "UPDATE jos_extensions set enabled = %s WHERE type='plugin' AND `folder` = %s AND `element` = %s"
	else:
		sql = "UPDATE jos_plugins set published = %s WHERE `folder` = %s AND `element` = %s"
	data = (enable, pluginFolder, pluginElement)
	db.query_rowcount(sql, data)	


def getPluginEnabled(pluginFolder, pluginElement, hubName = ""):
	"""
	Get published field for the given plugin in the jos_plugins table
	"""
	
	# database config
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
		
	if hubzero.utilities.misc.JOOMLA_25:
		sql = "SELECT enabled FROM jos_extensions  WHERE type='plugins' AND `folder` = %s AND `element` = %s"
	else:
		sql = "SELECT published FROM jos_plugins WHERE `folder` = %s AND `element` = %s"
	data = (pluginFolder, pluginElement)
	published = db.query_selectscalar(sql, data)
	return published


def getHubSiteConfig(hubname, sectionName, valueName):
	"""
	get a value from the /etc/hubzero-cms/HUBNAME.conf file
	"""
	filename = "/etc/hubzero-cms/" + hubname + ".conf"
	
	if not os.path.exists(filename):
		print "Cannot find " + filename
	
	hubzeroCMSConfig = ConfigParser.ConfigParser()
	hubzeroCMSConfig.readfp(open(filename))
	
	try:
		rv = hubzeroCMSConfig.get(sectionName, valueName)
	except ConfigParser.NoOptionError:
		rv = ''
        except ConfigParser.NoSectionError:
                rv = ''
	
	return rv
	
