# @package      hubzero-python
# @file         user.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import datetime
import hubzero.config.passwords
import hubzero.utilities.misc
import hubzero.config.webconfig
import hubzero.data.db
import ldap
import ldap.modlist


def _del_jos_users(db, username):
	sql = 'select id from `jos_users` where username = %s'
	data = (username)
	userID = db.execSelectQueryScalar(sql, data)	

	sql = 'delete from `jos_users` where userid = %s'
	data = (userid)
	userID = db.execSelectQueryScalar(sql, data)	

	return userID


def _del_jos_core_acl_aro(db, userID):

	sql = 'select id from `jos_core_acl_aro` where value = %s'
	data = (userID)
	aroID = db.execSelectQueryScalar(sql, data)	

	sql = "DELETE FROM `jos_core_acl_aro` where `value` = %s  "
	data = (userID)
	db.execQuery(sql, data)
	
	return aroID

def _del_jos_core_acl_groups_aro_map(db, aroID):
	sql = "DELETE FROM `jos_core_acl_groups_aro_map` WHERE `aro_id` = %s"
	data = (aroID)
	db.execQuery(sql, data)


def _del_jos_xprofiles(db, uidNumber):
	sql = "DELETE FROM `jos_xprofiles` WHERE `uidNumber` = %s"
	data = (uidNumber)
	db.execQuery(sql, data)


def _insert_jos_users(db, joomlauserid, name, username, email, pw, jgid, jgidNumber):

	pwhash = hubzero.config.passwords.MD5PasswordHash(pw)

	if joomlauserid == -1:
		sql = "INSERT INTO `jos_users` (`name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
		sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
		data = (name, username, email, pwhash , jgid, 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '');
		userID = db.execInsertQuery(sql, data)
	else:
		sql = "INSERT INTO `jos_users` (`id`, `name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
		sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
		data = (joomlauserid, name, username, email, pwhash, jgid, 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '');
		userID = db.execInsertQuery(sql, data)

	return userID


def _insert_jos_core_acl_aro(db, userID, fullname):
	sql = "INSERT INTO `jos_core_acl_aro` (`section_value`, `value`, `order_value`, `name`, `hidden`) "
	sql += "VALUES (%s, %s, %s, %s, %s)"
	data = ( 'users',  userID, 0, fullname, 0)
	aroID = db.execInsertQuery(sql, data)
	return aroID
	

def _insert_jos_core_acl_groups_aro_map(db, jgidNumber, aroID):
	sql = "INSERT INTO `jos_core_acl_groups_aro_map` (`group_id`, `section_value`, `aro_id`) "
	sql += "VALUES (%s, %s, %s)"
	data = ( jgidNumber, '', aroID)
	db.execQuery(sql, data)


def _insert_jos_xprofiles(db, hubname, uid, fname, lname, username, email, pw, joomlagid, joomlagidNumber, gid, gidNumber, loginShell, homeDir):
	
	sql = "INSERT INTO `jos_xprofiles` (`uidNumber`, `name`, `username`, `email`, `registerDate`, `gidNumber`, `homeDirectory`, `loginShell`, "
	sql += "`ftpShell`, `userPassword`, `gid`, `orgtype`, `organization`, `countryresident`, `countryorigin`, `gender`, `url`, `reason`, "
	sql += "`mailPreferenceOption`, `usageAgreement`, `jobsAllowed`, `modifiedDate`, `emailConfirmed`, `regIP`, `regHost`, `nativeTribe`, `phone`, "
	sql += "`proxyPassword`, `proxyUidNumber`, `givenName`, `middleName`, `surname`, `picture`, `vip`, `public`, `params`, `note`, `shadowExpire`) "
	sql += "VALUES (" + '%s,'*37 + "%s );" 
	
	data = (uid, fname + ' ' + lname, username, email, datetime.datetime.now(), gidNumber, homeDir, loginShell, 
	        '/usr/lib/sftp-server', hubzero.config.passwords.MD5PasswordHash(pw), gid, '', '', '', '', '', '', '', 
	        0, 0, 3, '0000-00-00 00:00:00', 1, '', '', '', '',
	        '', '', fname, '', lname, '', 0, 0, '', '', None)
	db.execQuery(sql, data)


def _insert_jos_users_password(db, userid, pw):
	sql = "INSERT INTO jos_users_password (`user_id`, `passhash`) " 
	sql += "VALUES (%s, %s)"
	
	data = (userid, hubzero.config.passwords.MD5PasswordHash(pw))
	db.execQuery(sql, data)


def _insert_jos_xgroups(db, name, description):
	sql =  "INSERT INTO jos_xgroups ( `cn`, `description`, `published`, `type`, `access`, `public_desc`, `private_desc`, `restrict_msg`, "
	sql += "`join_policy`, `privacy`, `discussion_email_autosubscribe`, `logo`, `overview_type`, `overview_content`, `plugins`, `created`,"
	sql += "`created_by`, `params`) " 
	sql += "VALUES (" + '%s,'*17 + "%s );" 

	data = (name, description, 0, 1, 0, '', '', '',   1, 1, 0, '', None, None, None, datetime.datetime.now(), None, None)
	gidNumber = db.execInsertQuery(sql, data)
	return gidNumber


def _insert_jos_xgroups_memers(db, gidNumber, uidNumber):
	sql =  "INSERT INTO jos_xgroups ( gidNumber, uidNumber "
	sql += "VALUES ( %s, %s );" 

	data = (gidNumber, uidNumber)
	gidNumber = db.execQuery(sql, data)


def _del_jos_xgroups(db, name):
	sql = "DELETE FROM jos_xgroups WHERE `CN`=%s "
	data = (name)
	db.execQuery(sql, data)


def _add_ldap_group(groupName, groupDescription, gidNumber):
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrPW")	
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrDN")	
	hubLDAPBaseDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPBaseDN")

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)
	
	dn = "cn=" + groupName + ",ou=groups," + hubLDAPBaseDN
	attrs = {}
	attrs['objectclass'] = ['posixGroup', 'top']
	attrs['cn'] = groupName
	attrs['gidNumber'] =str( gidNumber)
	attrs['description'] = groupDescription
	#attrs['ou'] = 'users'
	ldif = ldap.modlist.addModlist(attrs)
	print ldif
	l.add_s(dn, ldif)


def _del_ldap_group(groupName):
	# get ldapManagerUserDN and PW
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrPW")	
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrDN")	
	hubLDAPBaseDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPBaseDN")

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + groupName + ",ou=groups," + hubLDAPBaseDN
	print dn
	l.delete_s(dn)


def _add_ldap_user(username, password):

	# get ldapManagerUserDN and PW
	
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW= hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")	

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + username + "," + hubLDAPBaseDN
	attrs = {}
	attrs['objectclass'] = ['organizationalRole', 'simpleSecurityObject']
	attrs['cn'] = username
	attrs['userPassword'] = "none"
	attrs['uid'] = username
	
	print "add ldap user dn: " + dn
	print "ldif: "
	print attrs
	
	ldif = ldap.modlist.addModlist(attrs)
	l.add_s(dn, ldif)


def _delete_ldap_user(username):
	# get ldapManagerUserDN and PW
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrPW")	
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPAcctMgrDN")	
	hubLDAPBaseDN = hubzero.config.webconfig.getWebConfigOption("hubLDAPBaseDN")

	l = ldap.open("localhost")
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + username + "," + hubLDAPBaseDN
	l.delete_s(dn)


def delhubgroup(groupName):
	# grab config options
	dbPW = hubzero.config.passwords.getMySqlPW()
	db =  hubzero.data.db.mySql("localhost", "myhub", "myhub", dbPW)
	_del_ldap_group(groupName)
	_del_jos_xgroups(db, groupName)


def addhubgroup(db, groupName, groupDescription):

	gidNumber = _insert_jos_xgroups(db, groupName, groupDescription)
	return gidNumber

	
def addhubuser(hubname, username, fname, lname, email, pw, joomlagid, joomlagidNumber, gid, gidNumber, joomlauserid, loginShell, homeDir, createGroup, skipldap):
	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	
	db =  hubzero.data.db.mySql("localhost", dbName, dbUserName, dbPW)
	
	fullname = fname + ' ' + lname

	# usually homeDir will be blank, but not always
	if homeDir == "":
		homeDir = '/home/' + hubname + "/" + username


	userID = _insert_jos_users(db, joomlauserid, fullname, username, email, pw, joomlagid, joomlagidNumber)
	aroID = _insert_jos_core_acl_aro(db, userID, fullname)
	_insert_jos_core_acl_groups_aro_map(db, joomlagidNumber, aroID)
	_insert_jos_xprofiles(db, hubname, userID, fname, lname, username, email, pw, joomlagid, joomlagidNumber, gid, gidNumber, loginShell, homeDir)
	_insert_jos_users_password(db, userID, pw)


	# create linux group for user named same as username
	if createGroup:
		rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(["addgroup", username])
		print procStdOut
		if rc:
			print procStdErr
		gidNumber = _insert_jos_xgroups(db, name, description)	
		_insert_jos_xgroups_memers(gidNumber, userID)
		

	# create home directory
	rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(["mkdir", homeDir, "--mode", "0750"])
	print procStdOut
	if rc:
		print procStdErr

	
	# set ownerships
	rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(["chown", username + "." + username, homeDir])
	print procStdOut
	if rc:
		print procStdErr

	
	if not skipldap:
		_add_ldap_user(username, pw)

	
def delhubuser(username):
	# grab config options
	dbPW = hubzero.config.passwords.getMySqlPW()
	db =  hubzero.data.db.mySql("localhost", "myhub", "myhub", dbPW)
	
	sql = 'select id from `jos_users` where username = %s'
	data = (username)
	userID = db.execSelectQueryScalar(sql, data)
	aroID = _del_jos_core_acl_aro(db, userID)
	_del_jos_core_acl_groups_aro_map(db, aroID)
	_del_jos_xprofiles(db, userID)	
	_delete_ldap_user(username)
