# @package      hubzero-python
# @file         group.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import ldap.modlist as modlist

import datetime
import hubzero.config.passwords
import hubzero.utilities.misc
import hubzero.config.webconfig
import hubzero.data.db
import ldap

def groupCleanup(db, groupname):
	"""
	We don't use foriegn keys, so messy cleanup here when you delete a group
	"""

	sql = 'SELECT gidNumber FROM `jos_xgroups` WHERE `cn` = %s'
	data = (groupname)
	gidNumber = db.query_selectscalar(sql, data)

	# don't bother if the group doesn't exist
	if not gidNumber:
		return

	sql = "DELETE FROM `jos_xgroups_applicants` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_events` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_inviteemails` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_invitees` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_managers` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_memberoption` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_members` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_reasons` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_roles` WHERE `gidNumber`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)

	sql = "DELETE FROM `jos_xgroups_tracperm` WHERE `group_id`=%s "
	data = (gidNumber)
	db.query_rowcount(sql, data)


def groupExists(groupname):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()

	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	if _groupExists(db, groupname):
		return True
	else:
		return False

def groupNumberLookup(groupName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()

	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)
	return _groupExists(db, groupName)


def _groupExists(db, groupname):
	sql = 'SELECT gidNumber FROM `jos_xgroups` WHERE `cn` = %s'
	data = (groupname)
	gidNumber = db.query_selectscalar(sql, data)

	# will be None if none exists
	return gidNumber


def _del_jos_xgroups(db, name):
	sql = "DELETE FROM jos_xgroups WHERE `CN`=%s "
	data = (name)
	db.query_rowcount(sql, data)


def _del_jos_xgroups_members(db, gidNumber, uidNumber):
	sql = "DELETE FROM jos_xgroups_members WHERE `gidNumber`=%s AND `uidNumber`=%s"
	data = (gidNumber, uidNumber)
	db.query_rowcount(sql, data)


def del_jos_xgroups_members(gidNumber, uidNumber):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	_del_jos_xgroups_members(db, gidNumber, uidNumber)


def add_ldap_group(cn, groupDescription, gidNumber, members = {}):
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	l = ldap.initialize('ldap://localhost')
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + cn + ",ou=groups," + hubLDAPBaseDN
	attrs = {}
	attrs['objectclass'] = ['posixGroup', 'top']
	attrs['cn'] = cn
	attrs['gidNumber'] = str(gidNumber)
	attrs['description'] = groupDescription
	attrs['memberuid'] = members
	
	ldif = ldap.modlist.addModlist(attrs)
	l.add_s(dn, ldif)

def delete_ldap_group(groupName):
	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	l = ldap.initialize('ldap://localhost')
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	dn = "cn=" + groupName + ",ou=groups," + hubLDAPBaseDN
	
	try:
		l.delete_s(dn)
	except ldap.NO_SUCH_OBJECT:
		pass 

	
def _insert_jos_xgroups(db, name, description='', grouptype=0):
	sql =  "INSERT INTO jos_xgroups ( `cn`, `description`, `published`, `type`, `public_desc`, `private_desc`, `restrict_msg`, "
	sql += "`join_policy`, `discussion_email_autosubscribe`, `logo`, `overview_type`, `overview_content`, `plugins`, `created`,"
	sql += "`created_by`, `params`) " 
	sql += "VALUES (" + '%s,'*15 + "%s );" 
	data = (name, description, 0, int(grouptype), '', '', '', 1, 0, '', None, None, None, datetime.datetime.now(), None, None)

	gidNumber = db.query_lastrowid(sql, data)

	return gidNumber


def _insert_jos_xgroups_members(db, gidNumber, uidNumber):
	sql =  "INSERT INTO jos_xgroups_members(gidNumber, uidNumber) "
	sql += "VALUES ( %s, %s );" 

	data = (gidNumber, uidNumber)
	gidNumber = db.query_rowcount(sql, data)


def insert_jos_xgroups_members(userName, groupName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	# look up uidNumber from username
	uidNumber = hubzero.utilities.user.userLookup(userName)

	# look up gidNumber from groupname
	gidNumber = groupNumberLookup(groupName)

	# enroll user in db
	_insert_jos_xgroups_members(db, gidNumber, uidNumber)


def delhubgroup(hubname, groupName):
	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
	dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)

	db =  hubzero.data.db.MySQLConnection("localhost", dbUserName, dbName, dbPW)
	delete_ldap_group(groupName)
	_del_jos_xgroups(db, groupName)
	groupCleanup(db, groupName)


def addhubgroup(groupName, description='', type=0):

	# grab config options
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

        gidNumber = _insert_jos_xgroups(db, groupName, description, type)
        add_ldap_group(groupName, description, gidNumber)

	return gidNumber	


def _ldapGroupExists(groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")

	l = ldap.initialize('ldap://localhost')
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	resultid = l.search(hubLDAPBaseDN, ldap.SCOPE_SUBTREE, '(&(objectclass=posixGroup)(cn=' + groupName + '))', None)
	result_type, result_data = l.result(resultid, 0)
	if result_data == []:
		return False
	else:
		return True


def addusertogroup_ldap(userName, groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	l = ldap.initialize('ldap://localhost')
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	# add the data to ldap
	mods = [(ldap.MOD_ADD, 'memberUid', userName)]

	try:
		l.modify_s('cn=' + groupName + ',ou=groups,' + hubLDAPBaseDN, mods)
	except ldap.TYPE_OR_VALUE_EXISTS:
		pass


def deleteuserfromgroup_ldap(userName, groupName):

	hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerdn")
	hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam("com_system", "ldap_managerpw")
	hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam("com_system", "ldap_basedn")
	l = ldap.initialize('ldap://localhost')
	l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

	# add the data to ldap
	mods = [(ldap.MOD_DELETE, 'memberUid', userName)]

	l.modify_s('cn=' + groupName + ',ou=groups,' + hubLDAPBaseDN, mods)



def addUserToGroup(userName, groupName):
	addusertogroup_ldap(userName, groupName)
	insert_jos_xgroups_members(userName, groupName)


def deleteuserfromgroup(userName, groupName):
	uidNumber = hubzero.utilities.user.userLookup(userName)
	gidNumber = groupNumberLookup(groupName)

	del_jos_xgroups_members(gidNumber, uidNumber)
	deleteuserfromgroup_ldap(userName, groupName)



def getUserGroups(userName):
	dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
	dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
	dbName = hubzero.config.webconfig.getWebConfigDBName()
	db =  hubzero.data.db.MySQLConnection("localhost", dbName, dbUserName, dbPW)

	# get all user group memberships
	sql = "SELECT `p`.`username`, `g`.`cn` FROM jos_xgroups_members m JOIN jos_xprofiles p on (p.uidNumber = m.uidNumber) JOIN jos_xgroups g on (g.gidNumber = m.gidNumber) WHERE `p`.`username` = %s "

	result = db.query(sql, (userName))

	userGroupMembershipList = []

	if len(result) > 0:
		for userGroupMembership in result:
			userGroupMembershipList.append(userGroupMembership.cn)

	return userGroupMembershipList

def userInGroup(userName, groupName):
	groups = getUserGroups(userName)

	if groupName in groups:
		return True

	return False

