/**
 * @file       fs.c
 * @copyright  Copyright (c) 2016-2020 The Regents of the University of California.
 * @license    http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2016-2020 The Regents of the University of California.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of The Regents of the University of California.
 */

#ifdef linux
/* For pread()/pwrite() */
#define _XOPEN_SOURCE 500
#endif

#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>

#ifdef HAVE_SETXATTR
#include <sys/xattr.h>
#endif

#include <getopt.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <syslog.h>
#include <stdarg.h>
#include <limits.h>
#include <stdlib.h>

#include "fs.h"
#include "path.h"
#include "globals.h" // TODO: for source_dir
#include "cache.h"
#include "lock.h"
#include "git.h"



int xmp_getattr(const char *path, struct stat *stbuf)
{
  struct fuse_context *fc = fuse_get_context();
  path_t *rpath = NULL;
  int res;

  debug("xmp_getattr(%s, stat)", path);
  // Workaround for GridSFTP stupidity
  if ((res = can_access(path, &rpath, BL_NOFREE)) != ACC_OK) {
    if (res == ACC_BLACKLISTED) {
      if (strcmp(rpath->file_name, ".git") != 0) {
        free_path(rpath);
        return -EACCES;
      }
      /* Otherwise, let it fall through. Stat will make it look accessible,
       * but all other operations will fail.
       */
    }
    else {
      // It isn't blacklisted
      return -EACCES;
    }
  }

  res = lstat(rpath->full_path, stbuf);
  debug("xmp_getattr() returning %d", res);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  // Small tweak to make ownership information look legitimate
  stbuf->st_uid = fc->uid;

  free_path(rpath);
  return 0;
}

int xmp_access(const char *path, int mask)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_access(%s, %d)", path, mask);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  res = access(rpath->full_path, mask);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  free_path(rpath);
  return 0;
}

int xmp_readlink(const char *path, char *buf, size_t size)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_readlink(%s, %s, %d)", path, buf, size);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  res = readlink(rpath->full_path, buf, size - 1);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  buf[res] = '\0';
  free_path(rpath);
  return 0;
}

int xmp_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
             off_t offset, struct fuse_file_info *fi)
{
  struct fuse_context *fc = fuse_get_context();
  DIR *dp;
  struct dirent *de;
  path_t *rpath = NULL;

  (void) offset;
  (void) fi;

  debug("xmp_readdir(%s, %s, filler, %d, fi)", path, buf, offset);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;
  
  dp = opendir(rpath->full_path);
  if (dp == NULL) {
    free_path(rpath);
    return -errno;
  }

  while ((de = readdir(dp)) != NULL) {
    // The root behaves differently
    if (strcmp(source_dir, rpath->full_path) == 0) {
      debug("xmp_readdir(): got root");
      char cpath[PATH_MAX + 256];
      cpath[0] = '/';
      strcpy(&(cpath[1]), de->d_name);
      if (can_access(cpath, NULL, BL_FREE) != ACC_OK) {
        continue;    
      }
    }

    struct stat st;
    memset(&st, 0, sizeof(st));
    st.st_ino = de->d_ino;
    st.st_mode = de->d_type << 12;
    st.st_uid = fc->uid;
    debug("xmp_readdir(): setting st.st_uid = %d", fc->uid);
    if (filler(buf, de->d_name, &st, 0))
      break;
  }

  closedir(dp);
  free_path(rpath);
  return 0;
}

int xmp_mknod(const char *path, mode_t mode, dev_t rdev)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_mknod(%s, %d, %d)", path, mode, rdev);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  if (lock_repo(rpath->repo_name) != 0) return -EACCES;

  /* On Linux this could just be 'mknod(path, mode, rdev)' but this
     is more portable */
  if (S_ISREG(mode)) {
    res = open(rpath->full_path, O_CREAT | O_EXCL | O_WRONLY, mode);
    if (res >= 0)
      res = close(res);
  } else if (S_ISFIFO(mode)) {
    res = mkfifo(rpath->full_path, mode);
  }
  else {
    res = mknod(rpath->full_path, mode, rdev);
  }
  if (res == -1) {
    unlock_repo(rpath->repo_name);
    free_path(rpath);
    return -errno;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_ADD, rpath, NULL, fc->uid);
#endif

  unlock_repo(rpath->repo_name);
  free_path(rpath);
  return 0;
}

int xmp_mkdir(const char *path, mode_t mode)
{
  // Directories aren't tracked by git.
  path_t *rpath = NULL;
  int res;

  debug("xmp_mkdir(%s, %d)", path, mode);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  // This is atomic anyway, no need for a lock
  res = mkdir(rpath->full_path, mode);
  if (res == -1) {
    free_path(rpath); 
    return -errno;
  }

  free_path(rpath);
  return 0;
}

int xmp_unlink(const char *path)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_unlink(%s)", path);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  if (lock_repo(rpath->repo_name) != 0) return -EACCES;

  res = unlink(rpath->full_path);
  if (res == -1) {
    unlock_repo(rpath->repo_name);
    free_path(rpath);
    return -errno;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_RM, rpath, NULL, fc->uid);
#endif

  unlock_repo(rpath->repo_name);

  free_path(rpath);
  update_mwb();
  return 0;
}

int xmp_rmdir(const char *path)
{
  // Directories aren't tracked by git
  path_t *rpath = NULL;
  int res;

  debug("xmp_rmdir(%s)", path);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  debug("xmp_rmdir(): rpath->file_name: %p rpath->file_path: %p",
        rpath->file_name, rpath->file_path);

  if (rpath->file_name == NULL) {
    error("xmp_rmdir(): attempting to remove repo base directory");
    return -EACCES;
  }

  res = rmdir(rpath->full_path);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  free_path(rpath);
  return 0;
}

// TODO: Make sure spath and dpath point to the same repo, or lock both repos
int xmp_symlink(const char *from, const char *to)
{
  path_t *spath = NULL;
  path_t *dpath = NULL;
  int res;

  debug("xmp_symlink(%s, %s)", from, to);

  if (can_access(from, &spath, BL_FREE) != ACC_OK) return -EACCES;
  
  if (can_access(from, &dpath, BL_FREE) != ACC_OK) {
    free_path(spath);
    return -EACCES;
  }
  
  if (lock_repo(spath->repo_name) != 0) return -EACCES;

  res = symlink(spath->full_path, dpath->full_path);
  if (res == -1) {
    unlock_repo(spath->repo_name);
    free_path(spath);
    free_path(dpath);
    return -errno;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_ADD, dpath, NULL, fc->uid);
#endif

  unlock_repo(spath->repo_name);

  free_path(spath);
  free_path(dpath);
  return 0;
}

int xmp_rename(const char *from, const char *to)
{
  path_t *spath = NULL;
  path_t *dpath = NULL;
  int res;

  debug("xmp_rename(%s, %s)", from, to);

  if (can_access(from, &spath, BL_FREE) != ACC_OK)
    return -EACCES;

  if (can_access(to, &dpath, BL_FREE) != ACC_OK) {
    free_path(spath);
    return -EACCES;
  }

  if (lock_repo(spath->repo_name) != 0) return -EACCES;

  res = rename(spath->full_path, dpath->full_path);
  if (res == -1) {
    unlock_repo(spath->repo_name);
    free_path(spath);
    free_path(dpath);
    return -errno;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  // Git itself will do the actual rename
  res = rename(dpath->full_path, spath->full_path);
  if (res == -1) {
    unlock_repo(spath->repo_name);
    free_path(spath);
    free_path(dpath);
    return -errno;
  }
 
  git(GIT_MV, spath, dpath, fc->uid);
#endif

  unlock_repo(spath->repo_name);

  free_path(spath);
  free_path(dpath);
  return 0;
}

int xmp_link(const char *from, const char *to)
{
  path_t *spath = NULL;
  path_t *dpath = NULL;
  int res;

  debug("xmp_link(%s, %s)", from, to);

  if (can_access(from, &spath, BL_FREE) != ACC_OK)
    return -EACCES;

  if (can_access(to, &dpath, BL_FREE) != ACC_OK) {
    free_path(spath);
    return -EACCES;
  }

  if (lock_repo(spath->repo_name) != 0) return -EACCES;

  if (access(dpath->full_path, F_OK) == 0) {
    unlock_repo(spath->repo_name);
    free_path(spath);
    free_path(dpath);
    return -EEXIST;
  }

#ifndef IRODS_LINK
  res = link(spath->full_path, dpath->full_path);
#else
  // This is broken on irods, do a copy and hope it's Close Enough (TM)
  res = copy_file(spath->full_path, dpath->full_path);
#endif
  if (res < 0) {
    unlock_repo(spath->repo_name);
    free_path(spath);
    free_path(dpath);
    return res;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_ADD, dpath, NULL, fc->uid);
#endif

  unlock_repo(spath->repo_name);

  free_path(spath);
  free_path(dpath);
  return 0;
}

int xmp_chmod(const char *path, mode_t mode)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_chmod(%s, %d)", path, mode);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  if (rpath->file_name == NULL) {
    error("xmp_chmod(): attempting to chmod repo base directory");
    return -EACCES;
  }

  if (lock_repo(rpath->repo_name) != 0) return -EACCES;

  res = chmod(rpath->full_path, mode);
  if (res == -1) {
    unlock_repo(rpath->repo_name);
    free_path(rpath);
    return -errno;
  }

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_ADD, rpath, NULL, fc->uid);
#endif

  unlock_repo(rpath->repo_name);

  free_path(rpath);
  return 0;
}

// I don't see this one working.
int xmp_chown(const char *path, uid_t uid, gid_t gid)
{
  struct fuse_context *fc = fuse_get_context();
  //int res;

  debug("xmp_chown(%s, %d, %d)", path, uid, gid);

  if (can_access(path, NULL, BL_FREE) != ACC_OK) return -EACCES;

/*
  res = lchown(path, uid, gid);
  if (res == -1)
    return -errno;
*/

  if (fc->uid == uid && fc->gid == gid)
    return 0;
  else
    return -EACCES;
}

int xmp_truncate(const char *path, off_t size)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_truncate(%s, %d)", path, size);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  if (lock_repo(rpath->repo_name) != 0) return -EACCES;

  res = truncate(rpath->full_path, size);
  if (res == -1) {
    unlock_repo(rpath->repo_name);
    free_path(rpath);
    return -errno;
  }

  unlock_repo(rpath->repo_name);

#ifdef USE_GIT
  struct fuse_context *fc = fuse_get_context();

  git(GIT_ADD, rpath, NULL, fc->uid);
#endif

  free_path(rpath);
  update_mwb();
  return 0;
}

int xmp_utime(const char *path, struct utimbuf *buf)
{
  int res;
  path_t *rpath = NULL;

  debug("xmp_utime(%s, buf)", path);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  res = utime(rpath->full_path, buf);
  debug("xmp_utime(): returned %d, errno %d", res, errno);
  if (res == -1) {
    free_path(rpath); 
    return -errno;
  }

  free_path(rpath);
  return 0;
}

int xmp_open(const char *path, struct fuse_file_info *fi)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_open(%s, fi)", path);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  update_mwb();

  res = open(rpath->full_path, fi->flags);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  close(res);
  free_path(rpath);
  return 0;
}

int xmp_read(const char *path, char *buf, size_t size, off_t offset,
          struct fuse_file_info *fi)
{
  path_t *rpath = NULL;
  int fd;
  int res;

  debug("xmp_read(%s, buf, %d, %d, fi)", path, size, offset);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  fd = open_cache(rpath, fi->flags);
  if (fd < 0) {
    free_path(rpath);
    return -errno;
  }

  res = pread(fd, buf, size, offset);
  if (res == -1) {
    res = -errno;
  }

  free_path(rpath);
  return res;
}

int xmp_write(const char *path, const char *buf, size_t size,
           off_t offset, struct fuse_file_info *fi)
{
  path_t *rpath = NULL;
  int fd;
  int res;

  (void) fi;

  debug("xmp_write(%s, buf, %d, %d, fi)", path, size, offset);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  fd = open_cache(rpath, fi->flags);
//  fd = open(rpath->full_path, fi->flags);
  if (fd < 0) {
    free_path(rpath);
    return fd;
  }

  if (size > max_write_bytes) {
    error("xmp_write() device is filled beyond threshold. Abort.");
    free_path(rpath);
    res = -ENOSPC;
    return res;
  }

  debug("xmp_write(): writing %d bytes to fd %d", size, fd);
  res = pwrite(fd, buf, size, offset);
  if (res == -1) {
    res = -errno;
  }

  debug("xmp_write(): res = %d", res);
  free_path(rpath);
  return res;
}

int xmp_statfs(const char *path, struct statvfs *stbuf)
{
  path_t *rpath = NULL;
  int res;

  debug("xmp_statfs(%s, stbuf)", path);

  if (can_access(path, &rpath, BL_FREE) != ACC_OK) return -EACCES;

  res = statvfs(rpath->full_path, stbuf);
  debug("xmp_statfs(): returned %d", res);
  if (res == -1) {
    free_path(rpath);
    return -errno;
  }

  free_path(rpath);
  return 0;
}

int xmp_release(const char *path, struct fuse_file_info *fi)
{
  /* Releases come as uid 0. Sigh.
   */
  path_t *rpath = NULL;

  (void) path;
  (void) fi;

  debug("xmp_release(%s, fi)", path);

  if ((rpath = normalize_path(path)) == NULL)
    return -EACCES;

  /* Find the file in the open_files list. Close them all if multiple.
   * (we have to do this, because we don't have the uid)
   * If something is closed erroneously, it'll get reopened on the next
   * read or write.
   */
  int found = 0;
  path_list_t *p = open_files;
  path_list_t *prev = NULL;
  while (p) {
    if (strcmp(p->path->full_path, rpath->full_path) == 0) {
      // Found it, save the next item
      path_list_t *next = p->next;
      flush_file(p);
      free_path_list(p);
      p = NULL;
      if (prev) prev->next = next;
      else open_files = next;
      found++;
      p = next;
    }
    else {
      prev = p;
      p = p->next;
    }
  }

  if (found == 0)
    error("xmp_release(): unable to find file '%s' in open_files",
          rpath->full_path);
  else
    debug("xmp_release(): removed %d file(s) matching path '%s'", found,
          rpath->full_path);

  dump_open_files();
  free_path(rpath);
  update_mwb();
  return 0;
}

int xmp_fsync(const char *path, int isdatasync,
           struct fuse_file_info *fi)
{
  /* Just a stub.  This method is optional and can safely be left
     unimplemented */

  (void) path;
  (void) isdatasync;
  (void) fi;

  debug("xmp_fsync(%s, %d, fi)", path, isdatasync);

  if (can_access(path, NULL, BL_FREE) != ACC_OK) return -EACCES;

  return 0;
}

#ifdef HAVE_SETXATTR
/* xattr operations are optional and can safely be left unimplemented */
int xmp_setxattr(const char *path, const char *name, const char *value,
            size_t size, int flags)
{
  int res = lsetxattr(path, name, value, size, flags);
  if (res == -1)
    return -errno;
  return 0;
}

int xmp_getxattr(const char *path, const char *name, char *value,
          size_t size)
{
  int res = lgetxattr(path, name, value, size);
  if (res == -1)
    return -errno;
  return res;
}

int xmp_listxattr(const char *path, char *list, size_t size)
{
  int res = llistxattr(path, list, size);
  if (res == -1)
    return -errno;
  return res;
}

int xmp_removexattr(const char *path, const char *name)
{
  int res = lremovexattr(path, name);
  if (res == -1)
    return -errno;
  return 0;
}
#endif /* HAVE_SETXATTR */

