/**
 * @file       projfs.c
 * @copyright  Copyright (c) 2016-2020 The Regents of the University of California.
 * @license    http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2016-2020 The Regents of the University of California.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of The Regents of the University of California.
 */

#ifdef linux
/* For pread()/pwrite() */
#define _XOPEN_SOURCE 500
#define _DEFAULT_SOURCE // vsyslog()
#endif

#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>

#ifdef HAVE_SETXATTR
#include <sys/xattr.h>
#endif

#include <getopt.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <stdarg.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/fsuid.h>
#include <syslog.h>
#include <string.h>

#include "globals.h"
#include "path.h"
#include "fs.h"

struct fuse_operations xmp_oper = {
  .getattr	= xmp_getattr,
  .access	= xmp_access,
  .readlink	= xmp_readlink,
  .readdir	= xmp_readdir,
  .mknod	= xmp_mknod,
  .mkdir	= xmp_mkdir,
  .symlink	= xmp_symlink,
  .unlink	= xmp_unlink,
  .rmdir	= xmp_rmdir,
  .rename	= xmp_rename,
  .link	= xmp_link,
  .chmod	= xmp_chmod,
  .chown	= xmp_chown,
  .truncate	= xmp_truncate,
  .utime	= xmp_utime,
  .open	= xmp_open,
  .read	= xmp_read,
  .write	= xmp_write,
  .statfs	= xmp_statfs,
  .release	= xmp_release,
  .fsync	= xmp_fsync,
#ifdef HAVE_SETXATTR
  .setxattr	= xmp_setxattr,
  .getxattr	= xmp_getxattr,
  .listxattr	= xmp_listxattr,
  .removexattr= xmp_removexattr,
#endif
};



// TODO: This should go somewhere else
void update_mwb(void) {
  struct statvfs buf;
  debug("update_mwb(void)");
  if (!statvfs(source_dir, &buf)) {
    unsigned long freeb = buf.f_bfree * buf.f_bsize;
    // Leave at least 1% space free (used to be 10% but that's excessive on very large disks, and unexpected)
    unsigned long limit = buf.f_blocks * buf.f_bsize / 100;
    debug("update_mwb(): free: %ul, limit: %ul", freeb, limit);
    if (limit > freeb) max_write_bytes = 0;
    else max_write_bytes = freeb - limit;
  }
  else {
    max_write_bytes = 0;
  }
  debug("update_mwb(): max_write_bytes: %ul", max_write_bytes);
  return;
}
 
void debug(const char *format, ...)
{
#ifdef DEBUG
  va_list ap;
  va_start(ap, format);

  if (do_debug) {
    vfprintf(stderr, format, ap);
    fprintf(stderr,"\n");
    fflush(NULL);
  }
  else {
    FILE *fp = fopen(log_file, "a");
    if (fp == NULL) {
      error("Cannot write to log file %s", log_file);
      return;
    }
    vfprintf(fp, format, ap);
    fprintf(fp, "\n");
    fclose(fp);
  }
  va_end(ap);

#else
  return;
#endif
}

void error(const char *format, ...)
{
  va_list ap;
  va_start(ap, format);

  if (do_debug) {
    vfprintf(stderr, format, ap);
    fprintf(stderr,"\n");
  }
  else {
    //vsyslog(LOG_ERR, format, ap);
    FILE *fp = fopen(log_file, "a");
    if (fp == NULL) {
      syslog(LOG_ERR, "Cannot write to log file %s", log_file);
      return;
    }
    vfprintf(fp, format, ap);
    fprintf(fp, "\n");
    fclose(fp);
  }
  va_end(ap);
}

#define PRINT_HELP 1

void print_help(char *argv[])
{
  error("usage: %s source_dir mountpoint [options]", argv[0]);
  error("options:");
  error("  -d        enable debug output");
  error("  -f        run in foreground");
  error("  -n        mount without writing in /etc/mtab");
  error("  -o opt,[opt...]   options");
}

void append_fuse_option(const char *opt)
{
  if (fuse_options == NULL) {
    fuse_options = malloc(strlen(opt) + 1);
    strcpy(fuse_options, opt);
  } else {
    // Allocate room for original string, comma, new string
    fuse_options = realloc(fuse_options, strlen(fuse_options) + 1
                       + 1
                       + strlen(opt) + 1);
    strcat(fuse_options, ",");
    strcat(fuse_options, opt);
  }
}

void parse_options(char *opt)
{
  char *cursor = NULL;
  char *ptr = NULL;

  ptr = strtok_r(opt, ",", &cursor);
  while(ptr) {
    if (strncmp(ptr,"rw",2) == 0) {
      // Ignore 'rw'.  It's inserted by autofs.
    } else if (strncmp(ptr,"source_user=",12) == 0) {
      strncpy(source_user,&ptr[12],sizeof(source_user));
      source_user[sizeof(source_user)-1] = 0;
    } else if (strncmp(ptr,"source_group=",13) == 0) {
      strncpy(source_group,&ptr[13],sizeof(source_group));
      source_group[sizeof(source_group)-1] = 0;
    } else if (strncmp(ptr,"log_file=",9) == 0) {
      strncpy(log_file,&ptr[9],sizeof(log_file));
      log_file[sizeof(log_file)-1] = 0;
    } else {
      append_fuse_option(ptr);
    }
    ptr = strtok_r(NULL, ",", &cursor);
  }
}

int main(int argc, char *argv[])
{
  struct fuse *fuse;
  int multithreaded;
  int res;
  int fd;

  openlog("projfs", 0, LOG_DAEMON);

  // Do we have the minimum args?
  if (argc < 3) {
    do_debug = 1;
    print_help(argv);
    return 1;
  }

  // Parse additional options
  extern int optind, opterr, optopt;
  struct option long_options[] = {
    // name, has_arg, flag, value
    {"help",          no_argument, 0,       PRINT_HELP},
    {"nomtab",        no_argument, &no_mtab,       'n'},
    {"debug",         no_argument, &do_debug,      'd'},
    {"foreground",    no_argument, &do_foreground, 'f'},
    {"sloppy",        no_argument, 0,              's'},
    {"options", required_argument, 0,              'o'},
    {0,0,0,0},
  };

  while(1) {
    int c;
    int option_index;
    c = getopt_long(argc, argv, "ndshfo:", long_options, &option_index);
    
    if (c == -1) {
      break;
    }

    switch(c) {
      case 0:
        break;
      case PRINT_HELP:
        print_help(argv);
        return 0;
      case 'n':
        no_mtab=1;
        break;
      case 'd':
        do_debug=1;
        do_foreground=1;
        break;
      case 'f':
        do_foreground=1;
        break;
      case 'o':
        parse_options(optarg);
        break;
      case 's':
        break;
      default:
        print_help(argv);
        return 1;
    }
  }
  if (argc - optind != 2) {
    print_help(argv);
    return 1;
  }

  source_dir = strdup(argv[optind++]);
  const char *target_dir = argv[optind]; // only used temporarily for init

#ifdef DEBUG_ARGS
  error("source_dir = '%s'", source_dir);
  error("mountpoint = '%s'", target_dir);
  if (source_user[0] != '\0') error("source_user = '%s'", source_user);
  if (source_group[0] != '\0') error("source_group = '%s'", source_group);
  if (log_file[0] != '\0') error("log_file = '%s'", log_file);
  if (do_debug) error("debug");
  if (do_foreground) error("foreground");
#endif

  // Setup argc/argv for FUSE
  char **new_argv = malloc(2*sizeof(char *));
  new_argv[0] = strdup(argv[0]);
  new_argv[1] = strdup(target_dir);
  int new_argc = 2;

  if (do_debug) {
    new_argv = realloc(new_argv, (new_argc+1) * sizeof(char *));
    new_argv[new_argc] = strdup("-d");
    new_argc++;
  }

  if (do_foreground) {
    new_argv = realloc(new_argv, (new_argc+1) * sizeof(char *));
    new_argv[new_argc] = strdup("-f");
    new_argc++;
  }

  if (fuse_options != NULL) {
    new_argv = realloc(new_argv, (new_argc+2) * sizeof(char *));
    new_argv[new_argc] = strdup("-o");
    new_argv[new_argc+1] = strdup(fuse_options);
    new_argc += 2;
  }

#ifdef DEBUG_ARGS
  int i;
  for(i=0; i<new_argc; i++) {
    error("%2d: '%s'", i, new_argv[i]);
  }
  //return 0;
#endif

  // Make sure source_user is valid
  struct passwd *passwd = getpwnam(source_user);
  if (passwd == NULL) {
    error("User '%s' is not known.", source_user);
    return 1;
  }
  source_uid = passwd->pw_uid;
  source_gid = passwd->pw_gid;

  // Relative paths are not okay
  if (source_dir[0] != '/') {
    free(source_dir);
    source_dir = strdup(passwd->pw_dir);
  }

  // Make sure that source_dir exists...
  struct stat stat_buf;
  res = stat(source_dir, &stat_buf);
  if (res < 0) {
    error("Directory '%s' does not exist.", source_dir);
    return 1;
  }

  if (S_ISDIR(stat_buf.st_mode) == 0) {
    error("'%s' is not a directory.", source_dir);
    return 1;
  }

  if (source_group[0] != '\0') {
    struct group *group = getgrnam(source_group);
    if (group == NULL) {
      error("Group '%s' is not known.", source_group);
      return 1;
    }
    source_gid = group->gr_gid;
  }

  //-------------------------------------------------------------------------
  // The regular fuse setup.
  //-------------------------------------------------------------------------
  fuse = fuse_setup(new_argc, new_argv, &xmp_oper, sizeof(xmp_oper),
            &mountpoint, &multithreaded, &fd);

  if (fuse == NULL) {
    return 1;
  }

  umask(0);

  res = setfsuid(source_uid);
  if (res < 0) {
    perror("setfsuid");
    return 1;
  }

  // TODO: We used to chroot() here. But that doesn't work.
  /* Going theory: used to initgroups() and then chroot(), but now we have
   * any number of users that are unknown a priori. Cannot access auth entities
   * from the chroot()'d environment
   */

  res = setregid(source_gid,source_gid);
  if (res < 0) {
    perror("setregid");
    return 1;
  }

  res = setreuid(source_uid,source_uid);
  if (res < 0) {
    perror("setreuid");
    return 1;
  }

  // Single-threaded loop.
  res = fuse_loop(fuse);

  fuse_teardown(fuse, fd, mountpoint);
  if (res == -1)
    return 1;

  return 0;
}
