# @package      hubzero-mw2-common
# @file         app.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import re
from constants import GEOM_REGEXP, CMD_REGEXP
from log import log
from errors import SessionError

DEBUG = False

class App:
  """Application data and requirements to be run with VNC.
  Part of the rationale for creating this class is that the previous code used a list
  that was repeatedly packed and unpacked, resulting in unused variables being
  flagged in the static analysis.  This could obscure bugs.  It is not in its own file because
  it is used only by Session."""

  def __init__(self, db, appname, app_k, mysql_prefix, row = None):
    """Get the necessary information about an application from the DB.
    Note:  replace the appname with the one that's in the database instead of the one passed to us,
    to resolve capitalization issues.  The CMS can pass us randomly capitalized names.  This causes problems
    in the subsequent statistical analysis."""
    self.mysql_prefix = mysql_prefix
    self.K = app_k
    self.db = db
    if appname is not None:
      self.appname, self.tool_id, self.geometry, self.depth, self.command, self.timeout, self.appfullname = self.get_tool_info(appname)

      # validate command
      if re.match(CMD_REGEXP, self.command) is None:
        raise SessionError("Bad command for appname='%s'" % (self.appname))

      # validate timeout
      # this is used by expire-sessions.py to determine when to kill an abandoned session
      try:
        self.timeout = int(self.timeout)
      except (ValueError, TypeError):
        self.timeout = app_k["TIMEOUT_DEFAULT"]

      self.appfullname = str(self.appfullname)
      if self.appfullname == 'None' or self.appfullname == '':
        log("Bad title for appname='%s'" % self.appname)
        self.appfullname = self.appname
    elif row is not None:
      # expect row coming from get_all_apps
      self.tool_id = row[0]
      self.geometry = row[1]
      self.depth = row[2]
      appname = row[3]
    else:
      raise SessionError("Don't know how to construct App object")

    if self.tool_id is None:
      raise SessionError("Database integrity, can't get id for application named '%s'" % self.appname)
    # validate geometry, expecting number x number
    if re.match(GEOM_REGEXP, self.geometry) is None:
      raise SessionError("Bad geometry for appname='%s'" % (self.appname))
    # validate depth
    try:
      self.depth = int(self.depth)
    except (ValueError, TypeError):
      self.depth = app_k["DEPTH_DEFAULT"]

    self.hostreq = self.sum_req(self.tool_id)
    if self.hostreq is None:
      if DEBUG:
        log("hostreq not specified, using default")
      self.hostreq = self.sum_default_req()
    # make hostreq into an integer and validate
    try:
      self.hostreq = int(self.hostreq)
    except (ValueError, TypeError):
      raise SessionError("Bad hostreq for appname='%s'" % (self.appname))
    if DEBUG:
      log("got hostreq=%d for appname='%s'"% (self.hostreq, self.appname))

  def get_app_table(self, db):
    """Deprecated.  Old app table.  Kept for compatibility with old versions of HUBzero."""
    arr = db.getall("""
      SELECT geometry,depth,command,hostreq,timeout,description
      FROM app WHERE appname=%s""",
      [self.appname])

    if len(arr) < 1:
      raise SessionError("Cannot find application named '%s'" % self.appname)
    elif len(arr) > 1:
      raise SessionError("Multiple applications match appname '%s'" % self.appname)
    return arr[0]

  def sum_req(self, tool_id):
    hosttable = self.mysql_prefix + self.K["HOST_TABLE"]
    cmd = """SELECT SUM(value)
      FROM hosttype JOIN %s ON hosttype.name = %s.hostreq
      WHERE %s.tool_version_id =""" % (hosttable, hosttable, hosttable)
    return self.db.getsingle(cmd + "%s", [tool_id])

  def sum_default_req(self):
    # "SELECT SUM(value) FROM hosttype WHERE name='a' OR name='b' OR name='c'"
    # example row: | sessions   |    16 | Supports app sessions       |
    search = "SELECT SUM(value) FROM hosttype WHERE name='" + \
           "' OR name='".join(self.K["HOST_REQ_DEFAULTS"]) + "'"

    hostreq = self.db.getsingle(search, [])
    if hostreq is None:
      raise SessionError("Unable to calculate hostreq from table hosttype")
    return hostreq

  def get_tool_info(self, appname):
    """Newer storage schema for app information, separates hostreq into a separate table.  Both the old
    and new functions are provided to avoid breaking Hubs during the transition.
    The advantage of this schema is the flexible definition of categories of host
    requirements (e.g., "openvz").  For each tool, a "usage category" is selected, but the precise
    requirements of each category are defined elsewhere and only once.

    hostreq's are strings.  To convert to a number, need to sum the numeric ids
    corresponding to each string in the hosttype table.
    """
    tooltable = self.mysql_prefix + self.K["TOOL_TABLE"]

    # support for separating arguments from SQL doesn't work with table names, so split command in two
    cmd = "SELECT instance,id,vnc_geometry,vnc_depth,vnc_command,vnc_timeout,title FROM %s" % tooltable
    tool_arr = self.db.getall(cmd + """ WHERE instance=%s and state is not NULL""", [appname])
    if len(tool_arr) < 1:
      raise SessionError("Cannot find application named '%s'" % appname)
    elif len(tool_arr) > 1:
      raise SessionError("Multiple applications match appname '%s'" % appname)

    row = tool_arr[0]
    if DEBUG:
      log("app:" + str(row))
    return row

  @staticmethod
  def get_all_apps(db, mysql_prefix, app_k):
    cmd = "SELECT id, vnc_geometry, vnc_depth, instance FROM %s" % mysql_prefix + app_k["TOOL_TABLE"]
    return db.getall(cmd, [])
