# @package      hubzero-mw2-common
# @file         vnc_pass.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

class VNC_pass:
  """
  VNC password utility.  Goal: generate a hex encoded, semi-DES encrypted password for use by VNC.
  VNC always uses the same "Obfuscation Key": 23,82,107,6,35,78,88,7
  However, plugging this in to a general DES routine doesn't work.
  Nor does the DES implementation in VNC pass its own validation sets.
  The following code provides expanded DES permutation keys and an
  implementation of DES as it exists in the vncpasswd mechanism.
  """
  Encrypt_key = [690692107, 959852042, 1008472116, 219877899, \
                 422382336, 420946204, 153292807, 1058477350, \
                 253165858, 909248553, 1058418737, 607723797, \
                 923533838, 941439032, 53085236, 958338305, \
                 1042157353, 369688836, 339616771, 421334066, \
                 354615608, 454498817, 221457937, 1042686480, \
                 1041631756, 640031750, 1042023698, 138216962, \
                 822622977, 423559173, 1042225154, 237647920 ]

  Decrypt_key = [1042225154, 237647920, 822622977, 423559173, \
                 1042023698, 138216962, 1041631756, 640031750, \
                 221457937, 1042686480, 354615608, 454498817, \
                 339616771, 421334066, 1042157353, 369688836, \
                 53085236, 958338305, 923533838, 941439032, \
                 1058418737, 607723797, 253165858, 909248553, \
                 153292807, 1058477350, 422382336, 420946204, \
                 1008472116, 219877899, 690692107, 959852042 ]

  SP1 = [ 0x01010400L, 0x00000000L, 0x00010000L, 0x01010404L, \
          0x01010004L, 0x00010404L, 0x00000004L, 0x00010000L, \
          0x00000400L, 0x01010400L, 0x01010404L, 0x00000400L, \
          0x01000404L, 0x01010004L, 0x01000000L, 0x00000004L, \
          0x00000404L, 0x01000400L, 0x01000400L, 0x00010400L, \
          0x00010400L, 0x01010000L, 0x01010000L, 0x01000404L, \
          0x00010004L, 0x01000004L, 0x01000004L, 0x00010004L, \
          0x00000000L, 0x00000404L, 0x00010404L, 0x01000000L, \
          0x00010000L, 0x01010404L, 0x00000004L, 0x01010000L, \
          0x01010400L, 0x01000000L, 0x01000000L, 0x00000400L, \
          0x01010004L, 0x00010000L, 0x00010400L, 0x01000004L, \
          0x00000400L, 0x00000004L, 0x01000404L, 0x00010404L, \
          0x01010404L, 0x00010004L, 0x01010000L, 0x01000404L, \
          0x01000004L, 0x00000404L, 0x00010404L, 0x01010400L, \
          0x00000404L, 0x01000400L, 0x01000400L, 0x00000000L, \
          0x00010004L, 0x00010400L, 0x00000000L, 0x01010004L ]

  SP2 = [ 0x80108020L, 0x80008000L, 0x00008000L, 0x00108020L, \
          0x00100000L, 0x00000020L, 0x80100020L, 0x80008020L, \
          0x80000020L, 0x80108020L, 0x80108000L, 0x80000000L, \
          0x80008000L, 0x00100000L, 0x00000020L, 0x80100020L, \
          0x00108000L, 0x00100020L, 0x80008020L, 0x00000000L, \
          0x80000000L, 0x00008000L, 0x00108020L, 0x80100000L, \
          0x00100020L, 0x80000020L, 0x00000000L, 0x00108000L, \
          0x00008020L, 0x80108000L, 0x80100000L, 0x00008020L, \
          0x00000000L, 0x00108020L, 0x80100020L, 0x00100000L, \
          0x80008020L, 0x80100000L, 0x80108000L, 0x00008000L, \
          0x80100000L, 0x80008000L, 0x00000020L, 0x80108020L, \
          0x00108020L, 0x00000020L, 0x00008000L, 0x80000000L, \
          0x00008020L, 0x80108000L, 0x00100000L, 0x80000020L, \
          0x00100020L, 0x80008020L, 0x80000020L, 0x00100020L, \
          0x00108000L, 0x00000000L, 0x80008000L, 0x00008020L, \
          0x80000000L, 0x80100020L, 0x80108020L, 0x00108000L ]

  SP3 = [ 0x00000208L, 0x08020200L, 0x00000000L, 0x08020008L, \
          0x08000200L, 0x00000000L, 0x00020208L, 0x08000200L, \
          0x00020008L, 0x08000008L, 0x08000008L, 0x00020000L, \
          0x08020208L, 0x00020008L, 0x08020000L, 0x00000208L, \
          0x08000000L, 0x00000008L, 0x08020200L, 0x00000200L, \
          0x00020200L, 0x08020000L, 0x08020008L, 0x00020208L, \
          0x08000208L, 0x00020200L, 0x00020000L, 0x08000208L, \
          0x00000008L, 0x08020208L, 0x00000200L, 0x08000000L, \
          0x08020200L, 0x08000000L, 0x00020008L, 0x00000208L, \
          0x00020000L, 0x08020200L, 0x08000200L, 0x00000000L, \
          0x00000200L, 0x00020008L, 0x08020208L, 0x08000200L, \
          0x08000008L, 0x00000200L, 0x00000000L, 0x08020008L, \
          0x08000208L, 0x00020000L, 0x08000000L, 0x08020208L, \
          0x00000008L, 0x00020208L, 0x00020200L, 0x08000008L, \
          0x08020000L, 0x08000208L, 0x00000208L, 0x08020000L, \
          0x00020208L, 0x00000008L, 0x08020008L, 0x00020200L ]

  SP4 = [ 0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L, \
          0x00802080L, 0x00800081L, 0x00800001L, 0x00002001L, \
          0x00000000L, 0x00802000L, 0x00802000L, 0x00802081L, \
          0x00000081L, 0x00000000L, 0x00800080L, 0x00800001L, \
          0x00000001L, 0x00002000L, 0x00800000L, 0x00802001L, \
          0x00000080L, 0x00800000L, 0x00002001L, 0x00002080L, \
          0x00800081L, 0x00000001L, 0x00002080L, 0x00800080L, \
          0x00002000L, 0x00802080L, 0x00802081L, 0x00000081L, \
          0x00800080L, 0x00800001L, 0x00802000L, 0x00802081L, \
          0x00000081L, 0x00000000L, 0x00000000L, 0x00802000L, \
          0x00002080L, 0x00800080L, 0x00800081L, 0x00000001L, \
          0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L, \
          0x00802081L, 0x00000081L, 0x00000001L, 0x00002000L, \
          0x00800001L, 0x00002001L, 0x00802080L, 0x00800081L, \
          0x00002001L, 0x00002080L, 0x00800000L, 0x00802001L, \
          0x00000080L, 0x00800000L, 0x00002000L, 0x00802080L ]

  SP5 = [ 0x00000100L, 0x02080100L, 0x02080000L, 0x42000100L, \
          0x00080000L, 0x00000100L, 0x40000000L, 0x02080000L, \
          0x40080100L, 0x00080000L, 0x02000100L, 0x40080100L, \
          0x42000100L, 0x42080000L, 0x00080100L, 0x40000000L, \
          0x02000000L, 0x40080000L, 0x40080000L, 0x00000000L, \
          0x40000100L, 0x42080100L, 0x42080100L, 0x02000100L, \
          0x42080000L, 0x40000100L, 0x00000000L, 0x42000000L, \
          0x02080100L, 0x02000000L, 0x42000000L, 0x00080100L, \
          0x00080000L, 0x42000100L, 0x00000100L, 0x02000000L, \
          0x40000000L, 0x02080000L, 0x42000100L, 0x40080100L, \
          0x02000100L, 0x40000000L, 0x42080000L, 0x02080100L, \
          0x40080100L, 0x00000100L, 0x02000000L, 0x42080000L, \
          0x42080100L, 0x00080100L, 0x42000000L, 0x42080100L, \
          0x02080000L, 0x00000000L, 0x40080000L, 0x42000000L, \
          0x00080100L, 0x02000100L, 0x40000100L, 0x00080000L, \
          0x00000000L, 0x40080000L, 0x02080100L, 0x40000100L ]

  SP6 = [ 0x20000010L, 0x20400000L, 0x00004000L, 0x20404010L, \
          0x20400000L, 0x00000010L, 0x20404010L, 0x00400000L, \
          0x20004000L, 0x00404010L, 0x00400000L, 0x20000010L, \
          0x00400010L, 0x20004000L, 0x20000000L, 0x00004010L, \
          0x00000000L, 0x00400010L, 0x20004010L, 0x00004000L, \
          0x00404000L, 0x20004010L, 0x00000010L, 0x20400010L, \
          0x20400010L, 0x00000000L, 0x00404010L, 0x20404000L, \
          0x00004010L, 0x00404000L, 0x20404000L, 0x20000000L, \
          0x20004000L, 0x00000010L, 0x20400010L, 0x00404000L, \
          0x20404010L, 0x00400000L, 0x00004010L, 0x20000010L, \
          0x00400000L, 0x20004000L, 0x20000000L, 0x00004010L, \
          0x20000010L, 0x20404010L, 0x00404000L, 0x20400000L, \
          0x00404010L, 0x20404000L, 0x00000000L, 0x20400010L, \
          0x00000010L, 0x00004000L, 0x20400000L, 0x00404010L, \
          0x00004000L, 0x00400010L, 0x20004010L, 0x00000000L, \
          0x20404000L, 0x20000000L, 0x00400010L, 0x20004010L ]

  SP7 = [ 0x00200000L, 0x04200002L, 0x04000802L, 0x00000000L, \
          0x00000800L, 0x04000802L, 0x00200802L, 0x04200800L, \
          0x04200802L, 0x00200000L, 0x00000000L, 0x04000002L, \
          0x00000002L, 0x04000000L, 0x04200002L, 0x00000802L, \
          0x04000800L, 0x00200802L, 0x00200002L, 0x04000800L, \
          0x04000002L, 0x04200000L, 0x04200800L, 0x00200002L, \
          0x04200000L, 0x00000800L, 0x00000802L, 0x04200802L, \
          0x00200800L, 0x00000002L, 0x04000000L, 0x00200800L, \
          0x04000000L, 0x00200800L, 0x00200000L, 0x04000802L, \
          0x04000802L, 0x04200002L, 0x04200002L, 0x00000002L, \
          0x00200002L, 0x04000000L, 0x04000800L, 0x00200000L, \
          0x04200800L, 0x00000802L, 0x00200802L, 0x04200800L, \
          0x00000802L, 0x04000002L, 0x04200802L, 0x04200000L, \
          0x00200800L, 0x00000000L, 0x00000002L, 0x04200802L, \
          0x00000000L, 0x00200802L, 0x04200000L, 0x00000800L, \
          0x04000002L, 0x04000800L, 0x00000800L, 0x00200002L ]

  SP8 = [ 0x10001040L, 0x00001000L, 0x00040000L, 0x10041040L, \
          0x10000000L, 0x10001040L, 0x00000040L, 0x10000000L, \
          0x00040040L, 0x10040000L, 0x10041040L, 0x00041000L, \
          0x10041000L, 0x00041040L, 0x00001000L, 0x00000040L, \
          0x10040000L, 0x10000040L, 0x10001000L, 0x00001040L, \
          0x00041000L, 0x00040040L, 0x10040040L, 0x10041000L, \
          0x00001040L, 0x00000000L, 0x00000000L, 0x10040040L, \
          0x10000040L, 0x10001000L, 0x00041040L, 0x00040000L, \
          0x00041040L, 0x00040000L, 0x10041000L, 0x00001000L, \
          0x00000040L, 0x10040040L, 0x00001000L, 0x00041040L, \
          0x10001000L, 0x00000040L, 0x10000040L, 0x10040000L, \
          0x10040040L, 0x10000000L, 0x00040000L, 0x10001040L, \
          0x00000000L, 0x10041040L, 0x00040040L, 0x10000040L, \
          0x10040000L, 0x10001000L, 0x10001040L, 0x00000000L, \
          0x10041040L, 0x00041000L, 0x00041000L, 0x00001040L, \
          0x00001040L, 0x00040040L, 0x10000000L, 0x10041000L ]

  def vncdes(self, block, keys):
    leftt = block[0]
    right = block[1]
    work = ((leftt >> 4) ^ right) & 0x0f0f0f0fL
    right ^= work
    leftt ^= (work << 4)
    work = ((leftt >> 16) ^ right) & 0x0000ffffL
    right ^= work
    leftt ^= (work << 16)
    work = ((right >> 2) ^ leftt) & 0x33333333L
    leftt ^= work
    right ^= (work << 2)
    work = ((right >> 8) ^ leftt) & 0x00ff00ffL
    leftt ^= work
    right ^= (work << 8)
    right = ((right << 1) | ((right >> 31) & 1L)) & 0xffffffffL
    work = (leftt ^ right) & 0xaaaaaaaaL
    leftt ^= work
    right ^= work
    leftt = ((leftt << 1) | ((leftt >> 31) & 1L)) & 0xffffffffL

    #print "leftt = %x" % leftt
    #print "right = %x" % right
    #print "len(keys) = %d" % len(keys)

    for enc_round in range(0,8):
      #print "round %d" % round
      work  = (right << 28) | (right >> 4)
      work ^= keys[enc_round*4+0]
      fval  = self.SP7[ work                & 0x3fL]
      fval |= self.SP5[(work >>  8) & 0x3fL]
      fval |= self.SP3[(work >> 16) & 0x3fL]
      fval |= self.SP1[(work >> 24) & 0x3fL]
      work  = right ^ keys[enc_round*4+1]
      fval |= self.SP8[ work                & 0x3fL]
      fval |= self.SP6[(work >>  8) & 0x3fL]
      fval |= self.SP4[(work >> 16) & 0x3fL]
      fval |= self.SP2[(work >> 24) & 0x3fL]
      leftt ^= fval
      work  = (leftt << 28) | (leftt >> 4)
      work ^= keys[enc_round*4+2]
      fval  = self.SP7[ work                & 0x3fL]
      fval |= self.SP5[(work >>  8) & 0x3fL]
      fval |= self.SP3[(work >> 16) & 0x3fL]
      fval |= self.SP1[(work >> 24) & 0x3fL]
      work  = leftt ^ keys[enc_round*4+3]
      fval |= self.SP8[ work                & 0x3fL]
      fval |= self.SP6[(work >>  8) & 0x3fL]
      fval |= self.SP4[(work >> 16) & 0x3fL]
      fval |= self.SP2[(work >> 24) & 0x3fL]
      right ^= fval
      #print "leftt = %x right = %x" % (leftt,right)

    right = ((right << 31) | (right >> 1)) & 0x0ffffffffL
    work = (leftt ^ right) & 0xaaaaaaaaL
    leftt ^= work
    right ^= work
    leftt = ((leftt << 31) | (leftt >> 1)) & 0x0ffffffffL
    work = ((leftt >> 8) ^ right) & 0x00ff00ffL
    right ^= work
    leftt ^= (work << 8)
    work = ((leftt >> 2) ^ right) & 0x33333333L
    right ^= work
    leftt ^= (work << 2)
    work = ((right >> 16) ^ leftt) & 0x0000ffffL
    leftt ^= work
    right ^= (work << 16)
    work = ((right >> 4) ^ leftt) & 0x0f0f0f0fL
    leftt ^= work
    right ^= (work << 4)
    #print "leftt = %x right = %x" % (leftt,right)
    return right,leftt

  def text_to_integers(self, input_text):
    input_text = input_text[0:8]
    while len(input_text) < 8:
      input_text += chr(0)
    compact = [ 0, 0]
    compact[0] |= ord(input_text[0]) << 24
    compact[0] |= ord(input_text[1]) << 16
    compact[0] |= ord(input_text[2]) << 8
    compact[0] |= ord(input_text[3]) << 0
    compact[1] |= ord(input_text[4]) << 24
    compact[1] |= ord(input_text[5]) << 16
    compact[1] |= ord(input_text[6]) << 8
    compact[1] |= ord(input_text[7]) << 0
    return compact

  def integers_to_text(self, compact):
    output_text = ""
    output_text += chr((compact[0] >> 24) & 0x00ff)
    output_text += chr((compact[0] >> 16) & 0x00ff)
    output_text += chr((compact[0] >>  8) & 0x00ff)
    output_text += chr((compact[0] >>  0) & 0x00ff)
    output_text += chr((compact[1] >> 24) & 0x00ff)
    output_text += chr((compact[1] >> 16) & 0x00ff)
    output_text += chr((compact[1] >>  8) & 0x00ff)
    output_text += chr((compact[1] >>  0) & 0x00ff)
    return output_text

  def encrypt(self, cleartext):
    int_text = self.text_to_integers(cleartext)
    output = self.vncdes(int_text,self.Encrypt_key)
    output_text = self.integers_to_text(output)
    return output_text

  def decrypt(self, ciphertext):
    int_text = self.text_to_integers(ciphertext)
    output = self.vncdes(int_text,self.Decrypt_key)
    output_text = self.integers_to_text(output)
    return output_text

  def tohex(self, v):
    s = ""
    for x in v:
      s += "%02x" % ord(x)
    return s

  def fromhex(self, h):
    s = ""
    for i in range(0,16,2):
      s += chr(int("0x" + h[i:i+2],16))
    return s
