# @package      hubzero-mw2-common
# @file         mw_db.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
Communicate with MySQL database while setting, holding and releasing the 
appropriate locks, and checking for errors
"""

import MySQLdb # linux execution hosts
import time
from errors import MaxwellError
from log import log

class MW_DB:
  """Wrapper for database calls.  The main motivation for making this into a class
  (besides tidying up and enabling unit testing) is to get rid of a global, "lock_held".
  """

  def __init__(self, mysql_host, mysql_user, mysql_password, mysql_db, mysql_connect_attempts = 10):
    """Create database connection."""
    self.lock_held = None
    self.host = mysql_host
    self.user = mysql_user
    self.passwd = mysql_password
    self.dbname = mysql_db
    self.db = None
    self.retries = mysql_connect_attempts
    for dummy in range(0, self.retries):
      try:
        self.db = MySQLdb.connect(host=self.host, user=self.user, passwd=self.passwd, db=self.dbname)
        #log("db_connect finished on iteration %d" % x)
        self.c = self.db.cursor()
        return
      except MySQLdb.MySQLError, e:
        # MySQLdb.MySQLError is the base exception class for everything related to MySQL ops
        # if something worse happens there's no point trying again.
        log("Exception in db_connect %s" % e)
      time.sleep(1)
    raise MaxwellError("Unable to connect to database")

  def cursor(self):
    self.c = self.db.cursor()

  def close(self):
    self.db.close()
    self.c = None
    self.db = None

  def getall(self, cmd, params):
    for dummy in range(0, self.retries):
      try:
        self.c.execute(cmd, params)
        return self.c.fetchall()
      except MySQLdb.MySQLError, e:
        log("MySQL Exception %s" % e)
        self.db = MySQLdb.connect(host=self.host, user=self.user, passwd=self.passwd, db=self.dbname)
        self.c = self.db.cursor()
    raise MaxwellError("Unable to execute query")

  def getrow(self, cmd, params):
    for dummy in range(0, self.retries):
      try:
        self.c.execute(cmd, params)
        return self.c.fetchone()
      except MySQLdb.MySQLError, e:
        log("MySQL Exception %s" % e)
        self.db = MySQLdb.connect(host=self.host, user=self.user, passwd=self.passwd, db=self.dbname)
        self.c = self.db.cursor()
    raise MaxwellError("Unable to execute query")

  def getsingle(self, cmd, params):
    for dummy in range(0, self.retries):
      try:
        self.c.execute(cmd, params)
        row = self.c.fetchone()
        if row is None:
          return None
        return row[0]
      except MySQLdb.MySQLError, e:
        log("MySQL Exception %s" % e)
        self.db = MySQLdb.connect(host=self.host, user=self.user, passwd=self.passwd, db=self.dbname)
        self.c = self.db.cursor()
    raise MaxwellError("Unable to execute query")

  # the following is dependent on MySQL... Only one lock at a time, identified by string
  # postgresql allows any number of locks, but uses 2 ints or a bigint instead.
  # this could be made less dependent on MySQL by printing an int?

  def lock(self, name):
    """Get a MySQL advisory lock. Does not return a value."""

    if name == self.lock_held:
      return

    if self.lock_held is not None:
      # can only hold one advisory lock at a time!  Works globally, but if this process sets a new
      # lock, the previous is lost.
      raise MaxwellError("Already got lock '%s' when trying to get '%s'" % (self.lock_held, name))

    iteration = 0
    lock_r = 0
    while lock_r != 1:
      iteration += 1
      if iteration > 1:
        if iteration > 10:
          # 10 log entries about contention is enough
          raise MaxwellError("can't get lock, giving up")
        else:
          log("Contention while waiting on lock %s" % name)

      # get_lock returns 1 if lock was obtained, 0 or NULL otherwise
      lock_r = self.getsingle("""SELECT get_lock('%s',5)""" % name, ())
    self.lock_held = name

  def unlock(self, name):
    """Release a MySQL advisory lock. Does not return a value."""

    if self.lock_held != name:
      raise MaxwellError("lock held is '%s', not '%s'", (self.lock_held, name))
    self.lock_held = None

    try:
      lock_r = self.getsingle("SELECT release_lock('%s')" % name, ())
      # release_lock returns 1 if lock was released, 0 or NULL otherwise
      if lock_r != 1:
        log("Error c.unlocking %s" % name)
    except MySQLdb.MySQLError, ex:
      # log and continue, hoping for the best, as we're done with this lock.
      # if anything went wrong, the damage has already been done.
      log("Error c.unlocking '%s': '%s'." % (name, ex))


