# @package      hubzero-mw2-common
# @file         win_container.py
# @copyright    Copyright (c) 2015-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Brian Rohler, Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2015-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

"""
For containers running under Virtuozzo, on Windows execution hosts.
This file was created for the NEES project (nees.org) and has not been 
maintained since NEES stopped using HUBzero middleware (circa 2015).  
It is provided for historical interest and in case someone is interested
in supporting Windows execution hosts.  Use at your own risk.
"""
import subprocess
import socket
import os
import stat
import sys
import time
import string
import hashlib

from errors import MaxwellError
from constants import CONTAINER_K, VERBOSE
from log import log, log_exc

class Container():
  """A container: Virtual Private Server (VPS) uniquely identified by a veid.
  Note that new documentation uses "CTID" instead of veid: ConTainer's IDentifer (CTID)
  According to the OpenVZ Users Guide, CTIDs 0-100 are reserved and should not be used.
  OpenVZ only currently uses CTID 0 but recommends reserving CTID's 0-100 for non-use

  Paths used by OpenVZ  (see OpenVZ User's Guide Version 2.7.0-8 by SWsoft):

  ct_private  (e.g., "/vz/private/veid")
  This is a path to the Container private area where Virtuozzo Containers 4.0 keeps its private data.

  ct_root (e.g., "/vz/root/veid")
  This is a path to the Container root folder where the Container private area is mounted.

  vz_root
  This is a path to the Virtuozzo folder where Virtuozzo Containers program files are located.

  depends on script MW_LIB_PATH +"mergeauth"

"""

  def __init__(self, disp, machine_number, user, vncdir, overrides={}):
    self.K = CONTAINER_K
    self.K.update(overrides)
    self.vncdir = vncdir
    self.disp = disp
    self.vncpass = None
    self.veid = disp + self.K["VZOFFSET"]
    if False: #self.veid <= 100:
      raise MaxwellError(
        "Container IDs (CTIDs, VEIDs) 0-100 are reserved (got %d).  Is vzoffset set correctly?"\
        % self.veid)
    self.vz_private_path = "%s/private/%d" % (self.K["VZ_PATH"], self.veid)
    self.vz_root_path = "%s/root/%d" % (self.K["VZ_PATH"], self.veid)
    if (machine_number == 0):
      try:
        ## FIXIT: brohler 02-06-11 fix after release 2
        ## Server users two NICS, need to determine which one is 128.46.x.x
        ## machine_number = int(socket.gethostbyname(socket.getfqdn()).split('.')[3])
        ## Current method below is a hack.
        machine_number = int([ip for ip in socket.gethostbyname_ex(socket.gethostname())[2] if not ip.startswith("192.")][0].split('.')[3])
      except StandardError:
        raise MaxwellError("machine_number not set and unable to derive one from IP address.")
    if (machine_number == 0):
      raise MaxwellError("unable to set machine_number")

    self.veaddr = "10.%d.%d.%d" % (machine_number, self.veid/100, self.veid % 100)
    #if VERBOSE:
    #  log("machine number is %d" % machine_number)

  #=============================================================================
  # Start the command running and watch the view time.
  #=============================================================================
  def invoke_unix_command(self, user, session_id, timeout, command):
    """
    1. Setup the environment inside the container: permissions, password, group files,
    2. Setup the firewall rules on the host
    3. Setup X server authentication.  Call xauth so we're allowed to connect to the X server
    4. Invoke the command within the container
    5. Calculate time stats
    6. Restore the firewall rules
    
    Child will invoke the command.
    Parent will handle the timeout.
    When we are called, the log file has been closed and we're a dissociated process.
    Stdout and stderr have been redirected to files, so we use that for logging.
    user: string
    session_id: string (int+letter)
    timeout: int
    command: string
    """

    def printvzstats(tool_pid):
      """Print statistics for an OpenVZ VPS."""
      try:
        args = ['c:/windows/system32/Wbem/wmic.exe', 'process', 'where', 'processid ='+str(tool_pid), 'get', "processid,kernelmodetime,usermodetime"]
        process = subprocess.Popen(
          args,
          stdin = None,
          stdout = subprocess.PIPE,
          stderr = subprocess.PIPE
        )
        (stdout, stderr) = process.communicate(None)
        if (process.returncode == 0) and (stderr == ""):
          output = stdout.rstrip().split()
        else: 
          # force return variables to zero
          stdout = "0  0  0  0  0  0"
          output = stdout.rstrip().split()
      except OSError: # tool probably killed so no pid to check against, send back zeros
          pass
      return((int(output[3])/10000000.0),(int(output[5])/10000000.0))

    setup_start_time = time.time()
    log("Starting command '%s' for '%s' with timeout '%s'" % (command, user, timeout))

    # Generate user Samba password
    sambaUser = user
    mySuperSecret = CONTAINER_K["WIN_SMB_SECRET"]
    h = hashlib.new('ripemd160')  # Chosen because it generates 40 character digests
    h.update(mySuperSecret + sambaUser)
    sambaPassword = h.hexdigest()
    # Connect users home directory to H: by updating users c:\CTID_command\map_home_dir.cmd file
    # Write the tool execution command to the same c:\CTID_command\map_home_dir.cmd used above for home dir mapping.
    # Filewatcher will execute command both commands within container once file is updated
    try:
      filename = self.K["WIN_HOME_DRIVE_MAP_PATH_A"] + str(self.veid) + self.K["WIN_HOME_DRIVE_MAP_PATH_B"]
      f = open(filename,"w")
      home_dir_output = "c:\windows\system32\\net.exe use %s \\\\%s\%s /USER:%s %s /PERSISTENT:NO\n" % (self.K["WIN_HOME_DRIVE"], \
      self.K["WIN_HOME_DIR_SERVER"],str(sambaUser),str(sambaUser),str(sambaPassword))
      f.write(home_dir_output)
      f.write("start /MAX /b %s" % command)
      f.close()
    except OSError:
      raise MaxwellError("Unable to write net use command to %s CTID_command directory" % str(self.veid))

    ############################################
    # Tool has executed, start capturing stats #
    ############################################
    # Capture and write tool stats
    start_time = time.time() # Capture start time (real)
    
    # Filewatcher will execute command within container once file is updated.
    # Since not launching tool using vzctl, need a way to stay in the python 
    # script until user exits tool session

    # Capture PID of executed tool
    toolname = (command.upper()).split("/")[-1]

    ############################################################################
    # Check inside container for tool process to start so PID can be captured. #
    # Caveat - this brings up a small blue dialog box within VNC window, it's  #
    # ugly but until Parallels provides a way to know which container owns     #
    # which PID this is the only viable solution. A feature request has been   #
    # submitted                                                                #
    ############################################################################
    time_counter = 0  # wait up to "time_counter_max" seconds for tool to execute
    time_counter_max = 10 # number of times through the time_counter loop
    while (time_counter < time_counter_max):
      time.sleep(1) # wait loop in seconds
      # Check within container for tool PID to be available
      args = ['vzctl', 'exec', str(self.veid), 'tasklist', '/svc', '/NH','/FI', "imagename eq %s" % str(toolname.upper())]
      process = subprocess.Popen(
        args,
        stdin = None,
        stdout = subprocess.PIPE,
        stderr = subprocess.PIPE
      )
      (stdout, stderr) = process.communicate(None)

      if process.returncode != 0:
        raise MaxwellError("Capture '%s' PID error: '%s'" % (toolname, stderr))

      if (process.returncode == 0) and (stdout != ""):
        if str(stdout.upper()).split()[0] == toolname:
          pid = str(stdout.upper()).split()[1]
          break
      time_counter = time_counter + 1

    # IF PID not found, time_counter will be 0
    if time_counter == time_counter_max:
      raise MaxwellError("Tool '%s' never executed or took too long to execute" % (toolname))    

    # Capture "user" and "sys" start times by using wmic command using tool pid
    (kernelmodetime_start,usermodetime_start) = printvzstats(pid)

    ################################################################################
    # Watch (on host) for closure of tool to exit host tool execution monitor loop #
    # - stay in the loop while the pid is still listed                             #
    ################################################################################
    kernelmodetime=0
    usermodetime=0
    kernelmodetime_end = 0
    usermodetime_end = 0
    while (stdout.count(pid)) == 1:
      # Capture "user" and "sys" times, will need to save them because they
      # won't be available as soon as the tool is exited
      (kernelmodetime_loop,usermodetime_loop) = printvzstats(pid)
      if kernelmodetime_loop != 0:
        kernelmodetime_end = kernelmodetime_loop
      if usermodetime_loop != 0:
        usermodetime_end = usermodetime_loop

      args = ['tasklist', '/svc', '/NH','/FI', "imagename eq %s" % str(toolname.upper())]
      process = subprocess.Popen(
        args,
        stdin = None,
        stdout = subprocess.PIPE,
        stderr = subprocess.PIPE
      )
      (stdout, stderr) = process.communicate(None)
      if (stdout.count(pid)) == 1:
        time.sleep(1)

    ####################################################
    # Tool has finished executing, capturing end stats #
    ####################################################
    end_time = time.time() # Capture end time (real)

    # 5. Calculate time stats and write out to sessnum.err file
    if VERBOSE:
      log("Processing stats")
    sys.stderr.write("real %f\n" % ((end_time - start_time)/1.0))
    sys.stderr.write("user %f\n" % (usermodetime_end - usermodetime_start))
    sys.stderr.write("sys %f\n" % (kernelmodetime_end - kernelmodetime_start))
    status = 0 # everything went OK

    sys.stderr.flush()
    sys.stdout.flush()
    #os._exit(status)

    return 0

  def set_ipaddress(self):
    args = ["vzctl", "set", str(self.veid), "--save", "--ipdel", "all", "--ipadd", str(self.veaddr)]
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    if VERBOSE:
      log("STDERR8: %s" % stderr)
      log("STDOUT8: %s" % stdout)
      log("IP address: %s set in %d" % (self.veaddr, self.veid))
      
  def umount(self):
    # unmount container filesystems
    self.openVZ_umount(self.vz_root_path)
    self.openVZ_umount(self.vz_private_path)

  def openVZ_umount(self, fs_path):
    """If given path exists, call umount for that container
    old version:
    if os.path.exists(fs_path):
      log("%s already exists.  Ditching it." % fs_path)
      os.system("VEID=%d %s/%s" % (self.veid, self.K["VZ_CONF_PATH"], self.K["OVZ_SESSION_UMOUNT"]))
      try:
        os.rmdir(fs_path);
      except OSError:
        pass
      try:
        os.lstat(fs_path)
        log("%s still exists.  Giving up." % fs_path)
        sys.exit(1)
      except OSError:
        pass
    """
    if os.path.exists(fs_path):
      # tell openVZ to unmount that container's file system
      # internally, that umount script doesn't use abolute paths so we need
      # to set the PATH
      v_env = {"VEID" : str(self.veid), "PATH": "/bin:/usr/bin"}
      args = ["%s/%s" % (self.K["VZ_CONF_PATH"], self.K["OVZ_SESSION_UMOUNT"])]
      subprocess.check_call(args, env = v_env)
      try:
        os.rmdir(fs_path)
      except OSError:
        pass
      if os.path.exists(fs_path):
        raise MaxwellError("'%s' still exists.  Giving up." % fs_path)

  def create_xstartup(self):
    x_path = self.vncdir + "/xstartup"
    try:
      lock_stat = os.lstat(x_path)
    except OSError:
      # does not exist
      try:
#        xstartup = os.open(x_path, os.O_CREAT | os.O_WRONLY | os.O_NOFOLLOW, 0700)
        xstartup = os.open(x_path, os.O_CREAT | os.O_WRONLY, 0700)
        os.write(xstartup, "#!/bin/sh\n")
        os.close(xstartup)
        lock_stat = os.lstat(x_path)
      except OSError:
        raise MaxwellError("Unable to create '%s'." % x_path)

    # check that it has the expected permissions and ownership
    # check that we are the owner and that others can't write
    if lock_stat[stat.ST_MODE] & stat.S_IWOTH:
      raise MaxwellError("'%s' has unsafe permissions" % x_path)

    #usr_id = lock_stat[stat.ST_UID]
    #if usr_id != os.geteuid():
      #raise MaxwellError("'%s' has incorrect owner: %s" % (x_path, usr_id))

  def read_passwd(self):
    """VNC password is 8 bytes, we want the version encrypted for VNC, not the
    encoded version for web use
    """
    self.vncpass = sys.stdin.read(8)
    return

  def stunnel(self):
    """We handle tunnels and forwards here, to make the inside of containers visible to the outside.
    Containers are mapped to port ranges using the CTID (same as veid)
    REVISIT:  It is probably a mistake to use the display number instead of the CTID.  It has worked
    because in the past VZOFFSET has always been 0 which makes CTID=disp
    """
    # Setup socat mannually through Windows command line
    #START /B socat OPENSSL-LISTEN:4001,cert=c:/etc/mw-service/xvnc.pem,fork,verify=0 TCP4:10.198.1.1:5900
    # Command below is used to debug when SSL is disabled.
    args = ["c:/cygwin/bin/socat.exe", "-lf", "c:/var/log/mw-service/socat-ssl.log", "-lh", "OPENSSL-LISTEN:%d,cert=%s,fork,verify=0" % \
            #(self.K["STUNNEL_PORTS"]+self.disp, self.K["PEM_PATH"]), "TCP4:%s:%d" % (self.veaddr, self.K["WIN_PORTBASE"])]
    #START /B c:\cygwin\bin\socat OPENSSL-LISTEN:4001,cert=c:/etc/mw-service/xvnc.pem,fork,verify=0 TCP4:10.59.1.1:5900
    #args = ["c:/cygwin/bin/socat.exe", '"openssl-listen:%d" % (self.K["STUNNEL_PORTS"]+self.disp)', "-lf", "c:/var/log/mw-service/socat-ssl.log", "-lh", "OPENSSL-LISTEN:%d,cert=%s,fork,verify=0" % \
            (self.K["STUNNEL_PORTS"]+self.disp, self.K["PEM_PATH"]), "TCP4:%s:%d" % (self.veaddr, self.K["WIN_PORTBASE"])]
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    if VERBOSE:
      log("STDERR7: %s" % stderr)
      log("STDOUT7: %s" % stdout)    
      log("socat forwarder started")

  def delete_confs(self):
    """Get rid of these links if they exist."""
    for ext in ['conf', 'mount', 'umount']:
      try:
        os.unlink("%s/%d.%s" % (self.K["VZ_CONF_PATH"], self.veid, ext))
      except EnvironmentError:
        if False:
          log("File %s/%d.%s was already deleted or missing" % (self.K["VZ_CONF_PATH"], self.veid, ext))

    # vzquota "off" and "drop" are commands not available within PVS
    # stop quotas if they are already running
    # vzquota off 194
    # vzquota drop 194
    # this operation can fail if the operations are still ongoing; that's fine.
    #subprocess.call(['vzquota', 'off', '%d' % self.veid], stderr=open('c:/dev/null', 'w'))
    # drop safely removes the quota file -- this file can cause problems
    # e.g., when container template is changed
    #subprocess.call(['vzquota', 'drop', '%d' % self.veid], stderr=open('c:/dev/null', 'w'))

  def create_confs(self):
    try:
#      args=(maklink("%s/%d.conf" % (self.K["VZ_CONF_PATH"], self.veid), "%s/%s" % (self.K["VZ_CONF_PATH"], self.K["OVZ_SESSION_CONF"])))
#      subprocess.call(args)
      args= ""
      #args = (maklink("%s/%d.mount" % (self.K["VZ_CONF_PATH"], self.veid), "%s/%s" % (self.K["VZ_CONF_PATH"], self.K["OVZ_SESSION_MOUNT"])))
      #subprocess.call(args)
      #args = (maklink("%s/%d.umount" % (self.K["VZ_CONF_PATH"], self.veid), "%s/%s" % (self.K["VZ_CONF_PATH"], self.K["OVZ_SESSION_UMOUNT"])))
      #subprocess.call(args)
    except EnvironmentError:
      raise MaxwellError("Unable to create OpenVZ symlinks")

  def start_filexfer(self):
    # Start a forwarder for filexfer.  We never kill it.
    # If we can't start one, that means there's already one running.
    #port = self.disp + self.K["FILEXFER_PORTS"]
    #os.system("socat tcp4-listen:%d,fork,reuseaddr,linger=0 tcp4:%s:%d > /dev/null 2>&1 &" % (port,self.veaddr,port))
    args = ["c:/cygwin/bin/socat.exe", "-lf", "c:/var/log/mw-service/socat-tcp4.log", "-lh", "tcp4-listen:%d,reuseaddr,linger=0" % \
        (self.K["FILEXFER_PORTS"]+self.disp), "TCP4:%s:%d" % (self.veaddr, self.K["FILEXFER_PORTS"]+self.disp)]
    subprocess.check_call(args)
    self.log_status()

  def start(self, geom, user):
    ## Start querying the container's status
    start_time = time.time()
    # Send the status command to the container and monitor it's response
    args = ['vzctl', 'status', str(self.veid)]
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    if VERBOSE:
      log("STDERR3: %s" % stderr)
      log("STDOUT3: %s" % stdout)
    
    if string.strip(stdout) == ("Container %s exist unmounted down" % str(self.veid)):
      # Container is down and needs starting
      args = ["vzctl", "start", str(self.veid)]
      status = subprocess.call(args)
      if VERBOSE:
        log("STDERR4: %s" % stderr)
        log("STDOUT4: %s" % stdout)
    elif string.strip(stdout) == ("Container %s exist mounted running" % str(self.veid)):
      # Do nothing since the container is already up and running
      if VERBOSE:
        log("Start: Do nothing since the container is already running") 

    end_time = time.time()
    # log how long we waited
    log ("TIME: %f seconds" % (end_time - start_time))

    # FIXIT: Location to add VNC server configuration


  def setup_template(self):
    """Create (clone) a new container
    """
    start_time = time.time()
    # Check to see if a container already exist for this VEID
    # Send the status command to the container and monitor it's response
    args = ['vzctl', 'status', str(self.veid)]
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    if VERBOSE:
      log("STDERR1: %s" % stderr)
      log("STDOUT1: %s" % stdout)

    if string.strip(stdout) == ("Container %s exist unmounted down" % str(self.veid)):
      # Container is down and needs starting
      args = ["vzctl", "start", str(self.veid)]
      status = subprocess.check_call(args)
      if VERBOSE:
        log("%s%s" %(stdout, stderr))
    elif string.strip(stdout) == ("Container %s exist mounted running" % str(self.veid)):
      # Do nothing since the container is already up and running
      if VERBOSE:
        log("Setup: Do nothing since the container is already running")
    else: # assume there is no container with this VEID and create a new one
      #vzmlocal -sC <CT List> = <source_CTID>:<dest_CTID>
      args = ['vzmlocal', '-s', '-C', "99999:%d" % (self.veid)]
      process = subprocess.Popen(
        args,
        stdin = None,
        stdout = subprocess.PIPE,
        stderr = subprocess.PIPE
      )
      (stdout, stderr) = process.communicate(None)
      if VERBOSE:
        log("STDERR2: %s" % stderr)
        log("STDOUT2: %s" % stdout)
      
      # Wait for the container to finish being cloned
      # Check for 70 minutes between 1 second naps
      #while process.stdout != ("Command 'vzmlocal is successfully finished"):
      #  time.sleep(1)
      #  if time.time() - start_time > 70:
      #    raise MaxwellError("Timed out waiting for container to start")
        # log how long we waited
      #  if VERBOSE:
      
      end_time = time.time()
      log ("TIME: %f seconds" % (end_time - start_time))      
      
      if VERBOSE:
        log("New %d container created" % (self.veid))
    
  def delete_root(self):
    if os.path.isdir(self.vz_root_path):
      os.rmdir(self.vz_root_path)
    # not a directory if it doesn't exist

  def log_status(self):
    args = ['vzctl', 'status', str(self.veid)]
    p = subprocess.Popen(args, stderr = subprocess.PIPE, stdout = subprocess.PIPE)
    if VERBOSE:
      log(" ".join(p.communicate()))


  def halt(self):
    """ Hard halt for all processes in the container.  Does not wait for anything."""
    args = ['vzctl', 'status', str(self.veid)]
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    if VERBOSE:
      log("STDERR10: %s" % stderr)
      log("STDOUT10: %s" % stdout)

  def stop(self):
    """# Stops  and  cleans up the container
       # Old user files are saved to c:\reinstall\veid
    """
    # Disconnect users excp directory from H: by updating users c:\CTID_command\map_home_dir.cmd file
    # Filewatcher will execute command within container once file is updated
    filename = self.K["WIN_HOME_DRIVE_MAP_PATH_A"] + str(self.veid) + self.K["WIN_HOME_DRIVE_MAP_PATH_B"]
    f = open(filename, "w")
    cmd_output = "c:\windows\system32\\net.exe use * /d /y"
    f.write(cmd_output)
    f.close()    #FIXIT: Upon leaving a session - kill all possible tools (names ARE case-sensitive)

    args = ['vzctl', 'exec', str(self.veid), 'taskkill', '/F', '/IM', 'DeepSoil.exe', '/IM', 'N3DV.exe', \
            '/IM', 'SMIP-3DV.exe', '/IM', 'ZeusNL.exe', '/IM', 'epet.exe', '/IM', 'sapwood20.exe', '/IM', 'sap2000.exe', \
            '/IM', 'EQTools.exe', '/IM', 'nonlinpro.exe', '/IM', 'nonlin.exe', '/IM', 'bridgepbee.exe', \
            '/IM', 'PocketStatics.exe', '/IM', 'iexplore.exe']
    process = subprocess.Popen(
      args,
      stdin = None,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(None)
    #log("Stop Process Returncode = %s" % str(process.returncode))
    if (process.returncode != 0) and (process.returncode != 8) and (process.returncode != 128) and (process.returncode != 321):
      if VERBOSE:
        log("STOP_STDERR: %s in container %d" % (stderr,self.veid))
        log("STOP_STDOUT: %s in container %d" % (stdout,self.veid))
        #raise MaxwellError("%s tool stop loop error: '%s'" % (vzapp, stderr))
    
    

