module.exports = function(hg) {
	hg.views.filterVisible = function(item, args) {
		if (args.super) {
			return true;
		}
		if (!args.groupMap) {
			args.groupMap = {};
			args.groups.split(',').forEach(function(group) {
				args.groupMap[group] = true;
			});
		}
		if ((item.domain == 'projects' || item.domain == 'experiments') && ((item.visibility == 'USERS' && !args.uid) || (item.visibility == 'MEMBERS' && !args.groupMap[item.group_id]))) {
			return false;
		}
		if (item.domain == 'resources') {
			// logged in only
			if (item.access == 2 && !args.uid) {
				return false;
			}
			if ((item.access == 1 || item.access == 4) && !args.groupMap[item.group_ids[0]]) {
				return false;
			}
		}
		if (item.domain == 'groups') {
			if (item.discoverability == 1 && !args.groupMap[item.id]) {
				return false;
			}
		}
		if (item.domain == 'content' && item.access > 0 && !args.uid) {
			return false;
		}
		if (item.domain == 'members' && !item.public) {
			return false;
		}
		if (item.domain == 'discussions' && item.group_ids && item.group_ids[0]) {
			var gid = item.group_ids[0];
			if (!hg.groupPerms[gid] || !hg.groupPerms[gid].params.forum) {
				return false;
			}
			var perms = hg.groupPerms[gid].params.forum;
			if (perms == 'nobody' || (perms == 'registered' && !args.uid) || (perms == 'members' && !args.groupMap[gid])) {
				return false;
			}
		}

		if (item.access !== undefined && item.access > 0 && !args.uid) {
			return false;
		}

		if (item.group_ids) {
			var gid = item.group_ids.length ? item.group_ids[0] : item.group_ids;
			if (gid && gid != '') {
				if (!hg.groupPerms[gid]) {
					hg.log('warning: no group permissions set for ' + gid);
					return false;
				}
				if (hg.groupPerms[gid].privacy == 1 && !args.groupMap[gid]) {
					return false;
				}
				if (hg.groupPerms[gid].discoverability == 1) {
					return false;
				}
			}
		}
		return true;
	};
	var nonWord = new RegExp("[^-_'/\\\\\\w]+");

	var freqMap = function(str) {
		var rv = {}, total = 0;
		if (!str) {
			str = '';
		}
		str
			.toLowerCase()
			.split(nonWord)
			.filter(function(word) {
				return !hg.isStopWord(word);
			})
			.map(function(word) {
				return hg.stem(word);
			})
			.forEach(function(word) {
				if (rv[word] === undefined) {
					rv[word] = 0;
				}
				++rv[word];
				++total;
			})
			;
		return {'map': rv, 'total': total};
	};
	var textSimularity = function(a, b, dbg) {
		a = freqMap(a), b = freqMap(b);
		var common = 0, uncommon = 0;
		for (var stem in a.map) {
			if (b.map[stem]) {
				common += Math.min(a.map[stem], b.map[stem]);
			}
			else {
				uncommon += a.map[stem];
			}
		}
		// older ...
		var common = 0, total = 0;
		for (var stem in a.map) {
			if (b.map[stem]) {
				common += (a.map[stem] + b.map[stem]) / 2;
				total  += (a.map[stem] + b.map[stem]) / 2
			}
			else {
				total += a.map[stem]/2;
			}
		}
		for (var stem in b.map) {
			if (!a.map[stem]) {
				total += b.map[stem]/2;
			}
		}
		return total ? common/total : 0;
	};

	var reQuote = function(str) {
		return str.replace(new RegExp('[.\\\\+*?\\[\\^\\]$(){}=!<>|:\\/-]', 'g'), '\\$&');
	};

	var stopWordRe = new RegExp('(-|\\s+|' + hg.getStopWords().map(reQuote).join('|') + ')*');

	var baseWeight = {
		'titleWord': 1,
		'bodyWord': 0.1
	};

	var stripUtf = function(str) {
		var rv = '';
		for (var idx = 0, len = str.length; idx < len; ++idx) {
			if (str.charCodeAt(idx) <= 128) {
				rv += str[idx];
			}
		}
		return rv;
	};

	var parseTerms = function(str) {	
		var autoThresh = 500;
		var suggThresh = 25;
		var rv = {
			'positive': [],
			'negative': [],
			'mandatory': [],
			'map': {},
			'map_count': 0,
			'text': [],
			'autocorrected': {},
			'suggested': {}
		};
		str = stripUtf(hg.transliterate(str.replace(/\S{30,}/g, '')));
		if (!str) {
			return rv;
		}

		str = str.toLowerCase().replace(/^\s+|\s+$/, '');
		var len = str.length;
		var mand = false;
		var neg = false;
		var quote = false;
		var term = '';
		for (var idx = 0; idx < len; ++idx) {
			if (str[idx] === '+') {
				mand = true;
				++idx;
			}
			else if (str[idx] === '-') {
				neg = true;
				++idx;
			}
			if (str[idx] === '"') {
				quote = true;
				++idx;
			}

			for (; idx < len; ++idx) {
				if (quote) {
					if (str[idx] === '"') {
						++idx;
						break;
					}
				}
				else if (str[idx] == '-' || str[idx] === ' ' || str[idx] === '\t' || str[idx] === '\n' || str[idx] === '\r') {
					break;
				}
				term += str[idx];
			}
			while (str[idx] == '-' || str[idx] === ' ' || str[idx] === '\t' || str[idx] === '\n' || str[idx] === '\r') {
				++idx;
			}
			if (idx < len - 1) {
				--idx;
			}
			
			var key = mand ? 'mandatory' : (neg ? 'negative' : 'positive');
			var qids = [];
			var quoted = quote && /\s/.test(term);
			if (!quoted && !hg.isStopWord(term)) {
				var count = hg.getWordCount(term);
				var sugg  = hg.suggestSpelling(term, 1);
				if (sugg[0]) {
					sugg = sugg[0];
					if (false && count == 0) { // || (count < 5 && sugg.weight / count > autoThresh)) {
						rv.autocorrected[term] = sugg.word;
						term = sugg.word;
					}
					else if (sugg.weight > 5 && sugg.weight / count > suggThresh) {
						var common = hg.selectText(term).out('title-word').in('title-word');
						/// @TODO testing: does this really work better? I like it because "qdot" -> "quantum dot lab"
						if (common.items.length) {
							if (hg.stem(common.items[0].id) != hg.stem(term)) {
								rv.suggested[term] = hg.unstem(common.items[0].id);
							}
						}
						else {
							rv.suggested[term] = sugg.word;
						}
					}
				}
			}
			if (term !== 'pdf' && term !== 'ppt' && term !== 'swf' && term.length <= 40) {
				rv[key].push({'string': term, 'quoted': quoted });
			}
			if (!neg) {
				rv.text.push(term);
				if (quote) {
					term.split(/\s+/).forEach(function(t) {
						t = hg.stem(t); 
						if (hg.isStopWord(t)) {
							return;
						}

						rv.map[t] = 1;
						++rv.map_count;
					});
				}
				else {
					var t = hg.stem(term);
					if (!hg.isStopWord(t)) {
						rv.map[t] = 1;
						++rv.map_count;
					}
				}
			}
			term = '';
			pos = false;
			neg = false;
			mand = false;
			quote = false;
		}

		var addAlias = function(al) {
			if (typeof al != 'string') {
				al.forEach(function(str) {
					IaddAlias(str);
				});
			}
			else if (!rv.map[al]) {
				rv.positive.push({'string': al, 'quoted': true});
				rv.map[al] = 1;
			}
		};
		rv.positive.forEach(function(term) {
			if (hg.aliases && hg.aliases[term.string]) {
				addAlias(hg.aliases[term.string]);
			}
		});
		rv.text = rv.text.join(' ');
		return rv;
	};

	// @TODO eliminating stop words from phrases messes them up:
	// "horse to riding" -> /(horse[-\s]+riding)/gi
	var getTermRe = function(terms) {
		return new RegExp('(' + terms.map(function(term) {
			return reQuote(term.string.split(/[-\s]+/).filter(function(word) {
				return !hg.isStopWord(word);
			}).join(' ')).replace(/\s+/g, stopWordRe.toString().replace(/\//g, ''));
		})
		.filter(function(t) {
			return !hg.isStopWord(t);
		})
		.join('|')
		+ ')', 'gi');
	};

	var excerpt = function(t, termRe, isChild) {
		var ctx = {
			'before': 150,
			'after': 250,
			'total': isChild ? 200 : 350
		};
		if (t.length < 50) {
			return null;
		}

		var isTerminatingCharacter = function(ch, terms) {
			switch(ch) {
				case '.': case '!': case '?': return true;
				default: return false;
			}
		};
		var isWhitespace = function(ch) {
			switch(ch) {
				case ' ': case '\t': case '\n': case '\r': return true;
				default: return false;
			}
		};

		var pos = 0;
		var len = t.length;
		var ma;

		var lines = [];
		var total = 0;
		while ((ma = t.substr(pos, len - pos).search(termRe) != -1)) {
			var pre = '';
			pos = pos + ma;
			var begin = pos;
			while (begin > 0 && !isTerminatingCharacter(t[begin]) && pos - begin <= ctx.before) {
				--begin;
			}
			if (isTerminatingCharacter(t[begin])) {
				++begin
			}
			else if (begin > 0) {
				if (lines == []) {
					pre = '&hellip;';
				}
				while (begin < pos && !isWhitespace(t[begin])) {
					++begin;
				}
			}

			var end = pos + 1;
			while (end < len && !isTerminatingCharacter(t[end]) && end - pos <= ctx.after) {
				++end;
			}
			if (end < len && !isTerminatingCharacter(t[end])) {
				while (end > pos && !isWhitespace(t[end])) {
					--end;
				}
			}
			lines.push(pre + t.substr(begin, end - begin));

			total += end - begin;
			if (total > ctx.total) {
				break;
			}
			pos = end;
		}
		var rv = lines.join('&hellip;');
		if (rv.trim()) {
			return rv.replace('<', '&lt;').replace(termRe, '<span class="highlight">$1</span> ') + (pos == len ? '' : '&hellip;');
		}
		return t.length ? t.replace('<', '&lt;').substring(0, ctx.total) + '&hellip;' : '';
	};

	var redactMember = function(m) {
		var trans = {
			'tags': 'tag_ids',
			'org': 'organization',
			'bio': 'body'
		};
		// restrict access to explicitly disallowed fields
		for (var k in m.access) {
			if (!m.public || m.access[k] != 0) {
				m[trans[k] || k] = '';
			}
		}
		// many users do not have access defined for some fields, or for any fields. assume those are hidden
		['email', 'url', 'phone'].forEach(function(k) {
			if (!m.public || m.access[k] === undefined) {
				m[trans[k] || k] = '';
			}
		});
		return m;
	};

	var displayFormat = function(termRe, isChild) {
		return function(item) {
			if (item.domain == 'members') {
				item = redactMember(item);
			}
			if (item.body) {
				item.body = excerpt(item.body, termRe, isChild);
			}
			item.title = item.title.replace('<', '&lt;').replace(termRe, '<span class="highlight">$1</span>');
			return item;
		};
	};

	var cache = {};
	hg.views.search = function(args) {
		args.super = args.super == 'true';
		['tags', 'users', 'inGroup'].forEach(function(k) {
			args[k] = args[k] ? args[k].split(',') : [];
		});
		if (args.urlImplicitGroup) {
			args.domain = 'resources';
		}
		hg.log(args);
		var rv = {
			'js': null,
			'css': null,
			'html': null,
			'cache': (Math.random()*Math.pow(10,16)).toString(16),
			'tags': {},
			'contributors': {},
			'groups': {},
			'domains': {},
			'timeframe': {
				'year': ['prior year', 0],
				'month': ['prior month', 0],
				'week': ['prior week', 0],
				'day': ['today', 0]
			},
			'total': 0,
			'offset': (args.page - 1) * args.per,
			'page': args.page,
			'criteria': {},
			'results': hg.empty()
		};
		var terms = rv.terms = args.terms ? parseTerms(args.terms) : null, validators = [], selectors = [], termRe = /^$/;
		hg.log(terms);

		if (args.cache && cache[args.cache]) {
			var offset = rv.offset;
			rv = hg.clone(cache[args.cache]);
			rv.results = rv.results
				.slice(offset, offset + 1*args.per)
				.map(displayFormat(terms ? getTermRe(terms.positive.concat(terms.mandatory)) : termRe));
			return rv;
		}

		validators.push(function(item) {
			return hg.views.filterVisible(item, args);
		});

		validators.push(function(item) {
			/// @TODO tag-aliases
			return !!item.title && !!item.link;
		});

		if (terms && terms.negative.length) {
			var re = getTermRe(terms.negative);
			validators.push(function(item) { return !re.test(item.title) && !re.test(item.body); });
		}
		if (terms && terms.mandatory.length) {
			var re = getTermRe(terms.mandatory);
			validators.push(function(item) { return re.test(item.title) || re.test(item.body); })
		}
		if (terms && terms.positive.length) {
			var quotes = [];
			terms.positive.forEach(function(term) {
				if (term.quoted) {
					quotes.push(term);
				}
			});
			if (quotes.length) {
				var re = getTermRe(quotes);
				validators.push(function(item) { return re.test(item.title) || re.test(item.body); })
			}
		}
		if (args.tags.length) {
			var tm = {}, tmCount = 0;
			args.tags.forEach(function(tid) {
				tm[tid] = true;
				++tmCount;
			});
			validators.push(function(item) {
				var count = 0;
				if (!item.tag_ids) {
					return false;
				}
				item.tag_ids.forEach(function(tid) {
					if (tm[tid]) {
						++count;
					}
				});
				return count == tmCount;
			});
		}

		if (args.users && args.users.length) {
			var um = {}, umCount = 0;
			args.users.forEach(function(uid) {
				um[uid] = true;
				++umCount;
			});
			validators.push(function(item) {
				var count = 0;
				if (item.domain == 'members' && um[item.id]) {
					return true;
				}
				if (!item.contributor_ids) {
					return false;
				}
				item.contributor_ids.forEach(function(uid) {
					if (um[uid]) {
						++count;
					}
				});
				return count == umCount;
			});
		}
		
		if (args.inGroup && args.inGroup.length) {
			var gm = {}, gmCount = 0;
			args.inGroup.forEach(function(gid) {
				gm[gid] = true;
				++gmCount;
			});
			validators.push(function(item) {
				var count = 0;
				if (!item.group_ids) {
					return false;
				}
				item.group_ids.forEach(function(gid) {
					if (gm[gid]) {
						++count;
					}
				});
				return count == gmCount;
			});
		}
		var isValid = function(item, idx) {
			return validators.every(function(fun, fidx) { 
				return fun(item, idx);
			});
		};
		var isSelected = function(item, idx) {
			return selectors.every(function(fun) { return fun(item, idx); })
		};

		if (args.domain && args.domain != 'all') {
			selectors.push(function(item) {
				return item.domain == args.domain;
			});
		}

		var now = new Date(),
		    day = 1000 * 60 * 60 * 24,
		   week = day * 7,
		  month = day * 30,
		   year = day * 365
		;
		if (args.timeframe) {
			selectors.push(function(item) {
				if (!item.date) {
					return false;
				}
				var dt = new Date(item.date); 
				if (/^\d+$/.test(args.timeframe)) {
					return dt.getUTCFullYear() == args.timeframe;
				}

				var diff = now - dt.getTime()

				switch (args.timeframe) {
					case 'day': return diff <= day;
					case 'week': return diff <= week;
					case 'month': return diff <= month;
					case 'year': return diff <= year;
				}
				return true; // invalid date supplied by client, don't filter
			});
		}

		// select initial pool by text
		if (terms && (terms.positive.length || terms.mandatory.length)) {
			var text = terms.positive
				.map(function(t) { return t.string; })
				.concat(
					terms.mandatory
						.map(function(t) { return t.string })
				)
				.join(' ');
			
			termRe = getTermRe(terms.positive.concat(terms.mandatory));
			
			var textV = hg.selectText(text);
			// title-word matches
			rv.results.union(textV
				.out('title-word')
//					.use(['title'])
//					.adjustWeights(function(item) {
//						return textSimularity(item.title, text) + item.weight/2;
//					})
					.normalizeWeights()
					.scaleWeights(baseWeight.titleWord),
				false, // don't bother sorting yet
				true
			);
			// body-word matches
			rv.results.union(textV
				.out('body-word')
//					.use(['body'])
//					.adjustWeights(function(item) {
//						return textSimularity(item.body, text) + item.weight/4;
//					})
					.normalizeWeights()
					.scaleWeights(baseWeight.bodyWord),
				false, // don't bother sorting yet
				true
			);
			rv.results.use();
			rv.results.items = rv.results.items.filter(isValid);
			rv.results.adjustWeights(function(item) {
				return textSimularity(item.title, text) + item.weight/2;
			});
		}
		else {
			var seeded = false;
			if (args.inGroup.length) {
				seeded = true;
				rv.results = hg.select('aliases', 'group:' + args.inGroup[0]).out('alias');
			}
			else if (args.domain.length) {
				seeded = true;
				rv.results = hg.select('domains', args.domain).out('domain');
			}

			['tags', 'users'].forEach(function(al) {
				if (args[al].length) {
					args[al].forEach(function(id) {
						var items = hg.select('aliases', al.replace(/s$/, '')  + ':' + id).out('alias');

						if (seeded) {
							rv.results = rv.results.intersect(items);
						}
						else {
							rv.results = items;
							seeded = true;
						}
					});
					seeded = true;
				}
			});
			if (args.timeframe.length) {
				switch (args.timeframe) {
					case 'day': case 'week': case 'month': case 'year':
						// if we put in potentially-valid results by selecting the last two years, isValid will pick out the ones for the actually-relevant timeframe
						var items = hg.select('years', now.getFullYear()).out('timeframe')
							.union(hg.select('years', now.getFullYear() - 1).out('timeframe'));
						if (seeded) {
							rv.results = rv.results.intersect(items);
						}
						else {
							rv.results = items;
						}
					break;
					default:
						if (/\d{4}/.test(args.timeframe)) {
							var items = hg.select('years', args.timeframe).out('timeframe');
							if (seeded) {
								rv.results = rv.results.intersect(items);
							}
							else {
								rv.results = items;
								seeded = true;
							}
						}
				}
			}
			/// @TODO timeframe
			rv.results.items = rv.results.use().items.filter(isValid);
		}


		var parents = {}, needParents = {};
		rv.results.items.forEach(function(item, idx) {
			parents[item.domain + ':' + item.id] = true;
			if (item.standalone_id && item.standalone_id != item.id) {
				var domain = item.standalone_domain ? item.standalone_domain : item.domain,
				 parentQid = domain + ':' + item.standalone_id;
				if (!needParents[parentQid]) {
					needParents[parentQid] = [];
				}
				needParents[parentQid].push(item);
			}
			else {
				parents[item.domain + ':' + item.id] = idx;
			}
		});
		for (var k in needParents) {
			if (parents[k] === undefined) {
				var qid = k.split(':'),
				   item = hg.select(qid[0], qid[1]).use();
				if (item && (item = item.items[0])) {
					if (!isValid(item)) {
						continue;
					}
					item.weight = 0;
					rv.results.items.push(item);
					parents[k] = rv.results.items.length - 1;
				}
			}
			if (parents[k] !== undefined) {
				if (!rv.results.items[parents[k]].children) {
					rv.results.items[parents[k]].children = [];
				}
				needParents[k].forEach(function(item) {
					item = displayFormat(termRe)(item, true);
					rv.results.items[parents[k]].weight += item.weight / 10;
					rv.results.items[parents[k]].children.push(item);
				});
			}
		}
		rv.results.items = rv.results.items.filter(function(item) {
			return !item.standalone_id || item.standalone_id == item.id;
		});

		var maxContributions = -Infinity;
		// figure out metadata about the whole result set before we trim down to the current page
		rv.results.items.forEach(function(item) {
			if (item.domain == 'members') {
				var count = 0;
				for (var k in item.contributions) {
					count += item.contributions[k];
				}
				item.count = count;
				maxContributions = Math.max(maxContributions, count);
			}
			if (!rv.domains[item.domain]) {
				rv.domains[item.domain] = 1;
			}
			else {
				++rv.domains[item.domain];
			}
		});
		
		rv.total = rv.results.items.length;

		var anyTimes = false;
		// done with whole-set metadata, apply restrictions
		rv.results
			.filter(isSelected)
			// and figure out sorting of the things that are actually relevant
			.adjustWeights(function(item) {
				['tag', 'contributor', 'group'].forEach(function(type) {
					var key = type + '_ids';
					if (item[key]) {
						item[key].forEach(function(id) {
							if (rv[type + 's'][id]) {
								++rv[type + 's'][id][1];
							}
							else {
								rv[type + 's'][id] = [null, 1]
							}
						});
					}
				});
				/*
				var rels = {'tag': 'aliases', 'contributor': 'members', 'group': 'groups'};
				for (var type in rels) {
					var key = type + '_ids';
					if (item[key]) {
						item[key].forEach(function(id) {
							if (!rv[type + 's'][id]) {
								rv[type + 's'][id] = [hg.select(rels[type], type == 'tag' ? 'tag:' + id : id).use().items[0].title, 1];
							}
							else {
								++rv[type + 's'][id][1];
							}
						});
					}
				}
				*/

				if (item.date) {
					var dt = new Date(item.date),
					    yr = dt.getUTCFullYear(),
					  diff = now - dt.getTime()
					;

					if (!rv.timeframe[yr]) {
						anyTimes = true;
						rv.timeframe[yr] = [yr, 1];
					}
					else {
						++rv.timeframe[yr][1];
					}
					if (diff <= day) {
						++rv.timeframe['day'][1];	
					}
					else if (diff <= week) {
						++rv.timeframe['week'][1];
					}
					else if (diff <= month) {
						++rv.timeframe['month'][1];
					}
					else if (diff <= year) {
						++rv.timeframe['year'][1];
					}

				}
																																																		
				if (item.domain == 'members') {
					// bonus for being a contributor
					if (maxContributions > 0) {
						return item.weight + (item.count/maxContributions/2);
					}
					// nerf for being a non-contributor
					return item.weight/2;
				}
				return item.weight;
			});
		if (!anyTimes) {
			rv.timeframe = [];
		}
		
		// rv now contains a bunch of {id: [null, # of items in result]} entries for metadata
		// we look them now to fill them in so they have {id: ['Nice Title', # of items]} instead
		var facets = {'tag': 'tags', 'group': 'groups', 'user': 'contributors'};
		for (var k in facets) {
			// it's possible that some metadata items need to be included even though they were 
			// responsible for no result items, when the user has selected them as a facet for
			// a query that returns no results. they need to show up so that the client gives
			// the user an opportunity to de-select the facet
			var rk = k == 'user' ?'users' : facets[k];
			for (var id in args[rk]) {
				if (!rv[facets[k]][id]) {
					rv[facets[k]][id] = [null, -1];
				}
			}
			// look up titles
			for (var id in rv[facets[k]]) {
				var item = hg.select('aliases', k + ':' + id).use();
				// item is often unavailable during testing if I limit the number of entries to
				// pull in. with a full data set, however, it should always be found
				if (item.items.length) {
					rv[facets[k]][id] = [item.items[0].title, rv[facets[k]][id][1]];
				}
			}
		}
		
		rv.results = rv.results
			.sort()
			.items;
		cache[rv.cache] = hg.clone(rv);
		setTimeout(function() {
			delete cache[rv.cache];
		}, 1000 * 60 * 10);
		rv.results = rv.results 
			.slice(rv.offset, rv.offset + 1*args.per)
			.map(displayFormat(termRe));
		return rv;
	};
};
