#!/usr/bin/env python3
#
# @package      hubzero-forge
# @file         addrepoGITexternal.py
# @author       Steven M. Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2006-2018 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2006-2018 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademak of HUBzero Foundation, LLC.
#

#------------------------------------------------------------------------
# UTILITY: addrepo
#
# Creates a trac project and corresponding git repository for any
# tool development project on a HUBzero site.
#------------------------------------------------------------------------

import os
import sys
import argparse
import re
import subprocess
import glob
import stat
import shutil
import tempfile
import traceback

HUBCONFIGURATIONFILE = 'hubconfiguration.php'
INSTALLDIR           = os.path.dirname(os.path.realpath(os.path.abspath(__file__)))
GITHOMEDIRECTORY     = os.path.join(os.sep,'opt','gitExternal','tools')
TRACHOMEDIRECTORY    = os.path.join(os.sep,'opt','trac','tools')
TMPLDIR              = os.path.join(os.sep,'usr','share','hubzero-forge')

os.umask(0o022)

usage = """
USAGE: addrepo --project project [--title title --description description --password password --hubdir hubdir]

  options:  project ....... project shortname
            title ......... full name of the project
            description ... 1-line description of the project
            password ...... password needed to bind to LDAP
            hubdir ........ location of hubconfiguration.php
            gitURL ........ URL for external GIT repository

  examples:
    addrepo --project rappture --title "Rappture Toolkit" \\
            --description "Toolkit for building graphical interfaces" \\
            --password fy1k4zz \\
            --hubdir /www/myhub
"""

parser = argparse.ArgumentParser(description="Creates a trac project and corresponding git repository for any tool development project on a HUBzero site.")
parser.add_argument('--project',required=True,help="project shortname")
parser.add_argument('--title',required=True,help="full name of the project")
parser.add_argument('--description',required=True,help="1-line description of the project")
parser.add_argument('--password',required=False,help="password needed to bind to LDAP")
parser.add_argument('--hubdir',required=True,help="location of hubconfiguration.php")
parser.add_argument('--gitURL',required=True,help="URL for external GIT repository")
parser.add_argument('--publishOption',required=False,help="publishing option")

args = parser.parse_args()

#print(args)

###################### configuration scan (poor hack) ######################

def loadHubConfigurationData(configurationFileDirname,
                             configurationFileiBasename):
   hubconfigurationData = {}
   if   not os.path.isdir(configurationFileDirname):
      sys.stderr.write("ERROR: specified base directory does not exist at %s\n" % (configurationFileDirname))
      sys.exit(5)
   else:
      hubconfigurationPath = os.path.join(configurationFileDirname,configurationFileiBasename)
      if not os.path.isfile(hubconfigurationPath):
         sys.stderr.write("ERROR: specified hubconfiguration.php file does exist at %s\n" % (hubconfigurationPath))
         sys.exit(5)
      else:
         reHubconfigurationVar = re.compile("\s*(?:var|public)\s+\$(\w+)\s*=\s*\'*(.*?)\'*\s*\;\s*")
         try:
            with open(hubconfigurationPath,'r') as fpHubconfiguration:
               for line in fpHubconfiguration:
                  match = reHubconfigurationVar.match(line)
                  if match:
                     hubconfigurationData[match.group(1)] = match.group(2).strip(" \'\"\t").replace("\\'","'")
         except:
            sys.stderr.write("ERROR: specified hubconfiguration file %s could not be ingested\n" % (hubconfigurationPath))
            sys.stderr.write(traceback.format_exc())
            sys.exit(5)

   return(hubconfigurationData)


def getcfg(key):
   if key in hubconfigurationData:
      cfg = hubconfigurationData[key]
   else:
      cfg = ""

   return(cfg)

hubconfigurationData = loadHubConfigurationData(args.hubdir,HUBCONFIGURATIONFILE)

hubLDAPMasterHost   = getcfg('hubLDAPMasterHost')
hubLDAPBaseDN       = getcfg('hubLDAPBaseDN')
hubLDAPSearchUserDN = getcfg('hubLDAPSearchUserDN')
hubLDAPSearchUserPW = getcfg('hubLDAPSearchUserPW')
hubShortName        = getcfg('hubShortName')
hubLongURL          = getcfg('hubLongURL')
hubSupportEmail     = getcfg('hubSupportEmail')
forgeRepoURL        = getcfg('forgeRepoURL')
forgeName           = getcfg('forgeName')
projectURL          = os.path.join(getcfg('forgeURL'),'tools',args.project)
gitdir              = os.path.join(GITHOMEDIRECTORY,"%s" % (args.project))
tracdir             = os.path.join(TRACHOMEDIRECTORY,args.project)
if 'ldaps://' in hubLDAPMasterHost:
   hubLDAPPort = "636"
else:
   hubLDAPPort = "389"
hubLDAPMasterHost   = hubLDAPMasterHost.split('://')[1]
hubLDAPNegotiateTLS = getcfg('hubLDAPNegotiateTLS')
if hubLDAPNegotiateTLS == '1' or hubLDAPNegotiateTLS == 'yes':
   hubLDAPNegotiateTLS = 'true'
if hubLDAPNegotiateTLS == '0' or hubLDAPNegotiateTLS == 'no':
   hubLDAPNegotiateTLS = 'false'


def attempt(command):
   child = subprocess.Popen(command,
                            shell=False,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE,
                            close_fds=True,
                            universal_newlines=True)
   stdOutput,stdError = child.communicate()
   exitStatus = child.returncode
   if exitStatus != 0:
      sys.stderr.write("FAILED: %s\n" % (command))
      if stdError:
         sys.stderr.write("%s\n" % (stdError))
      if stdOutput:
         sys.stderr.write("%s\n" % (stdOutput))

   return(exitStatus,stdOutput,stdError)


def setGitFileDirectoryAttributes(gitPath,
                                  gitToolGroup):
   gitPathMode = os.lstat(gitPath).st_mode
#  add group write
#  gitPathMode |= stat.S_IWGRP
#  remove all other permissions
   gitPathMode &= ~stat.S_IRWXO
   if os.path.isdir(gitPath):
#     add set group id to directory
      gitPathMode |= stat.S_ISGID
   os.chmod(gitPath,gitPathMode)
   try:
      shutil.chown(gitPath,group=gitToolGroup)
   except:
      import grp
      gitToolGroupId = grp.getgrnam(gitToolGroup).gr_gid
      os.chown(gitPath,-1,gitToolGroupId)


def setTracFileDirectoryAttributes(tracPath,
                                   gitToolGroup):
   tracPathMode = os.lstat(tracPath).st_mode
#  add group write
   tracPathMode |= stat.S_IWGRP
   if os.path.isdir(tracPath):
#     add set group id to directory
      tracPathMode |= stat.S_ISGID
   os.chmod(tracPath,tracPathMode)
   try:
      shutil.chown(tracPath,group=gitToolGroup)
   except:
      import grp
      gitToolGroupId = grp.getgrnam(gitToolGroup).gr_gid
      os.chown(tracPath,-1,gitToolGroupId)


def getDistribution():
   distribution = "unknown"

   exitStatus,stdOutput,stdError = attempt(['lsb_release','-r','-i','-s'])

   if   re.match(".*debian.*",stdOutput,flags=re.IGNORECASE):
      distribution = "Debian"
   elif re.match(".*redhat.*",stdOutput,flags=re.IGNORECASE):
      distribution = "RedHat"
   elif re.match(".*centos.*",stdOutput,flags=re.IGNORECASE):
      distribution = "RedHat"

   return(distribution)

distribution = getDistribution()
if   distribution == "Debian":
   apacheGroup         = 'www-data'
   APACHEHOMEDIRECTORY = os.path.join(os.sep,'etc','apache2')
   apacheReloadCommand = [os.path.join(os.sep,'etc','init.d','apache2'),'reload']
elif distribution == "RedHat":
   apacheGroup         = 'apache'
   APACHEHOMEDIRECTORY = os.path.join(os.sep,'etc','httpd')
   apacheReloadCommand = [os.path.join(os.sep,'usr','sbin','apachectl'),'graceful']

#gitToolGroup = apacheGroup
gitToolGroup = 'hzgit'

###############################################################################
# GIT setup
###############################################################################

if 'HOME' in os.environ:
   del os.environ['HOME']

if not os.path.isdir(GITHOMEDIRECTORY):
   os.makedirs(GITHOMEDIRECTORY)

if not os.path.isdir(gitdir):
   os.chdir(GITHOMEDIRECTORY)
   attempt(["git","clone",args.gitURL,args.project])

   os.chdir(GITHOMEDIRECTORY)

   setGitFileDirectoryAttributes(gitdir,gitToolGroup)
   for dirPath,dirNames,fileNames in os.walk(gitdir):
      for dirName in dirNames:
         setGitFileDirectoryAttributes(os.path.join(dirPath,dirName),gitToolGroup)
      for fileName in fileNames:
         setGitFileDirectoryAttributes(os.path.join(dirPath,fileName),gitToolGroup)
else:
   sys.stdout.write("GIT repository already exists at %s... skipping git creation.\n" % (gitdir))

###############################################################################
# TRAC setup
###############################################################################

def substFile(substList,
              ifname,
              ofname):
   try:
      fpInput = open(ifname,'r')
      try:
         content = fpInput.readlines()
      except (IOError,OSError):
         self.syserr.write("%s could not be read" % (ifname))
      else:
         content = ''.join(content)

         for pattern in substList:
            if pattern[0] == '@':
               content = content.replace(pattern,substList[pattern])
            else:
               value = substList[pattern].replace('&','\\\&')
               value = value.replace('\\[0-9]','\\&')
               content = re.sub(pattern,value,content,count=1)

         try:
            fpOutput = open(ofname,'w')
            try:
               fpOutput.write(content)
            except (IOError,OSError):
               self.syserr.write("%s could not be written" % (ofname))
            finally:
               fpOutput.close()
         except (IOError,OSError):
            self.stderr.write("%s could not be opened" % (ofname))
      finally:
         fpInput.close()
   except (IOError,OSError):
      self.stderr.write("%s could not be opened" % (ifname))


def substString(substList,
                inputString):
   content = inputString
   for pattern in substList:
      if pattern[0] == '@':
         content = content.replace(pattern,substList[pattern])
      else:
         value = substList[pattern].replace('&','\\\&')
         value = value.replace('\\[0-9]','\\&')
         content = re.sub(pattern,value,content,count=1)

   return(content)
 

if not os.path.isdir(TRACHOMEDIRECTORY):
   os.makedirs(TRACHOMEDIRECTORY)

if not os.path.isdir(tracdir):
   os.chdir(TRACHOMEDIRECTORY)

   attempt(["trac-admin",tracdir,"initenv","app:%s" % (args.project),"sqlite:db/trac.db","git",os.path.join(gitdir,'.git')])

   setTracFileDirectoryAttributes(tracdir,apacheGroup)
   for dirPath,dirNames,fileNames in os.walk(tracdir):
      for dirName in dirNames:
         setTracFileDirectoryAttributes(os.path.join(dirPath,dirName),apacheGroup)
      for fileName in fileNames:
         setTracFileDirectoryAttributes(os.path.join(dirPath,fileName),apacheGroup)
#
# Copy the site.html file that adds the customized CSS
#
   shutil.copy2(os.path.join(TMPLDIR,'templates','site.html'),os.path.join(tracdir,'templates'))
#
# Tweak the page look.
#
   shutil.copy2(os.path.join(tracdir,'conf','trac.ini'),os.path.join(tracdir,'conf','trac.ini.bak'))

   pbasename = args.project
   pbasename = re.sub('^app-','',pbasename)
   pbasename = re.sub('^group-','',pbasename)

   ldapinfoSubst = {}
   ldapinfoSubst['@PROJECT@']          =  pbasename
   if args.password:
      ldapinfoSubst['@PASSWORD@']      = args.password
   else:
      ldapinfoSubst['@PASSWORD@']      = hubLDAPSearchUserPW
   ldapinfoSubst['@LDAP_HOST@']        = hubLDAPMasterHost
   ldapinfoSubst['@LDAP_BASEDN@']      = hubLDAPBaseDN
   ldapinfoSubst['@LDAP_PORT@']        = hubLDAPPort
   ldapinfoSubst['@LDAP_SEARCH_USER@'] = hubLDAPSearchUserDN
   ldapinfoSubst['@LDAP_USETLS@']      = hubLDAPNegotiateTLS

   if args.project.startswith('group-'):
      ldapConfigPath = os.path.join(TMPLDIR,'ldapconfGroup.in')
   else:
      ldapConfigPath = os.path.join(TMPLDIR,'ldapconf.in')

   try:
      fpLdapConfig = open(ldapConfigPath)
      try:
         content = fpLdapConfig.readlines()
      except (IOError,OSError):
         self.syserr.write("%s could not be read" % (ldapConfigPath))
      else:
         content = ''.join(content)
         ldapinfo = substString(ldapinfoSubst,content)
      finally:
         fpLdapConfig.close()
   except (IOError,OSError):
      self.stderr.write("%s could not be opened" % (ldapConfigPath))

   tracIniSubst = {}
   tracIniSubst["\nsrc +=[^\n]+logo.png"] = "\nsrc = /tools/images/forge.png"
   tracIniSubst["\nalt +=[^\n]+"] = "\nalt = " + forgeName
   tracIniSubst["\nlink +=[^\n]+"] = "\nlink = " + hubLongURL
   tracIniSubst["\nheight +=[^\n]+"] = "\nheight = 75"
   tracIniSubst["\nwidth +=[^\n]+"] = "\nwidth = 275"
   tracIniSubst["\nlog_level +=[^\n]+"] = "\nlog_level = ERROR"
   tracIniSubst["\nlog_type +=[^\n]+"] = "\nlog_type = syslog"
   tracIniSubst["\nmax_size +=[^\n]+"] = "\nmax_size = 10000000"
   tracIniSubst["\nmax_preview_size +=[^\n]+"] = "\nmax_preview_size = 3000000"
   tracIniSubst["\nsmtp_enabled +=[^\n]+"] = "\nsmtp_enabled = true"
   tracIniSubst["\nsmtp_replyto +=[^\n]+"] = "\nsmtp_replyto = " + hubSupportEmail
   tracIniSubst["\nsmtp_server +=[^\n]+"] = "\nsmtp_server = localhost"
   tracIniSubst["\nsmtp_from +=[^\n]+"] = "\nsmtp_from = [" + hubShortName + "] notifier"
   tracIniSubst["\npermission_store +=[^\n]+"] = "\npermission_store = HubzeroPermissionStore"
   tracIniSubst["\nurl +=[^\n]+"] = "\nurl = " + projectURL
   tracIniSubst["\nfooter +=[^\n]+"] = "\nfooter = Visit our companion site at<br /><a href=\"" + hubLongURL + "/\">" + hubLongURL + "</a>"
   tracIniSubst["\ndescr +=[^\n]+"] = "\ndescr = " + args.description
   tracIniSubst["\nicon +=[^\n]+"] = "\nicon = ncnicon.ico"
   tracIniSubst["\nmax_daysback +=[^\n]+"] = "\nmax_daysback = -1"
   tracIniSubst["\nsecure_cookies +=[^\n]+"] = "\nsecure_cookies = True"
   tracIniSubst["$"] = '\n' + ldapinfo

   substFile(tracIniSubst,os.path.join(tracdir,'conf','trac.ini.bak'),os.path.join(tracdir,'conf','trac.ini'))
#
# Copy wiki macros
#
   wikiMacroSubst = {}
   wikiMacroSubst["@PROJECT@"]     = args.project
   wikiMacroSubst["@TITLE@"]       = args.title
   wikiMacroSubst["@DESCRIPTION@"] = args.description

   substFile(wikiMacroSubst,os.path.join(TMPLDIR,'image.py.in'),os.path.join(tracdir,'plugins','image.py'))
   substFile(wikiMacroSubst,os.path.join(TMPLDIR,'link.py.in'),os.path.join(tracdir,'plugins','link.py'))
#
# Create wiki text
#
   wikiTextSubst = {}
   wikiTextSubst["@PROJECT@"]      = args.project
   wikiTextSubst["@TITLE@"]        = args.title
   wikiTextSubst["@DESCRIPTION@"]  = args.description
   wikiTextSubst["@HUBSHORTNAME@"] = hubShortName
   wikiTextSubst["@REPOURL@"]      = forgeRepoURL

   tempDirectory = tempfile.mkdtemp(prefix="addrepotrac_",dir=os.path.join(os.sep,'tmp'))

   tmpfile = os.path.join(tempDirectory,'images.txt')
   substFile(wikiTextSubst,os.path.join(TMPLDIR,'images.txt.in'),tmpfile)
   attempt(["trac-admin",args.project,"wiki","import",'Images',tmpfile])

   tmpfile = os.path.join(tempDirectory,'getstart.txt')
   substFile(wikiTextSubst,os.path.join(TMPLDIR,'getstartGITexternal.txt.in'),tmpfile)
   attempt(["trac-admin",args.project,"wiki","import",'GettingStarted',tmpfile])

   tmpfile = os.path.join(tempDirectory,'cover.txt')
   substFile(wikiTextSubst,os.path.join(TMPLDIR,'cover.txt.in'),tmpfile)
   attempt(["trac-admin",args.project,"wiki","import",'WikiStart',tmpfile])

   shutil.rmtree(tempDirectory,True)
else:
   sys.stdout.write("Trac project already exists at %s... skipping trac project creation.\n" % (tracdir))
#
# Update permissions for Apache git access
#
apacheGitConfigDir = os.path.join(APACHEHOMEDIRECTORY,hubShortName+'.conf.d','git')
gitconf    = os.path.join(apacheGitConfigDir,'gitExternal.conf')
gitconfbak = os.path.join(apacheGitConfigDir,'gitExternal.bak')

if not os.path.exists(gitconf):
   fpGitConf = open(gitconf,'a')
   fpGitConf.close()
   gitfileMode = os.lstat(gitconf).st_mode
#  add group read/write
   gitfileMode |= stat.S_IRGRP | stat.S_IWGRP
#  remove all other permissions
   gitfileMode &= ~stat.S_IRWXO
   os.chmod(gitconf,gitfileMode)

shutil.copy2(gitconf,gitconfbak)

if os.path.isdir(GITHOMEDIRECTORY):
   attempt([os.path.join(INSTALLDIR,'gengitExternalapache.py'),'--gitroot',GITHOMEDIRECTORY, \
                                                               '--giturl','/tools/@PROJECT@/git', \
                                                               '--gitconf',gitconf, \
                                                               '--projectType','tool'])

attempt(['sudo'] + apacheReloadCommand)

sys.exit(0)

