# ----------------------------------------------------------------------
#  FILEXFER SERVER - support for file transfer with user's desktop
#
#  This is the server that manages filexfer operations for the
#  "exportfile" and "importfile" client commands.  Clients communicate
#  with the server via a socket and send along a series of file
#  transfer requests.  The server then dispatches "open url" requests
#  to one or more viewers in the user's web browser.
# ======================================================================
#  AUTHOR:  Michael McLennan, Purdue University
#  Copyright (c) 2004-2007  Purdue Research Foundation
#
#  See the file "license.terms" for information on usage and
#  redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# ======================================================================
package require RapptureGUI

# load util procedures from this path
set installdir [file dirname [info script]]
if {"." == $installdir} {
    set installdir [pwd]
}
lappend auto_path $installdir

# Fork here and run the rest as a background daemon so the parent who
# launches it the first time doesn't have to wait for this to finish.
Rappture::daemon

namespace eval filexfer {
    variable fxport 0           ;# server is running on this port
    variable fxcookie ""        ;# magic cookie used to auth clients
    variable buffer             ;# request buffer for each client
    variable address            ;# incoming address for each client
    variable downloads          ;# array of all known download tokens
    variable downloadInfo       ;# maps download token => file, timeout, etc.
    variable uploads            ;# maps upload token => interested client
    variable countdown ""       ;# countdown to auto-shutdown
    variable log ""             ;# handle to log file for debug messages

    #
    # Translates mime type => file extension
    #        and file extension => mime type
    #
    # Used primarily for spooling data files.
    #
    variable mime2ext
    variable ext2mime
    variable mime2type

    foreach {mtype ext type} {
        application/octet-stream  .jar    binary
        application/octet-stream  .class  binary
        application/pdf           .pdf    binary
        application/postscript    .ps     ascii
        application/x-matlab      .mat    binary
        chemical/x-pdb            .pdb    ascii
        image/gif                 .gif    binary
        image/jpeg                .jpg    binary
        image/png                 .png    binary
        text/html                 .htm    ascii
        text/html                 .html   ascii
        text/plain                .txt    ascii
        video/mpeg                .mpeg   binary
    } {
        set mime2ext($mtype) $ext
        set ext2mime($ext) $mtype
        set mime2type($mtype) $type
    }

    # ------------------------------------------------------------
    # DEPRECATED -- used only for old Rappture filexfer clients
    # ------------------------------------------------------------
    # magic cookie for old applet auth

    # ------------------------------------------------------------
    # DEPRECATED -- used only for old Rappture filexfer clients
    # ------------------------------------------------------------
    # used to generate cookies -- see bakeCookie for details
    variable cookieChars {
        a b c d e f g h i j k l m n o p q r s t u v w x y z
        A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
        0 1 2 3 4 5 6 7 8 9
    }

    # ------------------------------------------------------------
    # DEPRECATED -- used only for old Rappture filexfer clients
    # ------------------------------------------------------------
    # maps client socket => active/inactive status
    variable clients

    # maps client socket => socket protocol
    # if it doesn't match, we warn user to restart the browser
    variable protocol
    set protocol(current) "1.0"
}

# ----------------------------------------------------------------------
# USAGE: filexfer::init <port> <cookie>
#
# Called in the main application to start listening to a particular
# port and start acting like a filexfer server.
# ----------------------------------------------------------------------
proc filexfer::init {port cookie} {
    variable fxport
    variable fxcookie
    variable clients

    # keep a list of most recently activated clients
    set clients(order) ""

    #
    # The port setting should have been set properly in the
    # "resources" file loaded at the beginning of the app.
    # If it wasn't, then don't do any filexfer.
    #
    if {$port > 0} {
        #
        # If the prescribed port is busy, then exit with a special
        # status code so the middleware knows to try again with another
        # port.
        #
        # OH NO! THE DREADED ERROR CODE 9!
        #
        if {[catch {socket -server filexfer::accept $port}]} {
            exit 9
        }
        set fxport $port
        set fxcookie $cookie

        #
        # Clean up all files marked for deletion when this program
        # shuts down.  If we're running on nanoHUB, we'll get a
        # SIGHUP signal when it's time to quit.  We should also
        # catch SIGKILL in case this gets killed dead.
        #
        Rappture::signal SIGHUP filexfer "filexfer::cleanup; exit"
        Rappture::signal SIGKILL filexfer "filexfer::cleanup; exit"

        #
        # Kick off the housekeeping option that will check all
        # timeouts and clean up files that are no longer needed.
        #
        filexfer::housekeeping
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::trigger <token>
#
# Used internally to trigger the download of the file associated
# with <token>.  Sends a message to all clients connected to this
# server telling them to fetch the URL for the file.
# ----------------------------------------------------------------------
proc filexfer::trigger {token} {
    variable resources
    variable fxcookie
    variable downloadInfo

    set tail [file tail $downloadInfo($token-file)]
    regsub -all {\?\&\;} $tail "" tail  ;# get rid of confusing chars

    variable clients
    variable protocol
    if {[llength $clients(order)] > 0} {
        # ------------------------------------------------------------
        # DEPRECATED -- used only for old Rappture filexfer clients
        # ------------------------------------------------------------
        set sent 0
        set protoproblems 0
        foreach cid $clients(order) {
            if {[info exists clients($cid)] && $clients($cid)} {
                if {![string equal $protocol($cid) $protocol(current)]} {
                    incr protoproblems
                }
                catch {
                    puts $cid [format "url /download/%s?token=%s" \
                      $tail $token]
                }
                set sent 1
            }
        }
        if {!$sent} {
            filexfer::log "couldn't send to deprecated filexfer: no clients"
        }
        if {$protoproblems >= 1} {
            filexfer::log "couldn't send to deprecated filexfer: old client"
        }
    } else {
        set path [format "%s/filexfer/%s/download/%s?token=%s" \
            $resources(huburl) $fxcookie $tail $token]
        if {[catch {exec /usr/lib/mw/bin/clientaction url $path} result]} {
            filexfer::log "clientaction url $path"
            filexfer::log "failed: $result"
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::cleanup ?<token> <token> ...?
#
# Used to clean up one or more tokens, which represent files being
# kept for download.  Forgets all information associated with each
# token.  If the file was marked for deletion, then it is deleted
# at this time.
#
# Note that if this procedure is called with no tokens, then it cleans
# up all tokens.  This is useful, for example, when the server is
# being killed by a SIGHUP.
# ----------------------------------------------------------------------
proc filexfer::cleanup {args} {
    variable downloads
    variable downloadInfo

    # no specific tokens?  then clean up all tokens
    if {[llength $args] == 0} {
        set args [array names downloads]
    }

    # run through all tokens and clean them up
    foreach t $args {
        if {$downloadInfo($t-delete)} {
            catch {file delete -force $downloadInfo($t-file)}
        }
        foreach tag [array names downloadInfo $t-*] {
            unset downloadInfo($tag)
        }
        unset downloads($t)
        filexfer::log "cleaned up $t"
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::accept <clientId> <address> <port>
#
# Invoked automatically whenever a client tries to connect to this
# server.  Sets up callbacks to handle further communication.
# ----------------------------------------------------------------------
proc filexfer::accept {cid addr port} {
    fileevent $cid readable [list filexfer::handler $cid]
    #
    # Use binary mode for both input and output, so the
    # byte counts (as in Content-Length:) are correct.
    #
    fconfigure $cid -buffering line -translation binary

    # we accept clients from any where, but only let them
    # speak FILEXFER protocol when they come from localhost
    variable address
    set address($cid) $addr
    filexfer::log "accepted client $cid on $addr"
}

# ----------------------------------------------------------------------
# USAGE: filexfer::handler <clientId>
#
# Invoked automatically whenever a message comes in from a client
# to handle the message.
# ----------------------------------------------------------------------
proc filexfer::handler {cid} {
    variable buffer

    if {[gets $cid line] < 0} {
        # eof from client -- clean up
        shutdown $cid
    } else {
        # clip out trailing carriage returns
        regsub -all {\r$} $line "" line

        #
        # Is the first line of the request?  Then make sure
        # that it's properly formed.
        #
        if {![info exists buffer($cid)]
               && [regexp {^ *[A-Z]+ +[^ ]+ +HTTP/1\.[01]$} $line]} {
            set buffer($cid) $line
            return   ;# wait for more lines to dribble in...
        } elseif {[info exists buffer($cid)]} {
            set line [string trim $line]
            if {"" == $line} {
                regexp {^ *([A-Z]+) +} $buffer($cid) match type
                if {$type == "POST"} {
                    if {[regexp {Content-Length: *([0-9]+)} $buffer($cid) match len]} {
                        set buffer($cid-post) [read $cid $len]
                    }
                    # finished post... process below...
                } else {
                    # finished get or other op... process below...
                }
            } else {
                append buffer($cid) "\n" $line
                return
            }
            # blank line -- process below...
        } elseif {[regexp { +(RAPPTURE|FILEXFER)(/[0-9\.]+)?$} $line]} {
            set buffer($cid) $line
            # special Filexfer request -- process below...
        } else {
            response $cid error -message "Your browser sent a request that this server could not understand.<P>Malformed request: $line"
            shutdown $cid
            return
        }

        #
        # We've seen a blank line at the end of a request.
        # Time to process it...
        #
        set errmsg ""
        set lines [split $buffer($cid) \n]
        unset buffer($cid)
        set headers(Connection) close

        # extract the TYPE and URL from the request line
        set line [lindex $lines 0]
        set lines [lrange $lines 1 end]
        filexfer::log "REQUEST: $line"

        if {![regexp {^ *([A-Z]+) +([^ ]+) +(HTTP/1\.[01])$} $line \
              match type url proto]
            && ![regexp { +((RAPPTURE|FILEXFER)(/[0-9\.]+)?)$} $line match proto]} {
            set errmsg "Malformed request: $line"
        }

        if {[string match HTTP/* $proto]} {
            #
            # HANDLE HTTP/1.x REQUESTS...
            #
            while {"" == $errmsg && [llength $lines] > 0} {
                # extract the "Header: value" lines
                set line [lindex $lines 0]
                set lines [lrange $lines 1 end]

                if {[regexp {^ *([-a-zA-Z0-9_]+): *(.*)} $line \
                      match key val]} {
                    set headers($key) $val
                } else {
                    set errmsg [format "Request header field is missing colon separator.<P>\n<PRE>\n%s</PRE>" $line]
                }
            }

            if {"" != $errmsg} {
                # errors in the header
                response $cid header -status "400 Bad Request" \
                    -connection $headers(Connection)
                response $cid error -message "Your browser sent a request that this server could not understand.<P>$errmsg"
                flush $cid
            } else {
                # process the request...
                switch -- $type {
                    GET {
                        request_GET $cid $url headers
                    }
                    POST {
                        set postdata ""
                        if {[info exists buffer($cid-post)]} {
                            set postdata $buffer($cid-post)
                            unset buffer($cid-post)
                        }
                        request_POST $cid $url headers $postdata
                    }
                    default {
                        response $cid header \
                            -status "400 Bad Request" \
                            -connection $headers(Connection)
                        response $cid error -message "Your browser sent a request that this server could not understand.<P>Invalid request type <b>$type</b>"
                        flush $cid
                    }
                }
            }
            if {$headers(Connection) == "close"} {
                shutdown $cid
            }
        } elseif {[string match FILEXFER* $proto]} {
            #
            # HANDLE REQUESTS FROM exportfile/importfile CLIENTS...
            #
            if {[regexp {^ *EXPORT +(.+) ([^ ]+) +FILEXFER(/[0-9\.]+)?$} \
                  $line match args token vers]} {
                request_EXPORT $cid $args $token $vers
            } elseif {[regexp {^ *IMPORT +(.+) ([^ ]+) +FILEXFER(/[0-9\.]+)?$} \
                  $line match template token vers]} {
                request_IMPORT $cid $template $token
            } elseif {[regexp {^ *BYE +([^ ]+) +FILEXFER(/[0-9\.]+)?$} \
                  $line match token vers]} {
                request_BYE $cid $token
            }
        } elseif {[string match RAPPTURE* $proto]} {
            #
            # HANDLE SPECIAL RAPPTURE REQUESTS...
            # ----------------------------------------------------------
            # DEPRECATED -- used only for old Rappture filexfer clients
            # ----------------------------------------------------------
            #
            set vers "0.0"
            if {[regexp {^ *(REGISTER) +([^ ]+) +([^ ]+) +([^ ]+) +RAPPTURE(/[0-9\.]+)?$} \
                  $line match type user addr cookie vers]} {
                  set vers [string trimleft $vers /]
                request_REGISTER $cid $user $addr $cookie $vers
            } elseif {[regexp {^ *UNREGISTER +RAPPTURE$} $line]} {
                request_UNREGISTER $cid
            } elseif {[regexp {^ *ACTIVATE +RAPPTURE$} $line]} {
                request_ACTIVATE $cid
            } elseif {[regexp {^ *DEACTIVATE +RAPPTURE$} $line]} {
                request_DEACTIVATE $cid
            } else {
                response $cid header \
                    -status "400 Bad Request" \
                    -connection $headers(Connection)
                response $cid error -message "Your browser sent a request that this server could not understand.<P>Invalid request type <b>$type</b>"
                flush $cid
            }
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::request_GET <clientId> <url> <headerVar>
#
# Used internally to handle GET requests on this server.  Looks for
# the requested <url> and sends it back to <clientId> according to
# the headers in the <headerVar> array in the calling scope.
# ----------------------------------------------------------------------
proc filexfer::request_GET {cid url headerVar} {
    global env
    variable downloads
    variable downloadInfo
    upvar $headerVar headers

    # we're busy -- no auto-shutdown
    countdown cancel

    #
    # Look for any ?foo=1&bar=2 data embedded in the URL...
    #
    if {[regexp -indices {\?[a-zA-Z0-9_]+\=} $url match]} {
        foreach {s0 s1} $match break
        set args [string range $url [expr {$s0+1}] end]
        set url [string range $url 0 [expr {$s0-1}]]

        foreach part [split $args &] {
            if {[llength [split $part =]] == 2} {
                foreach {key val} [split $part =] break
                set post($key) [urlDecode $val]
            }
        }
    }

    #
    # Interpret the URL and fulfill the request...
    #
    if {$url == "/debug" && [info exists env(FILEXFER_DEBUG)]} {
        variable fxport
        variable fxcookie
        #
        # ----------------------------------------------------------
        # DEPRECATED -- used only for old Rappture filexfer clients
        # ----------------------------------------------------------
        # DEBUG MODE:  Put out a web page containing the applet
        #   and parameters needed to drive this.  Allow only
        #   if the FILEXFER_DEBUG environment variable is set.
        #
        response $cid header \
            -status "200 OK" \
            -connection $headers(Connection)
        set s [clock seconds]
        set date [clock format $s -format {%a, %d %b %Y %H:%M:%S %Z}]
        puts $cid "Last-Modified: $date"

        set user "???"
        foreach var {USER USERNAME LOGNAME} {
            if {[info exists env($var)]} {
                set user $env($var)
                break
            }
        }

        response $cid body -type text/html -string [format {<html>
<head><title>filexfer Debug Page</title></head>
<body BGCOLOR=White>
This page contains the same Java applet that the nanoHUB includes
on each Rappture tool page.  The applet connects back to the
Rappture application and listens for file transfer requests
coming from the user.
<p>
<applet CODE="filexfer.class" ARCHIVE="filexfer.jar" width=300 height=200>
<param name="port" value="%s">
<param name="user" value="%s">
<param name="cookie" value="%s">
</applet>
</body>
</html>
} $fxport $user $fxcookie]
        flush $cid
    } elseif {[regexp {^(/filexfer/[0-9a-fA-F]+)?/?download/} $url]} {
        # NOTE: ^^^ /filexfer/xxxxxx part comes through only when using
        #       the deprecated filexfer clients, who post directly to
        #       this client without appropriate firewall transit.
        #
        # Send back an exported file...
        #
        if {![info exists post(token)]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  Missing file token."
        } elseif {![info exists downloads($post(token))]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  File token not recognized.  Perhaps the download took too long and the file has been forgotten.  Try your download again."
        } else {
            set t $post(token)

            #
            # If we have a special message for the top of the page,
            # send along a wrapper with internal frames.
            #
            if {[llength $downloadInfo($t-dlpage)] > 0} {
                variable fxcookie
                set ftail [file tail $downloadInfo($t-file)]
                set wrapper [format {<html><frameset id="container" rows="400,100%%"><frame src="/filexfer/%s/dltop/?token=%s"><frame name="file" src="/filexfer/%s/dlfile/%s?token=%s"></frameset><noframes>Oops!  Your browser doesn't handle frames.  <a href="/filexfer/%s/dlfile/%s?token=%s">Click here</a> to download your file.</noframes></html>} $fxcookie $t $fxcookie $ftail $t $fxcookie $ftail $t]
                response $cid header -status "200 OK"
                response $cid body \
                    -string $wrapper -type text/html

            } elseif {$downloadInfo($t-format) == "html"} {
                #
                # If the format is "html", then treat the body as
                # HTML data.  Be careful to rewrite any embedded
                # paths, so that images can be downloaded and links
                # can be traversed with respect to the original file.
                #

            } else {
                #
                # Otherwise, send the raw file itself.
                #
                response $cid file \
                    -path $downloadInfo($t-file) \
                    -connection $headers(Connection)
            }
        }
        flush $cid
    } elseif {[regexp {^(/filexfer/[0-9a-fA-F]+)?/?dlfile/} $url]} {
        # NOTE: ^^^ /filexfer/xxxxxx part comes through only when using
        #       the deprecated filexfer clients, who post directly to
        #       this client without appropriate firewall transit.
        #
        # Send back an exported file...
        #
        if {![info exists post(token)]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  Missing file token."
        } elseif {![info exists downloads($post(token))]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  File token not recognized.  Perhaps the download took too long and the file has been forgotten.  Try your download again."
        } else {
            set t $post(token)
            if {[info exists post(saveas)]} {
                set saveas $post(saveas)
            } else {
                set saveas 0
            }

            if {$downloadInfo($t-format) == "html"
                  && [regexp {^(/filexfer/[0-9a-fA-F]+)?/?dlfile/(.+)} $url match dummy tail]} {
                set dir [file dirname $downloadInfo($t-file)]
                foreach {mime type} [filexfer::mimetype [file join $dir $tail]] break
filexfer::log "loading file: $tail ($mime $type)"
                if {$mime == "text/html"} {
                    if {[catch {filexfer::loadHTML $t $dir $tail} result]} {
                        filexfer::log "error loading HTML: $result"
                        response $cid header \
                            -status "500 Internal Server Error" \
                            -connection $params(-connection)
                        response $cid error -status "500 Internal Server Error" -message "error while processing HTML file: $result"
                    } else {
                        response $cid header -status "200 OK"
                        response $cid body \
                            -string $result -type text/html
                    }
                } else {
                    response $cid file \
                        -path [file join $dir $tail] \
                        -connection $headers(Connection)
                }
                flush $cid
            } else {
                response $cid file \
                    -path $downloadInfo($t-file) \
                    -connection $headers(Connection) \
                    -saveas $saveas
            }
        }
        flush $cid
    } elseif {[regexp {^(/filexfer/[0-9a-fA-F]+)?/?dltop/} $url]} {
        # NOTE: ^^^ /filexfer/xxxxxx part comes through only when using
        #       the deprecated filexfer clients, who post directly to
        #       this client without appropriate firewall transit.
        #
        # Send back the heading for the download page...
        #
        if {![info exists post(token)]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  Missing file token."
        } elseif {![info exists downloads($post(token))]} {
            response $cid header -status "401 Unauthorized"
            response $cid error -status "401 Unauthorized" -message "You do not have the proper credentials to access file $post(token).  File token not recognized.  Perhaps the download took too long and the file has been forgotten.  Try your download again."
        } else {
            set t $post(token)

            set cmds {
                global installdir
                set fid [open [file join $installdir download.html] r]
                set html [read $fid]
                close $fid

                array set replacements $downloadInfo($t-dlpage)
                variable fxcookie
                set replacements(@PREFIX@) /filexfer/$fxcookie
                set replacements(@TOKEN@) $t
                set replacements(@FILE@) [file tail $downloadInfo($t-file)]
                if {![info exists replacements(@MESSAGE@)]} {
                    set replacements(@MESSAGE@) ""
                } elseif {[string length $replacements(@MESSAGE@)] > 0} {
                    set replacements(@MESSAGE@) "<p id=\"description\">$replacements(@MESSAGE@)</p>"
                }

                # If the file is binary, then add a warning to the
                # user about the blank area they'll see where the
                # document normally sits.
                set replacements(@WARNING@) ""
                foreach {mime type} [filexfer::mimetype $downloadInfo($t-file)] break
                if {$type == "binary" && ![string match image/* $mime]} {
                    set replacements(@WARNING@) "<p id=\"warning\">This type of file may not appear in the preview area below.  Your browser may decide to save it directly to your desktop.</p>"
                }

                # replace the embedded message first, in case it has
                # other @STRING@ strings embedded within it.
                set html [string map \
                    [list @MESSAGE@ $replacements(@MESSAGE@)] $html]

                set html [string map [array get replacements] $html]
            }
            if {[catch $cmds result]} {
                filexfer::log "can't create download wrapper: $result"
                response $cid header \
                    -status "500 Internal Server Error" \
                    -connection $params(-connection)
                response $cid error -status "500 Internal Server Error" -message "can't create download heading: $result"
            } else {
                response $cid header -status "200 OK" \
                    -connection $headers(Connection)
                response $cid body -string $html -type text/html
            }
        }
        flush $cid
    } elseif {[regexp {^/?[a-zA-Z0-9_]+\.[a-zA-Z]+$} $url match]} {
        #
        # Send back an applet file...
        # ----------------------------------------------------------
        # DEPRECATED -- used only for old Rappture filexfer clients
        # ----------------------------------------------------------
        #
        set url [string trimleft $url /]
        set file [file join $RapptureGUI::library filexfer $url]
        response $cid file -path $file -connection $headers(Connection)
        flush $cid
    } else {
        #
        # BAD FILE REQUEST:
        #   The user is trying to ask for a file outside of
        #   the normal filexfer installation.  Treat it the
        #   same as file not found.
        response $cid header \
            -status "404 Not Found" \
            -connection $headers(Connection)
        response $cid error -status "404 Not Found" -message "The requested URL $url was not found on this server."
        flush $cid
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::request_POST <clientId> <url> \
#          <headerVar> <postdata>
#
# Used internally to handle POST requests on this server.  Looks for
# the requested <url> and sends it back to <clientId> according to
# the headers in the <headerVar> array in the calling scope.
# ----------------------------------------------------------------------
proc filexfer::request_POST {cid url headerVar postData} {
    global env
    variable uploads
    upvar $headerVar headers

    # we're busy -- no auto-shutdown
    countdown cancel

    #
    # Look for any ?foo=1&bar=2 data embedded in the URL...
    #
    if {[regexp -indices {\?[a-zA-Z0-9_]+\=} $url match]} {
        foreach {s0 s1} $match break
        set args [string range $url [expr {$s0+1}] end]
        set url [string range $url 0 [expr {$s0-1}]]

        foreach part [split $args &] {
            if {[llength [split $part =]] == 2} {
                foreach {key val} [split $part =] break
                set post($key) [urlDecode $val]
            }
        }
    } elseif {[string length $postData] > 0} {
        #
        # If we have explicit POST data, then it is one of two
        # kinds.  It is either key=value&key=value&... or a
        # multipart key/value assignment with -------boundary
        # separators.
        #
        set part "single"
        if {[info exists headers(Content-Type)]} {
            set data $headers(Content-Type)
            regsub -all { *; *} $data "\n" data
            set type [lindex [split $data \n] 0]
            if {$type == "multipart/form-data"} {
                set part "multi"
                foreach assmt [lrange [split $data \n] 1 end] {
                    foreach {key val} [split $assmt =] break
                    if {$key == "boundary"} {
                        set boundary [string trimleft $val -]
                    }
                }
            }
        }

        switch -- $part {
            single {
                # simple key=value&key=value&... case
                foreach assmt [split $postData &] {
                    if {[regexp {([^=]+)=(.*)} $assmt match key val]} {
                        set post($key) [urlDecode $val]
                    }
                }
            }
            multi {
                #
                # Multipart data:
                #  ----------------------------406765868666254505654602083
                #  Content-Disposition: form-data; name="key"
                #
                #  value
                #  ----------------------------406765868666254505654602083
                #  ...
                #
                while {[string length $postData] > 0} {
                    if {![regexp -indices "(^|\r?\n)-+$boundary\r?\n" \
                           $postData match]} {
                        break  ;# no more boundaries -- done
                    }
                    foreach {b0 b1} $match break
                    set section [string range $postData [expr {$b1+1}] end]
                    set postData $section

                    if {![regexp -indices "(^|\r?\n)-+${boundary}(\r?\n|--)" \
                           $section match]} {
                        break  ;# can't find bottom boundary -- done
                    }
                    foreach {b0 b1} $match break
                    set section [string range $postData 0 [expr {$b0-1}]]
                    set postData [string range $postData $b0 end]

                    if {![regexp -indices "\r?\n\r?\n" $section match]} {
                        continue  ;# can't find header -- skip this
                    }
                    foreach {h0 h1} $match break

                    set header [string range $section 0 [expr {$h0-1}]]
                    regsub -all {\r\n} $header \n header
                    foreach line [split $header \n] {
                        if {[regexp {Content-Disposition:} $line]} {
                          regsub -all { *; *} $line "\n" line
                          foreach assmt [lrange [split $line \n] 1 end] {
                            foreach {key val} [split $assmt =] break
                            set element($key) [string trim $val \"]
                          }
                        }
                    }

                    set data [string range $section [expr {$h1+1}] end]
                    if {[info exists element(name)]} {
                        set post($element(name)) $data
                    }
                    if {[info exists element(filename)]} {
                        if {[string match dest* $element(name)]} {
                            if {[string match "*@USE_REMOTE@*" $post($element(name))]} {
			        regsub -all { } [lindex [split [file tail $element(filename)] "\\"] end] {_} localName
                                regsub -all {@USE_REMOTE@} $post($element(name)) $localName saveAsFile
                                set post($element(name)) $saveAsFile
                            }
                        }
                    }
                }
            }
            default {
                filexfer::log "unknown content type \"$part\": should be single or multi"
            }
        }
    }

    #
    # Interpret the URL and fulfill the request...
    #
    if {[regexp {^(/filexfer/[0-9a-fA-F]+)?/upload$} $url]} {
        if {[info exists post(client)]
              && [info exists uploads($post(client))]} {

            set client $uploads($post(client))
            set saved ""

            set i 1
            while {[info exists post(dest$i)]} {
                set file $post(dest$i)
                set which $post(which$i)
                if {[info exists post($which)]} {
                    set data $post($which)

                    if {![isbinary $data]} {
                        # not a binary file? then trim extra spaces
                        set data [string trim $data]
                    }
                    if {[string length $data] > 0} {
                        if {[catch {
                            set fid [open $file w]
                            if {[isbinary $data]} {
                                fconfigure $fid -translation binary \
                                    -encoding binary
                            }
                            puts -nonewline $fid $data
                            close $fid
                        } result]} {
                            catch {
                              puts $client [list ERROR "problem saving data to $file: $result"]
                              flush $client
                            }
                        } else {
                            lappend saved $file
                        }
                    }
                    incr i
                }
            }
            catch {
                puts $client "IMPORTED $saved"
                flush $client
            }
            shutdown $client

            #
            # Send back a response that closes the window that
            # posted this form.
            #
            response $cid header -status "200 OK" \
                -connection $headers(Connection)
            set s [clock seconds]
            set date [clock format $s -format {%a, %d %b %Y %H:%M:%S %Z}]
            catch { puts $cid "Last-Modified: $date" }
            response $cid body -type text/html -string {<html>
<head>
  <title>Upload Complete</title>
  <script language="JavaScript">
    function setup() {
        window.close()
    }
    window.onload = setup;
  </script>
</head>
<body>
<b>Data uploaded successfully.  This window will now close.</b><br/>
If this window doesn't close automatically, feel free to close it manually.
</body>
</html>}
            flush $cid
        } else {
            #
            # No client ID within the form.  What happened?
            #
            response $cid header \
                -status "403 Forbidden" \
                -connection $headers(Connection)
            if {![info exists post(client)]} {
                response $cid error -status "403 Forbidden" -message "This form was missing the \"client\" element which authorizes the transaction and directs the posted data."
            } else {
                response $cid error -status "403 Forbidden" -message "The command-line program that initiated this action seems to have exited.  Try your upload operation again."
            }
            flush $cid
        }
    } else {
        #
        # BAD FILE REQUEST:
        #   The user is trying to ask for a file outside of
        #   the normal filexfer installation.  Treat it the
        #   same as file not found.
        response $cid header \
            -status "404 Not Found" \
            -connection $headers(Connection)
        response $cid error -status "404 Not Found" -message "The requested URL $url was not found on this server."
        flush $cid
    }
}

# ----------------------------------------------------------------------
# USAGE: request_EXPORT <clientId> <arglist> <cookie> <protocol>
#
# Used internally to handle EXPORT requests on this server.  A client
# sends EXPORT requests when it wants to send a file out to other
# receiving clients.  The <args> are a proper Tcl list with:
#
#   FILEXFER/1.0
#     <filename> <timeoutSecs> <delete> <subs>
#
#   FILEXFER/1.1
#     <filename> <timeoutSecs> <delete> raw|html <subs>
#
# The <filename> is immediately sent out to listening clients via
# "clientaction url".  After <timeoutSecs>, the file is forgotten,
# and if the <delete> flag is set, it is deleted at that time.
#
# If the format is "html", then the server automatically rewrites
# embedded file paths for images and links so that those items can
# be downloaded with respect to the original file.
# ----------------------------------------------------------------------
proc filexfer::request_EXPORT {cid arglist cookie proto} {
    variable fxcookie
    variable address

    # clients must come from localhost and share a secret
    if {![string equal $address($cid) "127.0.0.1"]} {
        puts $cid [list ERROR "unauthorized access -- must come from localhost"]
        flush $cid
        shutdown $cid
        return
    }
    if {![string equal $cookie $fxcookie]} {
        puts $cid [list ERROR "unauthorized access"]
        flush $cid
        shutdown $cid
        return
    }

    if {$proto == "FILEXFER/1.0"} {
        if {[llength $arglist] < 3 || [llength $arglist] > 4} {
            puts $cid [list ERROR "wrong # args for EXPORT: expected \"file timeout delete ?subs?\""]
            flush $cid
            return
        }
        if {[llength $arglist] == 3} {
            lappend arglist "raw"
        } else {
            set arglist [linsert $arglist 3 "raw"]
        }
    } else {
        if {[llength $arglist] < 4 || [llength $arglist] > 5} {
            puts $cid [list ERROR "wrong # args for EXPORT: expected \"file timeout delete format ?subs?\""]
            flush $cid
            return
        }
    }

    set file [lindex $arglist 0]
    if {![file readable $file]} {
        puts $cid [list ERROR "EXPORT file not readable: $file"]
        flush $cid
        return
    }

    set timeout [lindex $arglist 1]
    if {![string is integer $timeout]} {
        puts $cid [list ERROR "bad EXPORT timeout \"$timeout\": should be integer"]
        flush $cid
        return
    }

    set del [lindex $arglist 2]
    if {![string is boolean $del]} {
        puts $cid [list ERROR "bad EXPORT delete flag \"$del\": should be boolean"]
        flush $cid
        return
    }

    set fmt [lindex $arglist 3]
    if {[lsearch {raw html} $fmt] < 0} {
        puts $cid [list ERROR "bad EXPORT format flag \"$fmt\": should be raw, html"]
        flush $cid
        return
    }

    # If an optional substitution list is specified, then substitute
    # these parameters into the download.html file with framesets and
    # display it above the download.
    set subs [lindex $arglist 4]

    # Create a unique token for this export file.
    # Make sure that it's not already being used.
    variable downloads
    variable downloadInfo
    while {1} {
        set t [bakeCookie]
        if {![info exists downloads($t)]} {
            break
        }
    }
    set downloads($t) 1
    set downloadInfo($t-file) $file
    set downloadInfo($t-time) [clock seconds]
    set downloadInfo($t-timeout) $timeout
    set downloadInfo($t-delete) $del
    set downloadInfo($t-format) $fmt
    set downloadInfo($t-dlpage) $subs

    countdown cancel

    #
    # Tell the clients to download this file
    #
    if {[catch {trigger $t} result]} {
        puts $cid [list ERROR "EXPORT error: $result"]
        flush $cid
    }
}

# ----------------------------------------------------------------------
# USAGE: request_IMPORT <clientId> <template> <cookie>
#
# Used internally to handle IMPORT requests on this server.  The
# <template> is the file containing the template for the form that
# this operation will post to the user.  This form should contain
# @FORM-START@ and @FORM-END@ placeholders within the text.  These
# are roughly equivalent to the <form> and </form> tags within the
# template, but they are substituted with the proper URL and other
# manditory form information before the form is posted.
# ----------------------------------------------------------------------
proc filexfer::request_IMPORT {cid template cookie} {
    variable fxcookie
    variable uploads
    variable downloads
    variable downloadInfo

    # clients must come from localhost and share a secret
    variable fxcookie
    variable address

    if {![string equal $address($cid) "127.0.0.1"]} {
        puts $cid [list ERROR "unauthorized access -- must come from localhost"]
        flush $cid
        shutdown $cid
        return
    }
    if {![string equal $cookie $fxcookie]} {
        puts $cid [list ERROR "unauthorized access"]
        flush $cid
        shutdown $cid
        return
    }

    # Create a unique token for this import operation.
    # Make sure that it's not already being used.
    while {1} {
        set t [bakeCookie]
        if {![info exists downloads($t)]} {
            break
        }
    }
    set downloads($t) 1
    set uploads($t) $cid  ;# when info comes back, tell this client

    #
    # Load the text from the template file so we can make substitutions.
    #
    if {[catch {
        set fid [open $template r]
        set tinfo [read $fid]
        close $fid
    } result]} {
        puts $cid [list ERROR "error reading IMPORT template: $result"]
        flush $cid
        return
    }


    variable clients
    if {[llength $clients(order)] > 0} {
        # --------------------------------------------------------------
        # DEPRECATED -- used only for old Rappture filexfer clients
        # --------------------------------------------------------------
        set fstart "<form action=\"/upload\" enctype=\"multipart/form-data\" method=\"post\">"
    } else {
        set fstart [format "<form action=\"/filexfer/%s/upload\" enctype=\"multipart/form-data\" method=\"post\">" $fxcookie]
    }
    if {[regsub {@FORM-START@} $tinfo $fstart tinfo] != 1} {
        puts $cid [list ERROR "missing @FORM-START@ spec in IMPORT template"]
        flush $cid
        return
    }

    set fend [format "<input type=\"hidden\" name=\"client\" value=\"%s\"></form>" $t]
    if {[regsub {@FORM-END@} $tinfo $fend tinfo] != 1} {
        puts $cid [list ERROR "missing @FORM-END@ spec in IMPORT template"]
        flush $cid
        return
    }

    if {[catch {
        set fid [open $template w]
        puts -nonewline $fid $tinfo
        close $fid
    } result]} {
        puts $cid [list ERROR "error updating IMPORT template: $result"]
        flush $cid
        return
    }

    # Create a unique token for this export file.
    # Make sure that it's not already being used.
    set downloadInfo($t-file) $template
    set downloadInfo($t-time) [clock seconds]
    set downloadInfo($t-timeout) 600
    set downloadInfo($t-delete) 1
    set downloadInfo($t-format) raw
    set downloadInfo($t-dlpage) ""

    countdown cancel

    #
    # Tell the clients to download this file
    #
    if {[catch {trigger $t} result]} {
        puts $cid [list ERROR "IMPORT error: $result"]
        flush $cid
    }
}

# ----------------------------------------------------------------------
# USAGE: request_BYE <clientId> <cookie>
#
# Used internally to handle BYE requests on this server.  Clients
# send this when they're done, and then they wait for the server
# to disconnect them.  That way, they get all of the information
# coming to them queued up on various requests.
# ----------------------------------------------------------------------
proc filexfer::request_BYE {cid cookie} {
    # clients must come from localhost and share a secret
    variable fxcookie
    variable address

    if {![string equal $address($cid) "127.0.0.1"]} {
        puts $cid [list ERROR "unauthorized access -- must come from localhost"]
        flush $cid
        shutdown $cid
        return
    }
    if {![string equal $cookie $fxcookie]} {
        puts $cid [list ERROR "unauthorized access"]
        flush $cid
        shutdown $cid
        return
    }

    # okay, service the client -- shutdown as requested
    shutdown $cid
}

# ----------------------------------------------------------------------
# DEPRECATED -- used only for old Rappture filexfer clients
# ----------------------------------------------------------------------
# USAGE: request_REGISTER <clientId> <user> <address> <cookie> <protocol>
#
# Used internally to handle REGISTER requests on this server.  A client
# sends REGISTER requests when it wants to be notified of file transfer
# operations.  The <cookie> must match the one for this server, so
# we know we can trust the client.  The <protocol> tells us what version
# of filexfer client we're talking to.  If the protocol doesn't match
# the current version, we warn the user to restart his browser.
# ----------------------------------------------------------------------
proc filexfer::request_REGISTER {cid user addr clientCookie proto} {
    variable clients
    variable fxcookie
    variable protocol

    if {![string equal $fxcookie $clientCookie]} {
        response $cid header -status "401 Unauthorized"
        response $cid error -status "401 Unauthorized" -message "Credentials are not recognized."
    } else {
        # add this client to the known listeners
        set clients($cid) 0
        set protocol($cid) $proto
    }
}

# ----------------------------------------------------------------------
# DEPRECATED -- used only for old Rappture filexfer clients
# ----------------------------------------------------------------------
# USAGE: request_UNREGISTER <clientId>
#
# Used internally to handle UNREGISTER requests on this server.
# A client sends this request when it is being destroyed, to let
# the server know that it no longer needs to handle this client.
# ----------------------------------------------------------------------
proc filexfer::request_UNREGISTER {cid} {
    variable clients

    set i [lsearch -exact $cid $clients(order)]
    if {$i >= 0} {
        set clients(order) [lreplace $clients(order) $i $i]
    }
    catch {unset clients($cid)}
}

# ----------------------------------------------------------------------
# DEPRECATED -- used only for old Rappture filexfer clients
# ----------------------------------------------------------------------
# USAGE: request_ACTIVATE <clientId>
#
# Used internally to handle ACTIVATE requests on this server.  A client
# must first REGISTER with its cookie for authorization.  Then, as
# its thread starts, it sends an ACTIVATE request, letting us know
# that the client is ready to receive notifications.
# ----------------------------------------------------------------------
proc filexfer::request_ACTIVATE {cid} {
    variable clients

    #
    # Activate only if the client has already registered
    # properly and is on our known list.
    #
    if {[info exists clients($cid)]} {
        set clients($cid) 1

        # move the most recently activated connection to the front
        set i [lsearch -exact $cid $clients(order)]
        if {$i >= 0} {
            set clients(order) [lreplace $clients(order) $i $i]
        }
        set clients(order) [linsert $clients(order) 0 $cid]
    }
}

# ----------------------------------------------------------------------
# DEPRECATED -- used only for old Rappture filexfer clients
# ----------------------------------------------------------------------
# USAGE: request_DEACTIVATE <clientId>
#
# Used internally to handle DEACTIVATE requests on this server.  A client
# must first REGISTER with its cookie for authorization.  Then, as
# its thread starts, it sends an ACTIVATE request.  When its thread
# stops (because the applet is swapped out of the web page), the
# client sends a DEACTIVATE request, and we stop sending messages to
# that client.
# ----------------------------------------------------------------------
proc filexfer::request_DEACTIVATE {cid} {
    variable clients

    #
    # Deactivate only if the client has already registered
    # properly and is on our known list.
    #
    if {[info exists clients($cid)]} {
        set clients($cid) 0

        # remove this from the list of activated connections
        set i [lsearch -exact $cid $clients(order)]
        if {$i >= 0} {
            set clients(order) [lreplace $clients(order) $i $i]
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::shutdown <clientId>
#
# Used internally to close and clean up a client connection.
# Clears any data associated with the client.
# ----------------------------------------------------------------------
proc filexfer::shutdown {cid} {
    variable clients
    variable buffer
    variable address
    variable uploads

    catch {close $cid}

    # did client give up on upload?  then forget about it
    foreach {key val} [array get uploads] {
        if {[string equal $cid $val]} {
            unset uploads($key)
        }
    }

    if {[info exists clients($cid)]} {
        unset clients($cid)
    }
    set i [lsearch -exact $clients(order) $cid]
    if {$i >= 0} {
        set clients(order) [lreplace $clients(order) $i $i]
    }

    if {[info exists buffer($cid)] && "" != $buffer($cid)} {
        unset buffer($cid)
    }
    unset address($cid)

    filexfer::log "disconnected client $cid"
}

# ----------------------------------------------------------------------
# USAGE: filexfer::housekeeping
#
# This gets set up to get invoked at regular intervals for housekeeping
# tasks.  During each invocation, this procedure checks for items that
# have timed out and removes them from the active list.  Items that
# are marked for deletion are cleaned up at that time.  If it's been
# a long time since any client has connected and there are no more
# pending requests, then the server shuts itself down.
# ----------------------------------------------------------------------
proc filexfer::housekeeping {} {
    variable downloads
    variable downloadInfo

    set status [catch {
        set now [clock seconds]
        foreach t [array names downloads] {
            if {$now >= $downloadInfo($t-time) + $downloadInfo($t-timeout)} {
                filexfer::cleanup $t
            }
        }

        if {[array size downloads] == 0} {
            countdown continue
        }
    } result]
    if {$status != 0} {
        filexfer::log "error in housekeeping: $result"
    }

    # invoke this at regular intervals according to the delay below
    after 5000 filexfer::housekeeping
}

# ----------------------------------------------------------------------
# USAGE: filexfer::countdown continue|cancel
#
# Usually invoked during the housekeeping step when there are no
# more files being tracked.  When the countdown reaches 0, the
# server exits.  Any new export or download operation cancels
# the countdown.
# ----------------------------------------------------------------------
proc filexfer::countdown {option} {
    variable countdown
    switch -- $option {
        continue {
            if {"" == $countdown} {
                set countdown 60
            } elseif {[incr countdown -1] <= 0} {
                filexfer::cleanup
                filexfer::log "inactive for a while -- shutting down..."
                exit
            }
        }
        cancel {
            set countdown ""
        }
        default {
            error "bad option \"$option\": should be continue or cancel"
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::loadHTML <token> <directory> <file>
#
# Used in the GET operation to load a file in format "html".  Looks
# for the <file> within the <directory> context.  Loads the file and
# rewrites all embedded file references (for images and links) so that
# they can be served up properly from this server.  This allows the
# user to view not only an HTML page, but all of the resources embedded
# and linked by the page as well.
# ----------------------------------------------------------------------
proc filexfer::loadHTML {token dir file} {
    package require tdom
    variable fxcookie

filexfer::log "loading file [file join $dir $file]..."
    set fid [open [file join $dir $file] r]
    set info [read $fid]
    close $fid

filexfer::log "parsing..."
    set doc [dom parse -html $info]
    set queue [$doc documentElement]

    while {[llength $queue] > 0} {
        set node [lindex $queue 0]
        set queue [lrange $queue 1 end]

        switch -- [string tolower [$node nodeName]] {
            a {
                if {[catch {$node getAttribute href} val] == 0 && "" != $val} {
                    if {![regexp -nocase {^https?://} $val]} {
                        set val [string trimleft $val /]
                        set newurl [format "/filexfer/%s/dlfile/%s?token=%s" \
                            $fxcookie $val $token]
                        $node setAttribute href $newurl
                    }
                }
            }
            img {
                if {[catch {$node getAttribute src} val] == 0 && "" != $val} {
                    if {![regexp -nocase {^https?://} $val]} {
                        set val [string trimleft $val /]
                        set newurl [format "/filexfer/%s/dlfile/%s?token=%s" \
                            $fxcookie $val $token]
                        $node setAttribute src $newurl
                    }
                }
            }
        }

        eval lappend queue [$node childNodes]
    }

    set html [$doc asHTML]
    $doc delete

    return $html
}

# ----------------------------------------------------------------------
# USAGE: response <channel> header -status <s> -connection <c>
# USAGE: response <channel> body -string <s> -type <t>
# USAGE: response <channel> error -message <m>
# USAGE: response <channel> file -path <f>
#
# Used internally to generate responses to the client.  Returns a
# string representing the requested response.
# ----------------------------------------------------------------------
proc filexfer::response {cid what args} {
    switch -- $what {
        header {
            Rappture::getopts args params {
                value -status ""
                value -connection close
            }
            set s [clock seconds]
            set date [clock format $s -format {%a, %d %b %Y %H:%M:%S %Z}]
            catch {
                puts $cid [format "HTTP/1.1 %s
Date: %s
Server: Rappture
Connection: %s" $params(-status) $date $params(-connection)]
            }
        }

        body {
            Rappture::getopts args params {
                value -string ""
                value -type "auto"
            }
            if {$params(-type) == "auto"} {
                if {[isbinary $params(-string)]} {
                    set params(-type) "application/octet-stream"
                } else {
                    set params(-type) "text/plain"
                }
            }
            catch {
                puts $cid [format "Content-type: %s\nContent-length: %d\n" \
                    $params(-type) [string length $params(-string)]]
            }

            variable mime2type
            if {$mime2type($params(-type)) == "binary"} {
                # binary data -- send data as raw bytes
                set olde [fconfigure $cid -encoding]
                fconfigure $cid -buffering none -encoding binary
                catch {
                    puts -nonewline $cid $params(-string)
                    flush $cid
                }
                fconfigure $cid -buffering line -encoding $olde
            } else {
                # ascii data -- send normally
                catch {
                    puts $cid $params(-string)
                }
            }
        }

        error {
            Rappture::getopts args params {
                value -status "400 Bad Request"
                value -message ""
            }
            set heading [lrange $params(-status) 1 end]
            set html [format "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">
<HTML><HEAD>
<TITLE>%s</TITLE>
</HEAD><BODY>
<H1>%s</H1>
%s
</BODY></HTML>" $params(-status) $heading $params(-message)]
            response $cid body -type text/html -string $html
        }

        file {
            Rappture::getopts args params {
                value -path ""
                value -connection close
                value -saveas 0
            }
            if {![file exists $params(-path)]} {
                #
                # FILE NOT FOUND:
                #   The user is requesting some file that is not part of
                #   the standard filexfer installation.
                #
                response $cid header \
                    -status "404 Not Found" \
                    -connection $params(-connection)

                response $cid error -status "404 Not Found" -message "The requested file $params(-path) was not found on this server."
            } elseif {[catch {
                    set fid [open $params(-path) r]
                    set data [read $fid]
                    close $fid
                } result]} {

                response $cid header \
                    -status "500 Internal Server Error" \
                    -connection $params(-connection)
                response $cid error -status "500 Internal Server Error" -message "The requested file $params(-path) is not installed properly on this server."
            } else {
                #
                # READ AND RETURN THE FILE
                #
                foreach {mtype ftype} [filexfer::mimetype $params(-path)] break
                if {$ftype == "binary"} {
                    # if this is binary data, read it again and get pure bytes
                    catch {
                        set fid [open $params(-path) r]
                        fconfigure $fid -translation binary -encoding binary
                        set data [read $fid]
                        close $fid
                    } result
                }
                response $cid header \
                    -status "200 OK" \
                    -connection $params(-connection)

                if {$params(-saveas)} {
                    set ftail [file tail $params(-path)]
                    catch { puts $cid "Content-Disposition: attachment; filename=\"$ftail\"" }
                }

                set s [file mtime $params(-path)]
                set date [clock format $s -format {%a, %d %b %Y %H:%M:%S %Z}]
                catch { puts $cid "Last-Modified: $date" }

                response $cid body -type $mtype -string $data
            }
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::urlDecode <string>
#
# Used internally to decode a string in URL-encoded form back to
# its normal ASCII equivalent.  Returns the input string, but with
# any %XX characters translated back to their ASCII equivalents.
# ----------------------------------------------------------------------
proc filexfer::urlDecode {string} {
    while {[regexp -indices {%[0-9A-Fa-f][0-9A-Fa-f]} $string match]} {
        foreach {p0 p1} $match break
        set hex [string range $string [expr {$p0+1}] $p1]
        set char [binary format c [scan $hex "%x"]]
        set string [string replace $string $p0 $p1 $char]
    }
    return $string
}

# ----------------------------------------------------------------------
# USAGE: isbinary <string>
#
# Used internally to see if the given <string> has binary data.
# If so, then it must be treated differently.  Normal translation
# of carriage returns and line feeds must be suppressed.
# ----------------------------------------------------------------------
proc filexfer::isbinary {string} {
    # look for binary characters, but avoid things like \t \n etc.
    return [regexp {[\000-\006\016-\037\177-\400]} $string]
}

# ----------------------------------------------------------------------
# USAGE: mimetype <file> <data>
#
# Used internally to determine the MIME type and ascii/binary
# nature of the data in <file>.
# ----------------------------------------------------------------------
proc filexfer::mimetype {file} {
    variable ext2mime
    variable mime2type

    set ext [file extension $file]
    if {[info exists ext2mime($ext)]} {
        set mtype $ext2mime($ext)
    } else {
        set data ""
        if {[catch {
            set fid [open $file r]
            fconfigure $fid -translation binary -encoding binary
            set data [read $fid 8096]  ;# read the first 8k and judge that
            close $fid
        } result]} {
            filexfer::log "error checking file type for \"$file\": $result"
        }
        
        if {[isbinary $data]} {
            set mtype application/octet-stream
        } else {
            set mtype text/plain
        }
    }
    return [list $mtype $mime2type($mtype)]
}

# ----------------------------------------------------------------------
# USAGE: bakeCookie
#
# Used internally to create a one-time use cookie, passed to clients
# to secure file transfer.  Only clients should know the cookie, so
# only clients will have access to files.
# ----------------------------------------------------------------------
proc filexfer::bakeCookie {} {
    variable cookieChars

    set cmax [expr {[llength $cookieChars]-1}]
    set cookie ""
    while {[string length $cookie] < 20} {
        set rindex [expr {round(rand()*$cmax)}]
        append cookie [lindex $cookieChars $rindex]
    }
    return $cookie
}

# ----------------------------------------------------------------------
# USAGE: filexfer::log -init
# USAGE: filexfer::log <message>
#
# Used to send log messages to some file for debugging.
# ----------------------------------------------------------------------
proc filexfer::log {mesg} {
    global env
    variable log
    if {"-init" == $mesg} {
        if {"" != $log} {
            catch {close $log}
        }
        set log ""

        if {[info exists env(SESSIONDIR)]} {
            set logfile [file join $env(SESSIONDIR) filexfer[pid].log]
        } elseif {[info exists env(HOME)] && [info exists env(SESSION)]} {
            set logfile [file join $env(HOME) data sessions $env(SESSION) filexfer[pid].log]
        } elseif {[file writable [pwd]]} {
            set logfile [file join [pwd] filexfer[pid].log]
        }
        if {[catch {open $logfile w} result] == 0} {
            set log $result
        }
    } elseif {"" != $log} {
        catch {
            puts $log $mesg
            flush $log
        }
    }
}

# ----------------------------------------------------------------------
#  MAIN SCRIPT
# ----------------------------------------------------------------------
# Load the settings from the user's resources file.
# In particular, figure out filexfer_port so we know how to talk
# to the filexfer server.
#
filexfer::log -init
filexfer::log "started on [clock format [clock seconds]]"

if {[catch {filexfer::resources} result]} {
    filexfer::log "can't load resource configuration: $result"
    filexfer::log "exiting..."
    exit 1
}
array set settings $result
if {[catch {filexfer::init $settings(port) $settings(cookie)} result]} {
    filexfer::log "can't start server: $result"
    filexfer::log "exiting..."
    exit 1
}

filexfer::log "listening on port $settings(port)"
vwait forever
