# ----------------------------------------------------------------------
#  USAGE: importfile ?options? ?--label text? file file ...
#
#    options:
#      -h or --help
#        Prints a help message.
#
#      -f or --for <text>
#        Short explanation of what the data will be used for; for
#        example, "for CNTBands 2.0".  If given, this text is inserted
#        into the upload form to help explain what it will be used for.
#
#      -l or --label <text>
#        Prompt for subsequent file arguments using this label string.
#        The default label just uses the file name.
#
#      --
#        Remaining arguments are treated as file names, even if
#        they start with a -.
#
#  This is the client that users invoke to transfer files from their
#  desktop into their session.  This works a lot like a download
#  operation (see exportfile command) but the file downloaded is
#  merely a form that the user can use to upload information.  This
#  client tries to connect to a server to handle the transfer.  If
#  necessary, the client tries to spawn the server and then connect
#  to it.  The server uses the "clientaction" program to open the
#  upload form on the user's desktop.  The user then chooses one
#  or more files and posts the results back to the server, which
#  stores the files in the specified names and then notifies this
#  client that they are ready.
#
# ======================================================================
#  AUTHOR:  Michael McLennan, Purdue University
#  Copyright (c) 2004-2007  Purdue Research Foundation
#
#  See the file "license.terms" for information on usage and
#  redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# ======================================================================

# load util procedures from this path
lappend auto_path [file dirname [info script]]

array set options {
  --for ""
  --label @DEFAULT
}
set usage "$argv0 \[-f|--for text\] \[-l|--label text\] file file ..."
set manpage "
USAGE: $usage

  options:
    -h or --help
      Prints this help message.

    -f or --for <text>
      Short explanation of what the data will be used for; for
      example, \"for CNTBands 2.0\".  If given, this text is inserted
      into the upload form to help explain what it will be used for.

    -l or --label <text>
      Prompt for subsequent file arguments using this label string.
      The default label just uses the file name.

    --
      Remaining arguments are treated as file names, even if they
      start with a -.

    file
      Uploaded file will be saved in this file name within your
      tool session. If file = @USE_REMOTE@ the remote filename
      will be retained locally.

You can use this command to transfer one or more files from your
desktop to your tool session via a web browser.  This command causes
a web page to pop up prompting you for various files on your desktop.
Choose one or more files and submit the form.  The files will be
uploaded to your tool session and saved in the file names specified
on the command line.

This command returns a list of names for files actually uploaded."

#
# Parse all command-line arguments and build up a list of files:
#   label1 file1  label2 file2 ...
#
set uploadlist ""

while {[llength $argv] > 0} {
    set opt [lindex $argv 0]
    set argv [lrange $argv 1 end]

    if {"-" != [string index $opt 0]} {
        lappend uploadlist $options(--label) $opt
        continue
    }

    switch -- $opt {
        -f - --for {
            if {[llength $argv] == 0} {
                puts stderr "missing text for $opt"
                exit 1
            }
            set options(--for) " [lindex $argv 0]"
            set argv [lrange $argv 1 end]
        }
        -l - --label {
            if {[llength $argv] == 0} {
                puts stderr "missing text for $opt"
                exit 1
            }
            set options(--label) [lindex $argv 0]
            set argv [lrange $argv 1 end]
        }
        -h - --help {
            puts $manpage
            exit
        }
        -- {
            foreach file $argv {
                lappend uploadlist $options(--label) $file
            }
            set argv ""
        }
        default {
            puts stderr "bad option \"$opt\""
            puts stderr $usage
            exit 1
        }
    }
}

#
# Load the template for the upload form.
#
set installdir [file dirname [info script]]
set fid [open [file join $installdir upload.html] r]
set html [read $fid]
close $fid

#
# Load the settings from the user's resources file.
# In particular, figure out filexfer_port so we know how to talk
# to the filexfer server.
#
if {[catch {filexfer::resources} result]} {
    puts stderr "can't load resource configuration:"
    puts stderr $result
    exit 1
}
array set settings $result

if {![info exists settings(port)]} {
    puts stderr "missing filexfer_port in resource settings"
    exit 1
}
if {![info exists settings(cookie)]} {
    puts stderr "missing filexfer_cookie in resource settings"
    exit 1
}

#
# Make a connection to the filexfer server.  Spawn the server, if
# necessary.
#
set buffer ""
proc server_mesg {sid} {
    global finished buffer

    if {[gets $sid line] < 0} {
        set finished 1
    } else {
        append buffer $line "\n"
        if {[info complete $buffer]} {
            set cmd $buffer
            set buffer ""
            set word [lindex $cmd 0]
            switch -- $word {
              ERROR {
                set mesg [lindex $cmd 1]
                puts stderr $mesg
                exit 1
              }
              IMPORTED {
                # print list of files created:
                puts stdout [lrange $cmd 1 end]
              }
              default {
                puts "HUH? $cmd"
              }
            }
        }
    }
}

if {[llength $uploadlist] > 0} {
    if {[catch {filexfer::connect $settings(port) server_mesg} sid]} {
        puts stderr "can't connect to filexfer server: $sid"
        exit 1
    }

    #
    # Substitute the upload list into the @REPEAT(...)@ area of
    # the HTML text.
    #
    while {[regexp -indices {@REPEAT\((.+?)\)@} $html match inner]} {
        foreach {s0 s1} $match break
        foreach {p0 p1} $inner break
        set template [string range $html $p0 $p1]

        set expanded ""
        set n 1
        foreach {label file} $uploadlist {
            if {"@DEFAULT" == $label} {
                set label "File [file tail $file]"
            }
            if {[file pathtype $file] == "relative"} {
                set file [file join [pwd] $file]
            }
            if {[string match "*@USE_REMOTE@*" "$file"]} {
                set disabled "disabled"
            } else {
                set disabled ""
            }

            append expanded [string map [list \
                @INDEX@ $n \
                @LABEL@ $label \
                @DEST@ $file \
                @DISABLED@ $disabled \
            ] $template]
            incr n
        }
        set html [string replace $html $s0 $s1 $expanded]
    }

    #
    # Substitute the rest of the @NAME@ fields.
    #
    set hubname "???"
    if {[info exists settings(hubname)]} {
        set hubname $settings(hubname)
    }

    set huburl ""
    if {[info exists settings(huburl)]} {
        set huburl $settings(huburl)
    }

    set html [string map [list \
        @FOR@ $options(--for) \
        @HUBNAME@ $hubname \
        @HUBURL@ $huburl \
    ] $html]

    set file "/tmp/upload[pid].html"
    set fid [open $file w]
    puts -nonewline $fid $html
    close $fid

    puts $sid [list IMPORT $file $settings(cookie) FILEXFER/1.0]

    vwait finished  ;# wait for server to close
}
