# ----------------------------------------------------------------------
#  FILEXFER UTILITIES
#
#  This file contains utility procedures that are used across the
#  server.tcl and its clients.
# ======================================================================
#  AUTHOR:  Michael McLennan, Purdue University
#  Copyright (c) 2004-2007  Purdue Research Foundation
#
#  See the file "license.terms" for information on usage and
#  redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# ======================================================================
namespace eval filexfer { # forward declaration }

# ----------------------------------------------------------------------
#  USAGE: filexfer::connect <port> <handler>
#
#  Used internally to create a connection to the filexfer server.
#  Returns the file descriptor for the socket connection.  Sets up
#  the connection so that if a message comes back from the server,
#  it is handled by the <handler> proc.
# ----------------------------------------------------------------------
proc filexfer::connect {port callback} {
    if {[catch {socket localhost $port} sid] == 0} {
        fconfigure $sid -buffering line
        fileevent $sid readable [list server_mesg $sid]
        return $sid
    }

    package require BLT
    set installdir [file dirname [info script]]
    exec tclsh [file join $installdir server.tcl]

    for {set tries 10} {$tries > 0} {incr tries -1} {
        after 1000  ;# wait a minute for new server to start

        if {[catch {socket localhost $port} sid] == 0} {
            fconfigure $sid -buffering line
            fileevent $sid readable [list server_mesg $sid]
            return $sid
        }
    }
    error "tried to spawn server, but it doesn't respond"
}

# ----------------------------------------------------------------------
# RESOURCE LOADING
# ----------------------------------------------------------------------
namespace eval filexfer {
    #
    # Set up a safe interpreter for loading filexfer options...
    #
    variable optionParser [interp create -safe]
    foreach cmd [$optionParser eval {info commands}] {
        $optionParser hide $cmd
    }
    # this lets us ignore unrecognized commands in the file:
    $optionParser invokehidden proc unknown {args} {}

    foreach {name proc} {
        filexfer_port filexfer::set_filexfer_port
        filexfer_cookie filexfer::set_filexfer_cookie
        hub_name filexfer::set_hub_name
        hub_url filexfer::set_hub_url
        session_token filexfer::set_session_token
    } {
        $optionParser alias $name $proc
    }
}

# ----------------------------------------------------------------------
# USAGE: filexfer::resources
#
# Loads a list of resources from the $SESSIONDIR/resources file
# and returns a list of the form {key1 value1 key2 value2 ...}
# for all resources related to filexfer.
# ----------------------------------------------------------------------
proc filexfer::resources {} {
    global env
    variable resources
    variable optionParser

    if {![info exists env(SESSIONDIR)]} {
        error "\$SESSIONDIR undefined"
    }
    set rfile [file join $env(SESSIONDIR) resources]
    if {![file exists $rfile]} {
        error "file $rfile not found"
    }

    set fid [open $rfile r]
    set rinfo [read $fid]
    close $fid

    catch {unset resources}
    $optionParser eval $rinfo

    return [array get resources]
}

# ----------------------------------------------------------------------
# RESOURCE: filexfer_port
# ----------------------------------------------------------------------
proc filexfer::set_filexfer_port {port} {
    variable resources

    if {![string is integer $port]} {
        error "bad value \"$port\" for filexfer_port: should be integer"
    }
    set resources(port) $port
}

# ----------------------------------------------------------------------
# RESOURCE: filexfer_cookie
# ----------------------------------------------------------------------
proc filexfer::set_filexfer_cookie {cookie} {
    variable resources
    set resources(cookie) $cookie
}

# ----------------------------------------------------------------------
# RESOURCE: hub_name
# ----------------------------------------------------------------------
proc filexfer::set_hub_name {text} {
    variable resources
    set resources(hubname) $text
}

# ----------------------------------------------------------------------
# RESOURCE: hub_url
# ----------------------------------------------------------------------
proc filexfer::set_hub_url {text} {
    variable resources
    #
    # BE CAREFUL:  Make sure that we have https:// at the
    #   start of the hub url, or else we'll get warnings
    #   about loading secure/non-secure items on IE.
    #
    regsub -nocase {^http://} $text https:// text
    if {![regexp {^https://} $text]} {
        set text "https://$text"
    }
    set resources(huburl) $text
}

# ----------------------------------------------------------------------
# RESOURCE: session_token
# ----------------------------------------------------------------------
proc filexfer::set_session_token {text} {
    variable resources
    set resources(session) $text
}
