# @package      hubzero-chuse
# @file         env_files.tcl
# @author       Derrick Kearney <dsk@purdue.edu>
# @copyright    Copyright (c) 2005-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2005-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

proc parse_env_dir {args} {
    global conflictDB

    set flist ""
    set envdata ""

    # grab all file names first
    foreach dirname $args {
        if {![file exists $dirname] || ![file isdirectory $dirname]} {
            continue
        }

        set flist [concat $flist [glob -nocomplain $dirname/*]]
    }

    # sort based on tail name
    # process files
    foreach f [lsort -command parse_compare $flist] {
        set environinfo [parse_env $f]
        if {[llength $environinfo] == 0} {
            continue
        }
        foreach {fname conflict desc help ver tags} $environinfo break
        lappend envdata $fname $conflict $desc $help $ver $tags
        lappend conflictDB($conflict) $fname
        lappend conflictDB($fname) $conflict
    }

    return $envdata
}

proc parse_compare {a b} {
    set a0 [file tail $a]
    set b0 [file tail $b]
    return [string compare $a0 $b0]
}

proc parse_active_envs {fname} {
    if {![file exists $fname] || ![file isfile $fname] || ![file readable $fname]} {
        return ""
    }

    if {[catch {open $fname r} fid]} {
        puts stderr "Warning: $fid"
        return ""
    }
    set envtext [split [read $fid] "\n"]
    close $fid
    return $envtext
}

# FIXME: use a tcl safe interp to parse the env files instead of regexp's
proc parse_env {fname} {
    if {![file exists $fname] || ![file isfile $fname] || ![file readable $fname]} {
        return ""
    }

    if {[catch {open $fname r} fid]} {
        puts stderr "Warning: $fid"
        return ""
    }
    set envtext [read $fid]
    close $fid

    set conflict ""
    set desc ""
    set help ""
    set version ""
    set tags ""

    regexp -nocase {\nversion=([\w[:punct:]]+)} $envtext match version
    regsub -all {\$\{version\}} $envtext $version envtext
    regexp -nocase {conflict\s(\w+)} $envtext match conflict
    regexp -nocase {desc\s\"([^\"]+)\"} $envtext match desc
    regexp -nocase {help\s\"([^\"]+)\"} $envtext match help
    regexp -nocase {tags\s([\s\w]+)\n} $envtext match tags

    return [list [file tail $fname] $conflict $desc $help $version $tags]
}

proc env_parse_test {} {

    foreach f [glob -nocomplain environ/*] {
        lappend envdata [parse_env $f]
    }

    puts $envdata
}
